/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Tristan Mehamli (Sep 2011)
** Modified by : Jean Paul Digeon (Sep 2011)
**=======================================================================*/
#ifndef IVT_PLUGINS_MANAGER_H
#define IVT_PLUGINS_MANAGER_H

//////////////////////////////////////////////////////////////////////////////
//
//  Class: IvtPluginsManager
//
//////////////////////////////////////////////////////////////////////////////

#include <IvTune/IvTuneExtender/IvtLibName.h>

#include <Inventor/SbString.h>

#include <Inventor/STL/list>
#include <Inventor/STL/map>
#include <Inventor/STL/pair>
#include <Inventor/STL/string>

#include <QObject>
#include <QtXml/QDomElement> 

#ifdef _WIN32
# if !defined(IvTuneExtenderAPI_EXPORTS)
#  ifndef OIV_DISABLE_AUTOLINK
#   pragma comment(lib,__IVTUNEEXTENDERAPILIB)
#  endif
# endif
#endif //win

class IvtPlugin;
class IvtProfile;
class IvtProfileSetting;

/** 
 * @VSGEXT Class that manages the loading of IvTune plug-ins.
 *
 * @ingroup IvTuneExtender
 *
 * @DESCRIPTION
 *
 * This class automatically loads the IvTune plug-ins and their extensions 
 * according to the profile configuration.
 *
 * @M_SINCE 9.0
 *
 * @SEE_ALSO
 *    IvtProfile
 */
class IVTEXTENDER_API IvtPluginsManager : public QObject
{
  Q_OBJECT
public:

  /**
   * Add the specified profile to the list of profiles managed by this 
   * plug-in manager.
   *
   * @param profile the profile to add.
   */
  void addProfile( IvtProfile* profile );

  /**
   * Returns the active profile.
   */
  IvtProfile* getActiveProfile() const;

  /**
   * Returns a plug-in given its name.
   *
   * @param name the name of the plug-in to get.
   */
  IvtPlugin* getPluginByName( const SbString& name );

  /**
   * Returns the loaded plug-ins list.
   */
   const std::list< IvtPlugin* >& getPlugins() const;
   
  /**
   * Returns an instance of this plug-ins manager.
   */
  static IvtPluginsManager* getInstance();

  /**
   * Returns a profile given its name.
   *
   * @param name the name of the profile to get.
   */
  IvtProfile* getProfileByName( const SbString& name );

  /**
  * Returns the profiles list.
  */
  const std::list<IvtProfile*>& getProfilesList() const;

  /**
   * Load the specified plug-in and return a pointer to it.
   *
   * @param path the path to the plug-in to load.
   * @param fileName the name of the plug-in to load.
   */
  IvtPlugin* loadPlugin( const SbString& path, const SbString& fileName );

  /**
   * Read the configuration file, load plug-ins and extensions, and set up the profiles.
   *
   * @param configFilePath the path to the configuration file to load.
   * @param isPath indicates whether it is a path to file or an in-memory file.
   */
  void read( const SbString& configFilePath, bool isPath = true );

  /**
   * Release the current instance of this plug-ins manager.
   * All Plug-ins and extensions are unloaded before.
   */
  static void releaseInstance();

  /**
   * Remove the given profile from the list of managed profiles.
   *
   * @param profile the profile to remove.
   */
  void removeProfile( IvtProfile* profile );

  /**
   * Set the given profile to be the active profile.
   * Extensions in the previous profile are deactivated and extensions in the given profile are activated.
   *
   * @param profile the new active profile.
   */
  void setActiveProfile( IvtProfile* profile );

  /**
   * Unload the specified plug-in.
   *
   * @param plugin the plugin to unload.
   */
  void unloadPlugin( IvtPlugin* plugin );

  /**
   * Unload all the loaded plug-ins.
   */
  void unloadAllPlugins();

  /**
   * Write the current configuration to the specified file.
   *
   * @param configFilePath the path to the configuration file.
   */
  void write( const SbString& configFilePath );

  /**
   * Destructor.
   */
  ~IvtPluginsManager();

private:
  /** Constructor. */
  IvtPluginsManager();

  /** Gets the controlled extensions from a profile setting. */
  void readControlledExtension ( QDomElement&  controlledExtension, IvtProfileSetting* s);

  /** Writes the controlled extensions of a profile setting to the configuration file. */
  void writeControlledExtension( QDomDocument& doc, QDomElement& extension, IvtProfileSetting* setting );

  /** Vectors of plug-ins instances. */
  std::list< IvtPlugin*> m_pluginsInstances;

  /** Pointer to the default profile. */
  IvtProfile*  m_defaultProfile;

  /** Pointer the active profile. */
  IvtProfile*  m_activeProfile;

  /** Vectors of profile. */
  std::list<IvtProfile*> m_profiles;

  /** Singleton */
  static IvtPluginsManager* m_instance;
};

#endif // IVT_PLUGINS_MANAGER_H
