/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Tristan Mehamli (Sep 2011)
**=======================================================================*/
#ifndef IVT_SHELL_H
#define IVT_SHELL_H

//////////////////////////////////////////////////////////////////////////////
//
//  Class: IvtShell
//
//////////////////////////////////////////////////////////////////////////////

#include <IvTune/IvTuneExtender/IvtLibName.h>

#include <Inventor/nodes/SoNode.h>

#include <QObject>

#ifdef _WIN32
# if !defined(IvTuneExtenderAPI_EXPORTS)
#  ifndef OIV_DISABLE_AUTOLINK
#   pragma comment(lib,__IVTUNEEXTENDERAPILIB)
#  endif
# endif
#endif //win

class QDockWidget;
class QMenu;
class QString;
class QToolBar;
class QTreeWidget;
class QWidget;

class IvtExtension;
class IvtEditor;

class SoField;
class SoType;

/** 
 * @VSGEXT Class to register Extension GUI in IvTune
 *
 * @ingroup IvTuneExtender
 *
 * @DESCRIPTION
 *
 * This class provides methods to access the different GUI 
 * elements of IvTune. Thus it is possible to modify or extend them.
 *
 * @M_SINCE 9.0
 *
 * @SEE_ALSO
 *    IvtServices
 *
 */
class IVTEXTENDER_API IvtShell: public QObject
{
  Q_OBJECT

public:
  
  // MainWindow
  /**
   * Register the provided dock widget and sets it to the designated area.
   *
   * @param dock the dock widget to register.
   * @param area the area where the dock widget should be placed.
   */
  virtual void registerDock( QDockWidget* dock, Qt::DockWidgetArea area ) = 0;

  /**
   * Unregister the provided dock widget.
   *
   * @param dock the dock widget to unregister.
   */
  virtual void unregisterDock( QDockWidget*  dock ) = 0;

  // Menus
  /**
   * Returns the list of menus that are registered with the provided type 
   * and its derived types.
   *
   * @param type the type of node/field associated with the menus.
   */
   virtual std::vector<QMenu*> getContextualMenus( const SoType& type ) = 0;

  /**
   * Register the menu and add it before the help menu of IvTune.
   *
   * @param newMenu the contextual menu to register.
   * @param type the type of node/field associated with the menus.
   * @param strict indicates if it is for the exact type or any derived type.
   */
  virtual void registerContextualMenu( const SoType& type, bool strict, QMenu* newMenu ) = 0;

  /**
   * Register the menu and add it before the help menu of IvTune.
   *
   * @param newMenu the menu to register.
   */
  virtual void registerMenu( QMenu* newMenu ) = 0;

  /**
   * Register an extension in the view menu of IvTune.
   *
   * @param extension the extension to register.
   * @param visibilityOnActivate indicates whether the extension is visible when 
   *        activated or not.
   */
  virtual void registerViewMenu( IvtExtension* extension, SbBool visibilityOnActivate = true ) = 0;

  /**
   * Unregister the context menu given as parameter.
   * It will be unregistered for strict and derived types.
   *
   * @param menu the contextual menu to unregister.
   */
   virtual void unregisterContextualMenu( QMenu* menu ) = 0;

  /**
   * Unregister the specified menu.
   *
   * @param menu the menu to unregister.
   */
   virtual void unregisterMenu( QMenu* menu ) = 0;

   /**
   * Unregister the extension from the view menu.
   *
   * @param extension the extension to unregister.
   */
   virtual void unregisterViewMenu( IvtExtension* extension ) = 0;

   // Preferences
   /**
   * Register the extension with the specified panel and add an entry to 
   * the preferences menu.
   *
   * @param extension the extension to register.
   * @param panel the panel where the extension will be registered.
   * @param slotInitialized a qt slot called when the preference panel is initialized to be showed to the user.
   * @param slotAccept a qt slot called when  new preferences are accepted by the user.
   */
   virtual void registerPreferencePanel( IvtExtension* extension, QWidget* panel, const char* slotInitialized, const char* slotAccept ) = 0;

   /**
   * Unregister the extension from the the panel and remove the associated 
   * entry in the preferences menu.
   *
   * @param extension the extension to unregister.
   */
   virtual void unregisterPreferencePanel( IvtExtension* extension ) = 0;

  // Status bar
  /**
   * Clear the message in the status bar.
   */
  virtual void clearStatusBarMessage() = 0;

  /**
   * Register a widget in the status bar of IvTune.
   *
   * @param widget the widget to register.
   * @param isPermanent indicates whether the widget may be obscured by messages.
   *        A permanent widget is located at the far right of status bar.
   *        Otherwise the widget is located at the far left of the first permanent 
   *        widget.
   * @param stretch is the factor used to compute a suitable size for the given 
   *        widget. By default, this factor is set to 0 which indicates giving the 
   *        widget a minimum of space.
   */
  virtual void regiterStatusBarWidget( QWidget* widget, SbBool isPermanent = false, int stretch = 0 ) = 0;

  /**
   * Show a messsage in the Status bar.
   *
   * @param newMessage the message to display in the status bar.
   * @param timeout the specified number of milliseconds to display the message in the status bar.
   *
   * Note: A timeout of 0 means the message remains displayed until clearStatusBarMessage() is called.
   */
  virtual void showStatusBarMessage( const SbString& newMessage, int timeout = 2000 ) = 0;

  /**
   * Unregister a widget from the status bar.
   *
   * @param widget the widget to unregister.
   */
  virtual void unregiterStatusBarWidget( QWidget* widget ) = 0;

  // Tool bar
  /**
   * Register the provided dock widget and set it in the designated area.
   *
   * @param tb the tool bar to register.
   * @param area the area where the dock widget should be placed.
   */
   virtual void registerToolBar( QToolBar* tb, Qt::ToolBarArea area = Qt::TopToolBarArea ) = 0;

   /**
   * Unregister a tool bar from the main window.
   *
   * @param tb the tool bar to unregister.
   */
   virtual void unregisterToolBar( QToolBar* tb ) = 0;

  /**
   * Returns an instance of this class.
   */
  static IvtShell* getInstance();

protected:
  
  /** Constructor */
  IvtShell();

  /**
   * Destructor
   */
  virtual ~IvtShell();

  /** Singleton */
  static IvtShell* m_instance;
};

#endif // IVT_SHELL_H


