/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Colas Fievet (MMM YYYY)
** Author      : Tristan Mehamli (MMM YYYY)
**=======================================================================*/

#include <Inventor/sys/port.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/SbVec.h>

#ifndef SoIvTune_included
#define SoIvTune_included

#if defined(_WIN32)
#  pragma warning(disable:4251)
#endif

class SoNode;
class SbViewportRegion;
class SoGuiRenderArea;
class SoSceneManager;
class SoRenderAreaCore;

/**
 * @VSGEXT Opens an IvTune window
 *
 * @ingroup IvTune
 *
 * @DESCRIPTION
 *
 * SoIvTune is a class to start and stop IvTune.
 * 
 * If the hot key (Shift-F12 by default) is enabled, the user can start
 * IvTune by pressing that key combination. Using the #start()
 * method, the application can programmatically start and stop IvTune.
 * This is useful if all key combinations are already in use by the application or just
 * to provide an IvTune item in a menu.
 * The IvTune hot key can be redefined
 * using the environment variable OIV_IVTUNE_SHORTCUT (see SoPreferences)
 * or using the #setShortcutValue() method.
 *
 * Note: @BR We recommend disabling the IvTune hot key when
 * you distribute your application, especially if it is a commercial application.
 * See the method SoDB::setIvTuneAllowed(). This will
 * prevent users from viewing or changing the application scene graph.
 * Even if the hot key is disabled, the application can still allow opening IvTune,
 * if desired, for example from an "expert" menu using the methods in this class.
 *
 * \if_dotnet
 * Note: @BR In order to use the IvTune hot key feature
 *       with an Open Inventor .NET application, you must explicitly add the
 *       IvTune assembly (OIV.IvTune) to the project references.
 * \endif
 *
 * When IvTune is started from an application, it opens a separate window.
 * This window includes a Tree_View panel which displays the scene graph, a Node_Overview
 * panel to display the fields of a node, and Field_Editor and Field_Watch
 * panels to edit and track interesting fields.
 *
 * IvTune viewer applications
 * can also be used to view Open Inventor scene graph files.
 *
 * IvTune can also be extensively customized and extended using the IvTuneExtender
 * API to create plugins. An IvTune plugin can modify the IvTune user interface,
 * replace standard IvTune panels and even provide new panels that implement
 * specialized operations such a color map editor for SoTransferFunction nodes.
 * Documentation and examples are provided in the source/IvTune subdirectory of the
 * Open Inventor SDK (after installing the Demos package).
 *
 * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{false}]
 * [OIV-WRAPPER-CLASS MODULE{IVTUNE}]
 */
class INVENTOR_API SoIvTune
{

public:

  /** 
   * Starts an IvTune session based on the scene graph managed
   * by the given SoRenderAreaCore.
   * This method has no effect if IvTune are already running.
   * [OIVNET-WRAPPER CUSTOM_CODE]
   */
  static void start( const SoRenderAreaCore* );

  /** 
   * Starts an IvTune session based on the specified scene graph,
   * for example a subset of the application scene graph.
   * This method has no effect if IvTune are already running.
   * [OIVNET-WRAPPER CUSTOM_CODE]
   */
  static void start( SoNode* );

  /** 
   * Ends IvTune. Calling stop() closes the IvTune windows.
   * A new IvTune session can be launched using one of the start() methods.
   */
  static void stop();

  /** 
   * Returns true if IvTune is currently running.
   */
  static bool isRunning();

  /**
   * Process Qt events (calls QCoreApplication::processEvents()).
   *
   * A non-Qt application should call this function periodically so that
   * IvTune's Qt events are handled.  Not necessary for Qt applications.
   *
   * Returns false when IvTune is not running.
   */
  static bool processEvents();

  /** 
   * Defines the keyboard shortcut to use to launch IvTune.
   */
  static void setShortcutValue(SoKeyboardEvent::Key);

  /** 
   * Retrieves the defined keyboard shortcut used to launch IvTune.
   */
  static SoKeyboardEvent::Key getShortcutValue();

#if SoDEPRECATED_BEGIN(9800)
  /**
   * Starts an IvTune session based on the scene graph in an existing SoSceneManager.
   * This method has no effect if IvTune is already running.
   * [OIVNET-WRAPPER CUSTOM_CODE]
   */
  SoDEPRECATED_METHOD(9800,"No longer used. Use start(SoRenderAreaCore*) or start(SoNode*) instead.")
  static void start( SoSceneManager* );
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9000)
  /**
   * Opens an IvTune window displaying the specified scene graph.
   * The argc and argv parameters are not normally used.
   * if createViewer is true then a viewer is created to do the rendering
   * else only the treeview part will be shown.
   */
  SoDEPRECATED_METHOD(9000,"No longer used. Use start() instead.")
  static void setInstance(int argc, char ** argv, SoNode * root, bool createViewer = false);

  /**
   * Opens an IvTune window displaying the scene graph from the specified
   * viewer. The argc and argv parameters are not normally used.
   * When called from non Qt application, the function enters a Qt process events loop
   * ( QCoreApplication::exec() ). Use launch() and processEvents() instead.
   *
   * You should pass the pointer returned by getGuiRenderArea() from your viewer
   * object (which can be derived from SoQtRenderArea, SoXtRenderArea
   * or SoWinRenderArea).
   *
   * \if_cpp
   * For example:
   *  \code
   *  SoIvTune::setInstance ( 0, NULL, pViewer->getGuiRenderArea());
   *  \endcode
   * \endif
   * NOTE: There is currently no way to programmatically close an IvTune
   * window. It can be closed manually, or it will be closed when the
   * program exits.
   */
  SoDEPRECATED_METHOD(9000,"No longer used. Use start() instead.")
  static void setInstance(int argc, char **argv, SoGuiRenderArea* viewer);

   /**
    * Opens an IvTune window displaying the scene graph from the specified
    * viewer, and returns immediatly. The argc and argv parameters are not normally used.
    * After launching IvTune, a non Qt application should call the processEvents function
    * periodically so that Qt events will be treated.
    *
    * You should pass the pointer returned by getGuiRenderArea() from your viewer
    * object (which can be derived from SoQtRenderArea, SoXtRenderArea
    * or SoWinRenderArea).
    *
    * \if_cpp
    * For example:
    *  \code
    *  SoIvTune::setInstance ( 0, NULL, pViewer->getGuiRenderArea());
    *  \endcode
    * \endif
    * NOTE: There is currently no way to programmatically close an IvTune
    * window. It can be closed manually, or it will be closed when the
    * program exits.
    */
  SoDEPRECATED_METHOD(9000,"No longer used. Use start() instead.")
  static void launch(int argc, char **argv, SoGuiRenderArea* viewer);

  /**
   * Returns TRUE if an IvTune window is currently open.
   */
  SoDEPRECATED_METHOD(9000,"No longer used. Use isRunning() instead.")
  static SbBool isInstance();

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  /** 
   * Internal use.
   * Same behavior as start(SoSceneManager*) but allow to reparent 
   * the main frame to a native window.
   * Usefull to ensure a correct behavior of IvTune within a
   * .Net application.
   * [OIVNET-WRAPPER VISIBILITY{Internal}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{Widget}]
   */
  static void start( SoSceneManager*, void* parentWindow );

  /** 
   * Internal use.
   * Same behavior as start(SoNode*) but allow to reparent 
   * the main frame to a native window.
   * Usefull to ensure a correct behavior of IvTune within a
   * .Net application.
   * [OIVNET-WRAPPER VISIBILITY{Internal}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{Widget}]
   */
  static void start( SoNode*, void* parentWindow );

  /**
   * NB: IvTune must be running when making a call to setPickAction().
   */
  static void setPickAction( const SbViewportRegion &region, SbVec2s mouseCoord );

  /**
   *
   */
  static void setSceneGraph(SoNode* node);

  /** Destructor. */
  ~SoIvTune();

private:

  /** Private constructor to avoid the default one generated by the compiler */
  SoIvTune();
};

#endif //SoIvTune_included

