/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T. DUFOUR (Sep 2007)
**=======================================================================*/


#ifndef  _SO_LDM_WRITER_
#define  _SO_LDM_WRITER_

#include <LDM/nodes/SoDataSet.h>
class SoLDMWriterConverter;
class SoLDMWriterReader;
class SoConverterParameters;

/**
 * @LDMEXT Write data to disk in LDM format.
 *
 * @ingroup LDMConverters
 *
 * @DESCRIPTION
 *   SoLDMWriter creates an LDM file (in the VSG .ldm format)
 *   and allows the application to store data blocks in any order.  The data blocks 
 *   may be specific tiles or arbitrary regions of the volume.  The most common 
 *   usage is to store blocks of full resolution data.  SoLDMWriter incorporates an 
 *   instance of the LDM converter (see SoConverter) which can automatically generate 
 *   the lower resolution (subsampled) tiles after the full resolution data has 
 *   been stored (for example when the finish() method is called).  However the 
 *   writeTile() method also allows the application to directly store lower 
 *   resolution tiles in case a proprietary subsampling algorithm is being used.  
 *   The result will normally be a .ldm file (LDM header) and a .dat file (data).
 *
 *   Because the LDA format used by Amira and Avizo is an LDM format, it's possible
 *   to generate a LDA file if this extension is required for the generated file.
 *   If the output filename has a ".lda" extension, the LDMWriter will
 *   keep this extension. 
 *   
 *   It is not necessary, in all cases, to create the lower resolution tiles or even 
 *   to create the complete set of full resolution tiles.  SoLDMWriter supports the 
 *   converter's partial conversion feature.  If some tiles are missing when the 
 *   #finish method is called, then in addition to the header and data files there 
 *   will also be a .fcp file with the same name as the header and data files.  The 
 *   LDM header file will contain a reference to this file in the CompletionFilename 
 *   tag.  The .fcp file stores information about which tiles actually exist.  Tiles 
 *   that do not exist are considered to be filled with a constant default value 
 *   (see getHoleData() in the SoLDMReader class).  This feature allows us, for 
 *   example, to compute and visualize a subset of the volume before committing to 
 *   computation on the whole volume.  However note that the converter currently 
 *   pre-allocates disk space for the ".dat" file assuming that all tiles will exist.
 *   So skipping creation of the subsampled tiles or writing only a subset of the 
 *   tiles can reduce computation time and disk I/O time, but it does not reduce the 
 *   disk space requirement for the volume.
 *   
 *   The output volume is not required to have the same characteristics as the input 
 *   volume.  This allows for example, to extract a subset of a volume as a new 
 *   (smaller) volume.  All the options of the LDM converter are available.  So the 
 *   LDM writer can even create a volume with a different data type.
 *   
 *   The return value from most methods is the enum SoConverter::ConverterError.
 *   Success is indicated by the value CVT_NO_ERROR.
 *   
 *   The first step is to create an SoLDMWriter object and initialize it with the 
 *   name of the output file to be created and the desired characteristics for the
 *   output volume. There are two versions of the #initialize() method:
 *   - One takes a reference to an existing data set (SoVolumeData) and
 *     creates a new volume with the same characteristics.  This is useful,
 *     for example, when creating a derived volume such as seismic attributes.
 *
 *   - The other takes explicit parameters for the volume dimensions, extent
 *     and data type.
 *   
 *   Using either version it will often be necessary to specify other parameters and
 *   options to the converter using an SoConverterParameters object. Some useful and
 *   commonly used options are the tile dimensions and data compression.
 *
 *   If the specified output file ('filename' parameter) does not exist,
 *   the class will create it.
 *
 *   @B NOTE@b: If the file already exists (and is in LDM
 *   format), the class will attempt to modify the existing file.  This can be useful,
 *   but can also be an unexpected result.  If the goal is to create a new file, we
 *   recommend checking if the file already exists (see SbFileHelper) and deleting it
 *   if necessary.
 *
 *   The next step is to write data into the LDM file using the
 *   \if_dotnet #WriteSubVolume() \else #writeSubVolume() \endif
 *   or \if_dotnet #WriteTile() \else #writeTile() \endif methods.  
 *   Writing a subvolume is the most powerful and convenient
 *   method because it accepts an arbitrary subvolume (writing partial tiles if
 *   necessary).  If your subvolume is an integral number of tiles and aligned 
 *   on tile boundaries, then writing tiles will be more 
 *   efficient for writing the full resolution tiles.  The subvolume method also has 
 *   an option to immediately generate the corresponding lower resolution tiles.
 *   If the subvolume contains "blocks" of 8 tiles that can be combined into 
 *   lower resolution tile, this method provides much better performance than
 *   the #finish method, because the full resolution data is already in memory.
 *   The following example writes eight 64x64x64 tiles into an LDM file of a volume with
 *   128*128*128 dimension.
 *
 *   The final step is to call the #finish method.  This will optionally
 *   generate the lower resolution tiles, cleanup and close the LDM file.
 *
 * @EXAMPLE
 *   \if_cpp
 *   \code
 *     // Initialize the LDM writer
 *     SbBox3f   extent( 0,0,0, 1,1,1 );
 *     SbVec3i32 dimensions( 128, 128, 128 );
 *     SbVec3i32 tileSize( 64, 64, 64 );
 *
 *     size_t TileSampleCount = tileSize[0];
 *     TileSampleCount *= tileSize[1];
 *     TileSampleCount *= tileSize[2];
 *
 *     SoDataSet::DataType type = SoDataSet::UNSIGNED_BYTE;
 *     SoConverterParameters parameters;
 *
 *     SoLDMWriter myLDMWriter;
 *     int rc = myLDMWriter.initialize( "myldmfile.ldm", extent, dimensions, type, &parameters );
 *
 *     if ( rc == SoConverter::CVT_NO_ERROR )
 *     {
 *       for ( int x = 0; x*tileSize[0] < dimensions[0]; ++x )
 *       {
 *         for ( int y = 0; y*tileSize[1] < dimensions[1]; ++y )
 *         {
 *           for ( int z = 0; z*tileSize[2] < dimensions[2]; ++z )
 *           {
 *             // Create buffer for tile data
 *             SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject;
 *             buffer->setSize( tileSampleCount * sizeof(unsigned char) ); // Tile size is 64
 *
 *             // Compute data and write tile
 *             computeData( buffer.ptr() );
 *             SbVec3i32 tilePosition( x, y, z );
 *             rc = myLDMWriter.writeTile( tilePosition, buffer.ptr() );
 *           }
 *         }
 *       }
 *
 *       // Complete file creation
 *       rc = myLDMWriter.finish();
 *     }
 *   \endcode
 *   \endif
 *   \if_dotnet
 *   \code
 *     // Initialize the LDM writer
 *     SbBox3f   extent     = new SbBox3f( 0,0,0, 1,1,1 );
 *     SbVec3i32 dimensions = new SbVec3i32( 128, 128, 128 );
 *     SbVec3i32 tileSize   = new SbVec3i32( 64, 64, 64 );
 *
 *     SoDataSet.DataTypes type = SoDataSet.DataTypes.UNSIGNED_BYTE;
 *     SoConverterParameters parameters = new SoConverterParameters();
 *
 *     SoLDMWriter myLDMWriter = new SoLDMWriter();
 *     int rc = myLDMWriter.Initialize( "myldmfile.ldm", extent, dimensions, type, parameters );
 *
 *     if ( rc == (int) SoConverter.ConverterErrors.CVT_NO_ERROR )
 *     {
 *       for ( int x = 0; x*tileSize[0] < dimensions[0]; ++x )
 *       {
 *         for ( int y = 0; y*tileSize[1] < dimensions[1]; ++y )
 *         {
 *           for ( int z = 0; z*tileSize[2] < dimensions[2]; ++z )
 *           {
 *             // Create buffer for tile data
 *             SoCpuBufferObject buffer = new SoCpuBufferObject();
 *             buffer.SetSize( tileSize[0] * tileSize[1] * tileSize[2] );
 *
 *             // Compute data and write tile
 *             computeData( buffer );
 *             SbVec3i32 tilePosition( x, y, z );
 *             rc = myLDMWriter.WriteTile( tilePosition, buffer );
 *           }
 *         }
 *       }
 *
 *       // Complete file creation
 *       rc = myLDMWriter.Finish();
 *     }
 *   \endcode
 *   \endif
 *   \if_java
 *   \code
 *     // Initialize the LDM writer
 *     SbBox3f   extent     = new SbBox3f( 0,0,0, 1,1,1 );
 *     SbVec3i32 dimensions = new SbVec3i32( 128, 128, 128 );
 *     SbVec3i32 tileSize   = new SbVec3i32( 64, 64, 64 );
 *
 *     SoDataSet.DataTypes type = SoDataSet.DataTypes.UNSIGNED_BYTE;
 *     SoConverterParameters parameters = new SoConverterParameters();
 *
 *     SoLDMWriter myLDMWriter = new SoLDMWriter();
 *     int rc = myLDMWriter.initialize( "myldmfile.ldm", extent, dimensions, type, parameters );
 *
 *     if ( rc == SoConverter.ConverterErrors.CVT_NO_ERROR.getValue() )
 *     {
 *       for ( int x = 0; x*tileSize[0] < dimensions[0]; ++x )
 *       {
 *         for ( int y = 0; y*tileSize[1] < dimensions[1]; ++y )
 *         {
 *           for ( int z = 0; z*tileSize[2] < dimensions[2]; ++z )
 *           {
 *             // Create buffer for tile data
 *             SoCpuBufferObject buffer = new SoCpuBufferObject();
 *             buffer.setSize( tileSize[0] * tileSize[1] * tileSize[2] ); // Tile size is 64
 *
 *             // Compute data and write tile
 *             computeData( buffer );
 *             SbVec3i32 tilePosition( x, y, z );
 *             rc = myLDMWriter.writeTile( tilePosition, buffer );
 *           }
 *         }
 *       }
 *
 *       // Complete file creation
 *       rc = myLDMWriter.finish();
 *     }
 *   \endcode
 *   \endif
 *
 * @SEE_ALSO
 *    SoVolumeConverter
 *
 *
 */
class LDM_API SoLDMWriter
{
 public:
 /**
   * Constructor.
   */
  SoLDMWriter();

  /**
   * Destructor.
   */
  virtual ~SoLDMWriter();

  /**
   * Specifies the callback that will be called when the LDM file header is generated.
   * This allows the application to add custom XML tags.
   * The custom tags can be read back using the method SoVRLdmFileReader::getXmlTag().
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  void setXmlCallback( void (*xmlCB)( FILE*, void * userData ), void * userData );

  /** 
   * Initializes the writer to build a volume with the same dimensions, size and
   * data type as @B inputVolData @b.  Other parameters (e.g. tile size)
   * and converter options can be specified using SoConverterParameters.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  int initialize( const SbString & filename,
                  const SoDataSet & inputVolData,
                  SoConverterParameters* parameters );

  /**
   * Initializes the writer to build a volume @B filename @b with the specified
   * dimensions, size and data type.  Other parameters (e.g. tile size)
   * and converter options can be specified using SoConverterParameters.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  int initialize( const SbString & filename,
                  const SbBox3f & size, const SbVec3i32 & dimension, SoDataSet::DataType dataType,
                  SoConverterParameters* parameters );

   /** 
   * Writes the @B data @b contained in the specified @B subVolume @b into the LDM file. @BR
   * All tiles that are completely inside the subvolume are written into the file overwriting any
   * already existing data for those tiles. (Overwritten values are still taken into account for the histogram).  
   * All tiles that are partially inside the subvolume are also written into the file.  
   * If a partially inside tile already exists in the file(perhaps from a previous call 
   * to writeSubVolume), that tile will first be read back from the file, updated with 
   * the new voxel values, then re-written into the file.
   *
   * If @B doLowerResolution @b is true (default), builds the lower resolution tiles immediately, 
   * if possible. The converter needs 8 neighbors tiles of max resolution to build one of resolution max-1
   * low resolution. So if the given subVolume does not contain those 8 neighbors, the low resolution tile
   * cannot be built during this call. @B It's then important to notice that a call to writeSubVolume 
   * with doLowerResolution set to true and a call to finish(false) can potentially result in missing 
   * low resolution tile in the generated LDM file. @b Otherwise, this can be significantly faster than 
   * building them in the finish() method, because the full resolution data is already in memory. 
   * If @B doLowerResolution @b is false, the lower resolution tiles are not generated during this call, 
   * but can still be generated later if the finish() method is called with @B doMultiResolution @b set to true.
   *
   * @B doOverlappingTiles @b is no longer used and ignored since Open Inventor 9.0
   *
   * Returns a result code defined by the enum SoConverter::ConverterError (zero if successful).
   */
  int writeSubVolume( const SbBox3i32 & subVolume, SoBufferObject* data, SbBool doLowerResolution = TRUE, SbBool doOverlappingTiles = TRUE );

  /**
   * Writes the tile which includes @B location @b in data coordinates,
   * at the specified resolution level, into the LDM file.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  int writeTile( const SbVec3i32 & location, SoBufferObject* data, int resolution = 0 );

  /**
   * Writes the specified tile @B tileID @b into the LDM file.
   */
  int writeTile( const SoLDMTileID & tileID, SoBufferObject* data );

  /**
   * Finishes writing the LDM file and optionally invokes the converter to 
   * generate the lower resolution tiles. Default is to invoke the converter.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  int finish( SbBool doMultiResolution = TRUE );

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_METHOD(8000,"Use writeSubVolume( const SbBox3i32 & , SoBufferObject*, SbBool , SbBool ) instead.")
  int writeSubVolume( const SbBox3i32 & subVolume, const void * data, SbBool doLowerResolution = TRUE, SbBool doOverlappingTiles = TRUE );

  SoDEPRECATED_METHOD(8000,"Use writeTile( const SbVec3i32 & , SoBufferObject*, int) instead.")
  int writeTile( const SbVec3i32 & location, void * data, int resolution = 0 );

  SoDEPRECATED_METHOD(8000,"Use writeTile( const SoLDMTileID &, SoBufferObject*) instead.")
  int writeTile( const SoLDMTileID & tileID, void * data );

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Initializes the writer to build a volume with the same dimensions, size and
   * data type as @B inputVolData @b.  Other parameters (e.g. tile size)
   * and converter options (e.g. "-q" to suppress output) can be specified using
   * @B argc @b and @B argv @b.  See SoConverter::convert for a list of command
   * line arguments.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  SoDEPRECATED_METHOD(9000,"Use equivalent method that take SoConverterParameters as parameter.")
  int initialize( const SbString & filename,
                  const SoDataSet & inputVolData,
                  int argc = 0, char** argv = NULL );

  /**
   * Initializes the writer to build a volume @B filename @b with the specified
   * dimensions, size and data type.  Other parameters (e.g. tile size)
   * and converter options (e.g. "-q" to suppress output) can be specified using
   * @B argc @b and @B argv @b.  See SoConverter::convert for a list of command
   * line arguments.
   * Result code is defined by the enum SoConverter::ConverterError.
   */
  SoDEPRECATED_METHOD(9000,"Use equivalent method that take SoConverterParameters as parameter.")
  int initialize( const SbString & filename,
                  const SbBox3f & size, const SbVec3i32 & dimension, SoDataSet::DataType dataType,
                  int argc = 0, char** argv = NULL );

#endif /** @DEPRECATED_END */

SoINTERNAL public:
  // Original declaration - didn't use standard OIV data type - kept for compatibility
  int finish( bool doMultiResolution )
    { return finish( (SbBool)doMultiResolution ); }

  void disableWarning(bool flag) { m_warning = flag; };

protected:
  void (*m_xmlCB)( FILE*, void* );
  void  *m_xmlCBUserData;

  bool   m_initiated;
  int    m_retCode;
  SoLDMWriterConverter *m_converter;
  SoLDMReader* m_reader;
  void   release();


  /** return true if init is done from an existing file. */
  bool   init(const SbString & filename);

  int    initiateConverter( const SbString & filename, int argc = 0, char** argv = NULL );
  int    initiateConverter( const SbString & filename, SoConverterParameters* parameters );

private:
  /** A simple flag to know if we need to warn the customer */
  bool m_warning;

};

#endif // _SO_LDM_WRITER_


