/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Hummel (Oct 2006)
**=======================================================================*/


#ifndef  _SO_DATA_RANGE__
#define  _SO_DATA_RANGE__

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFDouble.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFInt32.h>

#include <LDM/nodes/SoDataSet.h>
#include <Inventor/algorithms/SoConversion.h>

/**
 * @LDMEXT Range of data values to be mapped to the color map.
 *
 * @ingroup LDMNodes
 *
 * @DESCRIPTION
 *
 * This node allows you to specify the range of scalar data values in a data set
 * (SoDataSet or SoVolumeData) that will be mapped into the color map (SoTransferFunction).
 * When the volume data type is larger than the data type on the GPU, for example
 * float data scaled to byte data, it also specifies the range of scalar data values
 * that will be scaled to fit in the GPU data type.
 *
 *   @IMAGE colormapremap.jpg
 *
 * By default VolumeViz maps the entire range of the voxel's data type
 * (e.g. 0..65535 for unsigned short) into the colormap.  This is often
 * correct for byte (8 bit) voxels, but seldom correct for 16 bit voxels
 * and never correct for floating point voxels. Use an SoDataRange node
 * to specify the actual (or desired) range of data values to be mapped.
 * For example, a typical initial data range for DICOM data calibrated
 * in Hounsfield units might be -1000 to 3000.
 *
 * Data characteristics:
 *
 * - The voxel value data type can be queried using the SoDataSet method getDataType().
 *
 * - The number of bytes in a voxel value can be queried using the SoDataSet method getDataSize().
 *
 * - The actual min and max values for the data set can be queried using the SoDataSet
 *   method getMinMax(). @BR @BR
 *   Note that this method might force VolumeViz to load the entire data set if the
 *   volume reader for that format does not implement the getMinMax query. Normally
 *   for an LDM format data set, the min and max values are stored in the LDM header.
 *   For a non-LDM data set, if a filename and/or reader have been specified and the
 *   data set has not yet been loaded, VolumeViz will load the entire data set to 
 *   compute the min and max values. For a large data set this may take a long time.
 *
 * - DICOM: The Window Center (0028,1050) and Window Width (0028,1051) values (if
 *   present in the data set) can be queried by using an SoVRDicomData object obtained
 *   from the SoVRDicomFileReader or by using the MedicalHelper class.  You can also
 *   use the MedicalHelper method dicomAdjustDataRange.
 *
 * When using multiple volumes (see SoMultiDataSeparator), a single SoDataRange node
 * can be used to specify the data range for all volumes or each volume can have its
 * own independent data range. In the second case, create one SoDataRange node for each
 * volume and set the #dataRangeId equal to the SoDataSet::dataSetId of the corresponding volume.
 *
 * Note that the meaning of the min and max fields in SoDataRange is quite different than
 * the meaning of the minValue and maxValue fields in SoTransferFunction.
 * The fields in SoDataRange specify the range of voxel values that will be mapped into the full
 * color map. The fields in SoTransferFunction specify the range of indices in the
 * color map that will actually be used to store the color map. The visual effect
 * changing these fields can be quite similar, but there are trade-offs to be aware
 * of.  Changing the SoTransferFunction fields is generally much faster and can be a
 * useful approximation of changing the data range, but the resolution of the color
 * map (the ratio of data values to color map entries) is reduced.
 *
 * NOTE: Setting the min value greater than or equal to the max value will cause this node to be ignored.
 *
 * @B Brightness and contrast:@b
 * \par
 * Often the distribution of voxel values within the actual data range is not uniform and
 * more details can be seen by adjusting the SoDataRange values to increase brightness
 * and/or contrast. This is particularly true when using a gray scale color map (for
 * example the predefined INTENSITY map in SoTransferFunction ). In medical imaging
 * this range setting is often specified by the @I window center@i and @I window width@i.
 * @BR @BR
 * The window center is the image intensity that will be displayed as a medium-gray and the window
 * width is the range of data values between full black and full white. For example, if
 * the data volume contains byte data with a native range of 0 to 255, the default data
 * range (0 to 255) is effectively specifying a window center of 128 and width of 256. To increase
 * contrast in the resulting image we might set a data range of 20 to 110, which makes
 * the window center and width 65 and 90 respectively.
 *
 * @B Performance:@b
 * \par
 * For larger data types, changing the data range may require VolumeViz to recreate the data
 * textures on the GPU.  This is necessary to maximize use of the available bits of precision
 * on the GPU.  Since Open Inventor 9.6, recreating data textures should not be required for
 * 8 bit data or for 16 bit data when the #SoDataSet::texturePrecision field is set to 16.
 * @BR @BR
 * Recreating data textures may be slow for larger volumes, even in the 512^3 range.
 * There are several parameters that significantly affect this:
 *
 *   - Tile size: @BR
 *     For backward compatibility, the default tile size is still only 64. This is quite
 *     small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
 *     number of tiles that must be managed by VolumeViz. This overhead can be significant,
 *     especially for operations that require reloading the data textures on the GPU, for
 *     example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
 *     it can be efficient to set the tile size large enough to contain the entire volume.
 *     For very large volumes, larger tile sizes are efficient for SoVolumeRender but
 *     somewhat inefficient for slice rendering because complete tiles must be loaded
 *     even though the slice only uses part of the data.  Applications should experiment.
 *     @BR @BR
 *     For volumes stored in LDM file format, the tile size must be specified when the
 *     volume is converted to LDM (see SoConverter and the "-t" option).  For other data
 *     data formats the tile size should be specified after the SoVolumeData node is
 *     created, using the @I ldmResourceParameters@i field, but before setting the 
 *     @I filename@i field.
 *
 *   - Tile cache policy:  
 *     It specifies how the tiles are stored in CPU memory. The selected policy
 *     can significantly impact the data loading performance versus the CPU memory footprint.
 *     See SoLDMResourceParameters::tileCachePolicy for detail.
 *
 *   - SoDataSet::texturePrecision @BR
 *     When the data set's texturePrecision is at least equal to the numSignificantBits,
 *     then the data range scaling can be applied on the fly during rendering (i.e. on the GPU). This
 *     implies that data textures don't need to be regenerated when the data range changes
 *     and interactivity is much better.
 *
 *   - SoVolumeRender::subDivideTiles @BR
 *     This parameter doesn't actually resend data to the GPU but it does require updating
 *     the sub-tile min/max values, which can be quite costly.
 *     Setting subDivideTiles to FALSE may increase DataRange performance (the default is FALSE).
 *
 * If the data set contains explicit RGBA values (SoVolumeData dataRGBA field is true),
 * then SoDataRange and SoTransferFunction have no effect on rendering.
 *
 * @FILE_FORMAT_DEFAULT
 * DataRange{
 *    @TABLE_FILE_FORMAT
 *    @TR dataRangeId           @TD 1
 *    @TR min                   @TD 0
 *    @TR max                   @TD 0
 *    @TR mapOnFullColorRange   @TD TRUE
 *    @TABLE_END
 * }
 *
 * @ACTION_BEHAVIOR
 * SoCallbackAction,
 * SoGLRenderAction,
 * SoPickAction,
 * SoWriteAction,
 * SoGetBoundingBoxAction @BR
 * Sets a data range in the current traversal state.
 *
 * @SEE_ALSO
 *    SoDataSet,
 *    SoTransferFunction
 *
 *
 */
class LDM_API SoDataRange : public SoNode {
  SO_NODE_HEADER( SoDataRange );

public:
  /**
   * Constructor.
   */
  SoDataRange();

 /**
  *
  * This field allows the use of multiple data ranges for the same shape.
  * By default all data range nodes are initialized to a data range id of 1.
  * If you want to use multiple data ranges, different data range ids must be assigned.
  * The data range id is only used in a render compositing scheme (see SoDataSet).
  *
  * @FIELD_SINCE_OIV 6.1.2
  */
  SoSFInt32 dataRangeId;

  /**
   * Minimum data value of the data range.
   * @B Note:@b If #min is greater than or equal to #max, this node is ignored.
   */
  SoSFDouble min;

  /**
   * Maximum data value of the data range.
   */
  SoSFDouble max;

  /**
   * Specifies how to map data values that are outside of the data range.
   * If mapOnFullColorRange is TRUE (default), then any data values less than or equal
   * to  the #min data value are mapped to the first color entry and any data values
   * greater than or equal to the #max data value are mapped to the last color entry.
   *
   * If mapOnFullColorRange is FALSE, then any data value less than
   * #min data value is mapped to the first color entry and the #min data value
   * is mapped to the second color entry; any data value greater than
   * the #max data value is mapped to the last color entry and the #max data value is mapped
   * to the next to the last color entry.
   */
  SoSFBool mapOnFullColorRange;

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /**
   * Function that compute datarange to index for generic bufferObject.
   * LIMITATION: only UNSIGNED_BYTE/UNSIGNED_SHORT dataTypeDst is supported 
   */
  static void mapDataRangeToIndex(
    const SoDataSet::DataType dataTypeSrc, SoBufferObject* sourceBufferObject,
    const SoDataSet::DataType dataTypeDst, SoBufferObject* targetBufferObject,
    const double dataRangeMin, const double dataRangeMax, const bool dataRangeMap,
    const double undefinedValue = std::numeric_limits<double>::quiet_NaN()
    );

  
  /**
   * Function that compute datarange to RGBA for generic bufferObject.
   * LIMITATION: only UNSIGNED_INT32 (for RGBA) dataTypeDst is supported 
   */
  static void mapDataRangeToRgba(
    const SoDataSet::DataType dataTypeSrc, SoBufferObject* sourceBufferObject,
    const SoDataSet::DataType dataTypeDst, SoBufferObject* targetBufferObject,
    const double dataRangeMin, const double dataRangeMax, const bool dataRangeMap,
    SoBufferObject *rgba, const int numRgba, SoConversion::MappingMethod rgbaMapping
    );

  /**
   * Function that convert data to texture index for generic bufferObject.
   * LIMITATIONS: 
   * - only UNSIGNED_BYTE/UNSIGNED_SHORT dataTypeDst is supported 
   * - FLOAT dataTypeSrc is not supported
   */
  static void mapDataToIndex(
    const SoDataSet::DataType dataTypeSrc, SoBufferObject* sourceBufferObject,
    const SoDataSet::DataType dataTypeDst, SoBufferObject* targetBufferObject,
    const int numSigBits
    );

 
  /**
   * Function that convert data to rgba texture for generic bufferObject.
   * LIMITATIONS: 
   * - only UNSIGNED_INT32 (for RGBA) dataTypeDst is supported 
   * - FLOAT dataTypeSrc is not supported
   */
  static void mapDataToRgba(
    const SoDataSet::DataType dataTypeSrc, SoBufferObject* sourceBufferObject,
    const SoDataSet::DataType dataTypeDst, SoBufferObject* targetBufferObject,
    const int numSigBits,
    SoBufferObject *rgba, const int numRgba, SoConversion::MappingMethod rgbaMapping
    );

 
SoEXTENDER public:
  virtual void doAction( SoAction *action );
  virtual void GLRender( SoGLRenderAction *action );
  virtual void callback( SoCallbackAction *action );
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void pick(SoPickAction *action) ;
  virtual void write(SoWriteAction *action);

protected:
  // Destructor
  virtual ~SoDataRange();

private:

  static SoConversion* s_conversion;
};

#endif // _SO_DATA_RANGE__


