/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Hummel (Mar 2006)
** Modified by : Benjamin Grange (MMM yyyy)
** Modified by : Jean-Michel Godinaud. (MMM yyyy)
**=======================================================================*/


#ifndef _SO_DATA_SET_
#define _SO_DATA_SET_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

//includes
#include <Inventor/SbBox.h>
#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFieldContainer.h>
#include <Inventor/fields/SoSFFilePathString.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFDouble.h>
#include <Inventor/SbDataType.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/misc/SoMemoryObject.h>
#include <Inventor/misc/SoRef.h>
#include <Inventor/nodes/SoNode.h>
#include <LDM/SoLDM.h>
#include <LDM/SoLDMMediator.h>
#include <LDM/SoLDMTileID.h>
#include <LDM/fields/SoSFLDMResourceParameters.h>
#include <LDM/fields/SoSFLDMDataTransform.h>
#include <LDM/nodes/SoLDMResourceParameters.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/tiles/SoCpuBufferUniform.h>
#include <Inventor/sys/SoGLType.h>

// TODO: Should be removed : For compatibility only
#include <Inventor/devices/SoCpuBufferObject.h>

class SoGLRenderAction;
class SoCallbackAction;
class SoHandleEventAction;
class SoWriteAction;
class SoState;
class SoLDMGeomElement;
class SoLDMGeometry;
class SoLDMReader;
class SoTransferFunction;
class SoMultiDataSeparator;
class SoVolumeHistogram;
class SoVolumeReader;
class SoPerfCounterManager;
class SoPerfCounter;
class SoBufferObject;
class SoCpuBufferUniform;
class SoGLTexture;
class SoCache;

class SoDataExtract;
class SoConversion;
class SoAlgorithms;
class SoLDMAlgorithms;
class SoLdmValuationAction;
class SoDataRange;

class SoVolumeWriter;

SO_PIMPL_PUBLIC_DECLARATION(SoDataSet)


/**
*
* @LDMEXT Data set node.
*
* @ingroup LDMNodes
*
* @DESCRIPTION
*   This class specifies a data set and its properties, and also provides utilities for
*   extracting a subset of the data set and for resampling the data set. The data can be
*   read directly from a file or can be accessed via a user-defined volume reader.
*   This class is the parent class of the VolumeViz data nodes: SoVolumeData,
*   SoHeightFieldGeometry, SoHeightFieldProperty, etc. Those nodes provide the data for
*   VolumeViz rendering nodes: SoVolumeRender, SoOrthoSlice, SoObliqueSlice, etc.
*   See SoVolumeData for more details about loading and managing volume data.
*
* The data set can be specified by:
*
*   - Setting the #fileName field @BR
*      This implies that the data set is stored on disk, in one of the file
*      formats for which LDM has a built-in reader. LDM
*      will automatically select a reader based on the file extension,
*      for example ".am" for the AmiraMesh file format.
*
*   - Calling the setReader() method @BR
*      This is the most general method because an application can
*      specify one of the standard LDM readers or implement
*      a customized subclass of SoVolumeReader.
*      LDM will get the data set properties (dimensions, size, data type, etc) and
*      access the data through the specified reader object. This allows the
*      application to completely control how, and from where, the data is loaded.
*
* @SEE_ALSO
*    SoDataSetId,
*    SoDataCompositor,
*    SoVolumeData,
*    SoMeshGeometry,
*    SoMeshProperty
*
*
* [OIVJAVA-WRAPPER-CLASS ALL_DERIVABLE]
*/
class LDM_API SoDataSet : public SoNode
{
  SO_NODE_ABSTRACT_HEADER( SoDataSet );

  SO_PIMPL_PUBLIC_HEADER( SoDataSet );

public:

  /**
  * When using multiple SoDataSet nodes, the #dataSetId field uniquely
  * identifies each data set used in the compositing.
  * It also specifies the OpenGL texture unit in which the textures for
  * this data set will be stored when doing render compositing.
  * It is 1 by default (texture unit 0 is reserved for the color lookup table by default).
  * The number of available texture units depends on your hardware. You can query
  * this number using #getMaxNumDataSets.
  *
  * Note: The SoDataSetId node can also be used to define dataSetId.
  * If an SoDataSetId node is traversed before the SoDataSet node,
  * the id from the SoDataSetId node is used and this field is ignored.
  *
  * @FIELD_SINCE_OIV 6.0
  */
  SoSFInt32 dataSetId;

  /**
   * Indicates the file location containing the data set.
   * Default is an empty string. Depending on the file
   * type, the corresponding SoVolumeReader (if it exists) is used.
   *
   *    @TABLE_1B
   *       @TR @B File Extension @b @TD @B Loader Class @b  @TD @B Description @b
   *       @TR .am                  @TD SoVRAmFileReader    @TD Amira Mesh file format
   *       @TR .dc3, .dic, .dicom   @TD SoVRDicomFileReader @TD Dicom file format
   *       @TR .fld                 @TD SoVRAvsFileReader   @TD AVS field file format
   *       @TR .lda or .ldm         @TD SoVRLdmFileReader   @TD Large Data Management file format
   *       @TR .sgy or .segy        @TD SoVRSegyFileReader  @TD SEG Y revision 1 file format
   *       @TR .vol                 @TD SoVRVolFileReader   @TD Vol file format
   *       @TR .vox                 @TD SoVRVoxFileReader   @TD Vox file format
   *       @TR .lst	                @TD SoVRRasterStackReader @TD Lst file format
   *    @TABLE_END
   *
  * The filename extension is not case sensitive.
  *
  * Note: SoDataSet and its derived classes (SoVolumeData, etc) do @I not @i search
  * the SoInput directory list to find files specified in this field.  To check file
  * existence and other properties the SbFileHelper class may be useful.
  */

  SoSFFilePathString  fileName;

  /**
  * Controls use of OpenGL lossy texture compression for RGBA data (if available).
  * Default is TRUE.
  *
  * A significant advantage of compressed textures is that they conserve texture memory space.
  * At the moment, the extension works on RGBA textures only, not on paletted textures.
  * The compression factor is about 5.
  *
  * A potential drawback of compressed textures is that lossy compression can result in
  * values on the GPU that are slightly different from the actual value in the volume.
  * For example "gray scale" colors could have a slight color in some cases.
  * You should always set this option to false when using "RGBA" data to store 32-bit "id" values.
  */
  SoSFBool   useCompressedTexture;

  /**
  * For scalar (non-RGBA) data, specifies the size of voxel values on the GPU in bits.
  * The valid values are: 0 (default), 8 and 16. @BR
  * When set to 0, LDM chooses the best precision based on the size of the data type
  * (or if specified, the number of significant bits) - meaning that LDM will store
  * 8-bit values on the GPU for 8-bit data and store 16-bit values for all other data types.
  *
  * There are several trade-offs influenced by this value, including:
  * - 8 bit values use less memory on the GPU than 16-bit values, allowing more
  *   data to be stored on the GPU. 8-bit values are adequate for many visualization
  *   tasks. But...
  *
  * - If the specified (or chosen) precision is smaller than the data type, then
  *   LDM must @I scale down @i the data values before sending them to the GPU.
  *   - Limitation: This is currently unavoidable for 32-bit data.
  *   - Down scaling adds some time whenever a tile of data must be sent to the GPU.
  *   - In particular, all tiles currently stored on the GPU must be @I rescaled @i
  *     if the application changes the data range (SoDataRange node). @BR
  *     @B Note:@b As a result, changing the data range may be slow for large volumes.
  *   - Down scaling also means "aliasing" data values, i.e. multiple data values
  *     may scale to the same value on the GPU.
  *
  * @FIELD_SINCE_OIV 6.0
  */
  SoSFUShort texturePrecision;

  /**
   * The real size (extent) of the volume in modeling coordinates.
   * Default is -1,-1,-1 to 1,1,1.
   *
   * Note this can be used to define a volume with non-uniform voxel
   * spacing.  For example, if the volume dimensions (voxels) are 64x64x64,
   * but the voxel spacing in Z is twice the X and Y spacing, then you
   * might specify the volume size as -1 to 1 for X and Y but -2 to 2 for Z.
   */
  SoSFBox3f extent;

  /**
   * If set to an appropriate SoLDMDataTransform object,
   * the object's transformFunction method is called after each tile is loaded,
   * but before it is stored in main memory.
   * This allows you to modify the original data (for example, scaling or
   * filtering) before it is displayed. Default is no transform.
   *
   * Note: Before Open Inventor 8.0, the data transformFunction was set
   * \if_cpp
   * using the setLDMDataTransformFunction method. That method is now deprecated.
   * \endif
   * \if_java
   * using the setLDMDataModifier method. That method is now deprecated.
   * \endif
   * \if_dotnet
   * using the LDMDataTransformCallback property. That property is now deprecated.
   * \endif
   *
   * @FIELD_SINCE_OIV 8.0
   */
  SoSFLDMDataTransform dataTransform;

  /**
  * Indicates if resource allocation is done only on first render traversal
  * or as soon as the node is created. Default is TRUE. It must be set to FALSE
  * if the dataset is not inserted in a scene graph but will be used for data 
  * access for example. Otherwise, memory resources will still remain on default
  * value.
  */
  SoSFBool   allocateResourceOnRender;

  /**
   * Specifies the data value that should be considered undefined and therefore not rendered. @BR
   * By default, this field is set to NaN (Not a Number), meaning that no specific value is excluded. @BR
   * Otherwise, the value must match the data type.@BR
   *
   * @B Limitations: @b
   * There are several limitations in order for the undefined value to work:
   * - The texturePrecision must be at least equal to the number of significant bits of the
   * data type.
   * - The data type must be BYTE, SHORT (either SIGNED or UNSIGNED) or FLOAT
   * - The field is ignored in the following classes : SoHeightFieldProperty,
   * SoHeightFieldPropertyMask and SoVolumeMask. Setting this field has no effect in these cases.
   *
   * @B Note: @b
   * If the application uses custom shaders, particularly when overriding the VVizComputeFragmentColor() function:
   * - Undefined values are handled in the GLSL function VVizTransferFunction().
   * - If the voxel value is undefined, VVizTransferFunction() returns the RGBA value 0,0,0,0. You may return this value for your undefined value.
   * - If the application's shader does not call VVizTransferFunction() (e.g. voxel color is computed without using a colormap), the
   *   application's shader must explicity check for undefined values. This can be done using the function VVizIsUndefined()
   *
   * @FIELD_SINCE_OIV 10.11
   */
  SoSFDouble undefinedValue;

  /**
   * Specifies an image type: segmented or not segmented.
   * A segmented image only contains discrete values.
   * Used with #valueInterpretation field.
   */
  enum ValueInterpretation
  {
  /**
   * Specifies an image containing continuous scalar values (an image
   * which is not a segmented image or a label field).
   * Continuous values mean that doing an average of 2 different values makes sense.
   *
   * @ENUM_SINCE_OIV 2024.2
   */
    CONTINUOUS = 0,
  /**
   * Specifies an image containing discrete values that represent labels.
   * These types of image are usualy called a segmented image or a label field.
   * Discrete values mean that doing an interpolation between 2 different values of the dataset does not make sense.
   *
   * @ENUM_SINCE_OIV 2024.2
   */
    DISCRETE = 1
  };

  /**
   * Specifies if this dataset corresponds or does not correspond
   * to a segmented image. A segmented image contains only discrete values.
   *
   * This field is only useful if the SoVolumeShape::interpolation type used by the rendering
   * is either LINEAR or CUBIC. For other types of interpolation, this field is ignored
   * and the data set is always considered as continuous. This field is also ignored
   * for RGBA datasets (see SoVolumeData::dataRGBA).
   *
   * @useenum{ValueInterpretation}. Default is CONTINUOUS.
   *
   * @FIELD_SINCE_OIV 2024.2
   */
  SoSFEnum valueInterpretation;

  /**
   * Returns the data set dimension.
   */
  const SbVec3i32&    getDimension();

  /**
   * Returns the tile dimension.
   */
  SbVec3i32    getTileDimension();

  /**
   * Returns the number of bytes per voxel.
   */
  unsigned int getDatumSize();

  /**
   * Copies the specified tile into the provided buffer. @BR
   * Applies the data transformation (set in dataTransform field)
   * if @B transform@b is true.
   * Returns the size, in bytes, copied in the buffer. If size == 0
   * nothing was written (probably due to a memory allocation error).
   */
  virtual size_t readTile( SoBufferObject*& bufferObject, SoLDMTileID tileID, bool transform );

  /**
   * Copies the specified tile into the provided buffer. @BR
   * Applies the data transformation (set in dataTransform field)
   * if @B transform@b is true.
   */
  virtual void readTile(SoLDMTileID tileID, SoBufferObject* buffer, bool transform);

  /** 
   * Write the specified tile using the specified writer. No operations are done on 
   * buffer (dataTransform or edition), the buffer is written as is. The writer must 
   * be initialized and handles must be properly set 
   * (see SoVolumeReader::closeAllHandles and SoVolumeWriter::restoreAllHandles), 
   * ie. the current reader's handles must be closed, and the specified writer's handles 
   * must be opened.
   */
  virtual void writeTile( SoLDMTileID tileID, SoBufferObject* buffer, SoVolumeWriter* writer );

  /**
   * Indicates whether a tile is in main memory.
   * Called when the node front manager evaluates the texture front.
   */
  bool isInMemory( SoLDMTileID tileID ) const;

  /**
   * Indicates whether the data attached to a tile is in main memory.
   * Called when the node front manager evaluates the texture front.
   *
   * This allows to manage data that implements SoBufferAsyncInterface.
   * If the specified tile does not implement SoBufferAsyncInterface then
   * this method always returns true.  So effectively it always returns
   * true for non-LDM volumes.
   */
  bool isDataInMemory( const SoLDMTileID& tileID ) const;

  /**
   * Debug purpose only. Used for visual feedback to highlight tiles in main memory.
   * Only called if the SoVolumeRendering::DRAW_TOPOLOGY flag is true.
   */
  void getTileIDInMemory( std::vector<LDM_TILE_ID_TYPE>& tilesInMemory ) const;

  /**
   * Contains an SoLDMResourceParameters object which allows you to set LDM
   * resource parameters.
   */
  SoSFLDMResourceParameters ldmResourceParameters;

  /**
   * Returns a reference to the SoLDMDataAccess object. @BR
   * This object can be used to query data from the volume.
   */
  SoLDMDataAccess       & getLdmDataAccess();

  /** 
  * Initiate an editing transaction. @BR
  * Returns true if successful and also returns a unique transaction id.
  * This value is required for finishing the edit transaction (see #finishEditing())
  * and for undo/redo (see #undoEditing() and #redoEditing()).
  * Multiple transactions may be active at the same time.
  */
  virtual SbBool startEditing( int& transactionId );

  /** 
  * Terminates an editing transaction. @BR
  * Returns true if successful. May only be called with a valid transaction id
  * returned by the #startEditing() method. On error does nothing.
  * The finish method will schedule a redraw so the correct data is displayed.
  * To commit the edits, in other words to save the edited data back to the data
  * source, call #saveEditing().
  */
  virtual SbBool finishEditing( int transactionId );

  /** 
  * Replaces the contents of a tile with the given data. @BR
  * The buffer size (in bytes) must match the tile size (in bytes) exactly.
  * Call #startEditing() before calling this method. Returns 0 if successful.
  * @warning The modification may be postponed until the tile is really needed. Therefore the
  * contents of the userData buffer must not be modified until after saveEditing() is called. 
  * \if_cpp However the buffer may be unref'd. \endif
  */
  virtual int editTile( const SoLDMTileID& tileId, SoBufferObject* userData );

  /** 
  * Replaces the contents of a subvolume with the given data. @BR
  * The buffer size (in bytes) must match the subvolume size (in bytes) exactly.
  * The subvolume is specified in voxel/cell coordinates.
  * Call #startEditing() before calling this method. Returns 0 if successful.
  * @warning The modification may be postponed until the tiles are really needed. Therefore the
  * contents of the userData buffer must not be modified until after saveEditing() is called. 
  * \if_cpp However the buffer may be unref'd. \endif
  */
  virtual int editSubVolume( const SbBox3i32& subVolume, SoBufferObject* userData );

  /** 
  * Replaces the contents of a tile with the specified value. @BR
  * Call #startEditing() before calling this method. Returns 0 if successful.
  */
  virtual int editTile( const SoLDMTileID& tileId, const double& value );

  /** 
  * Replaces the contents of a subvolume with the specified value. @BR
  * The subvolume is specified in voxel/cell coordinates.
  * Call #startEditing() before calling this method. Returns 0 if successful.
  */
  virtual int editSubVolume( const SbBox3i32& subVolume, const double& value );

  /** 
  * Replaces all voxels intersecting the polygons or lines defined by the surfaceShape
  * and given thickness with the specified value.
  *
  * @I surfaceShape @i is defined in the same 3D space as the dataSet.
  * @I thickness @i is defined in voxels. The surfaceShape can be deleted after calling 
  * this function.
  *
  * Call #startEditing() before calling this method. Returns 0 if successful.
  * @warning The modification may be postponed until the tiles are really needed. Therefore the
  * surfaceShape node (and children if any) must not be modified until after saveEditing() is called. 
  */
  virtual int editSurfaceShape( const SoNode* surfaceShape, const float& thickness, const double& value );

  /** 
  * Replaces all voxels intersecting the given shape with the specified value.
  * The geometry defined under @I solidShape @i must represent a list of closed 
  * surfaces otherwise the result is unpredictable. solidShape must not
  * contain any lines or open polyhedrons (polyhedron with shapes). 
  * The result is based on the odd-winding rule, so the result is not necessarily
  * the union or the intersection of the closed surfaces.
  * If you want to voxelize lines or single polygons, see also the #editSurfaceShape method.
  *
  * If the goal is to define a shape which is the result of the intersection/union of
  * multiple closed surfaces, see the SoCSGShape node.
  *
  * solidShape is defined in the same 3D space as the dataSet. The solidShape can
  * be deleted after calling this function.
  *
  * Call #startEditing() before calling this method. Returns 0 if successful.
  * @warning The modification may be postponed until the tiles are really needed. Therefore the
  * surfaceShape node (and children if any) must not be modified until after saveEditing() is called.  
  */
  virtual int editSolidShape( const SoNode* solidShape, const double& value );

  /**
  * Replace all voxels in the region defined by a list of boxes with the specified value.
  *
  * @I boxCenter @i and @I boxSize @i are defined in ijk (voxel) space.
  *
  * Call #startEditing() before calling this method. Returns 0 if successful.
  * [OIVJAVA-WRAPPER NOT_DERIVABLE]
  */
  virtual int editBoxes(
    const std::vector<SbVec3i32>& boxCenters,
    const int& boxSize,
    const double& newValue
  );

  /** 
  * Undo all modifications associated with the specified transaction id.
  * Returns true if successful. On error does nothing.
  * May only be called after the specified transaction has been terminated
  * (see #finishEditing() method).
  * Schedules a redraw so the correct data is displayed.
  *
  * Note: Undo is no longer available after transactions are saved. After 
  * calling #saveEditing(), the transaction ids are no longer valid and the
  * modified data is the new reference.
  */
  virtual SbBool undoEditing( int transactionId );

  /** 
  * Redo all modifications associated with the specified transaction id.
  * Returns true if successful. On error does nothing.
  * May only be called after the specified transaction has been terminated
  * (see #finishEditing() method) and undone (see #undoEditing() method).
  * Schedules a redraw so the correct data is displayed.
  *
  * Note: Redo is no longer available after transactions are saved. After 
  * calling #saveEditing(), the transaction ids are no longer valid and the
  * modified data is the new reference.
  */
  virtual SbBool redoEditing( int transactionId );


  /** 
  * This is the declaration to use for #saveEditing callback functions.
  * These callbacks are called just before a tile is written.
  * At the beginning of the saving process, the SaveEditingCB is called
  * with @I tilesRemaining @i = @I totalTiles @i.
  * Once the save process is done (all tiles written and finalized), 
  * the SaveEditingCB is called with tilesRemaining = 0.
  *
  * @param dataSet DataSet used.
  * @param writer Writer used.
  * @param tilesRemaining Remaining tiles to save
  * @param totalTiles Total tiles to save.
  *
  * @return TRUE to continue save process. FALSE to abort.
  */
  typedef SbBool SaveEditingCB( SoDataSet* dataSet,
                                SoVolumeWriter* writer, 
                                int tilesRemaining, 
                                int totalTiles );
  /** 
  * Commit all transactions.
  * All modified tiles are written back to the data source using the current reader format.
  * May only be called when no threads are editing, i.e. after all threads that
  * called #startEditing have called #finishEditing.
  *
  * Returns true if successful. On error does nothing returns false. Returns false if:
  * - No edits have been done.
  * - Transaction(s) still in progress (finishEditing has not been called).
  * - Volume was not loaded using a volume reader (direct from memory doesn't work).
  * - No volume writer class exists for this volume reader.
  *
  * @B Warning:@b Effectively, saveEditing currently only works for data loaded using
  * an SoVRLDMFileReader (an LDM format tiled data set) or an SoVRMemoryReader (data set
  * completely in memory).
  *
  * You can specify a callback (see #SaveEditingCB) that will be called before 
  * saving each tile (even if the data uses a sliced format it is saved tile by tile).
  * If this callback returns FALSE, the saving process is aborted.
  * @B Warning@b: In such a case, lower resolutions are not updated and the result will 
  * be undefined.
  *
  * Conversion Parameters: @BR
  * - Only apply if the volume reader is an SoVRLDMFileReader. In this case an instance
  *   of SoVRLDMFileWriter is used.
  * - The options are the same as the LDM converter (see SoConverter), but only a
  *   @I subset@i of those options are supported.
  * - The writer automatically sets verbose = true and histogram = false. @BR
  *   This is because if histogram = true, ALL tiles must be re-written.
  * - The parameter strings must be formatted exactly the same as on an LDM converter
  *   command line (see SoConverter).
  * - The supported options are:
  *   - -histogram : sets histogram true or false
  *   - -q : sets verbose = false
  *   - -c : compression name
  *   - -l : compression level
  *
  * [OIVNET-WRAPPER CUSTOM_CODE]
  */
  virtual SbBool saveEditing( bool recomputeLowerResolution,
    const std::vector<char*> conversionParameters = std::vector<char*>(), SaveEditingCB* callback = NULL );

  /**
  * Commit all transactions.
  * All modified tiles are written back to the data source using the current reader format.
  * May only be called when no threads are editing, i.e. after all threads that
  * called #startEditing have called #finishEditing.
  *
  * Returns true if successful. On error does nothing returns false. Returns false if:
  * - No edits have been done.
  * - Transaction(s) still in progress (finishEditing has not been called).
  * - Volume was not loaded using a volume reader (direct from memory doesn't work).
  * - No volume writer class exists for this volume reader.
  *
  * @B Warning:@b Effectively, saveEditing currently only works for data loaded using
  * an SoVRLDMFileReader (an LDM format tiled data set) or an SoVRMemoryReader (data set
  * completely in memory).
  *
  * [OIVNET-WRAPPER CUSTOM_CODE]
  */
  SbBool saveEditing();

  /**
  *  Returns true if DataSet has edited tiles.
  */
  virtual bool hasEditedTile();

  /**
   * Returns a pointer to the current data set reader object. @BR The actual type will be
   * a subclass of SoVolumeReader, either one of the predefined subclasses or an
   * application-defined subclass. Returns NULL if no reader is currently being used
   * (for example if setVolumeData() was called).
   */
  SoLDMReader*   getLDMReader();

  /**
   * Sets the LDM volume reader object to use. @BR This method allows the data to
   * be read directly from the disk using the specified subclass of
   * SoVolumeReader.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetReader(reader)}]
   */
  void           setLDMReader(SoLDMReader* reader);

  /**
  * Returns the current data set reader object. @BR The actual type will be
  * a subclass of SoVolumeReader, either one of the predefined subclasses or an
  * application-defined subclass. Returns null if no reader is currently set.
  * Applications should always verify the class type before using the
  * returned object as a specific sub-class of SoVolumeReader.
  * [OIVNET-WRAPPER PROPERTY{Reader},GETTER,SINCE{9.0}]
  */
  inline SoVolumeReader* getReader()
  {
    return const_cast<const SoDataSet*>(this)->getReader();
  }

  /**
  * Returns the current data set reader object. @BR The actual type will be
  * a subclass of SoVolumeReader, either one of the predefined subclasses or an
  * application-defined subclass. Returns null if no reader is currently set.
  * Applications should always verify the class type before using the
  * returned object as a specific sub-class of SoVolumeReader.
  * [OIV-WRAPPER-NOT-WRAP]
  */
  SoVolumeReader* getReader() const;

  /**
   * Supported data type
   */
  enum DataType {
    /** unsigned byte */
    UNSIGNED_BYTE  = SbDataType::UNSIGNED_BYTE,
    /** unsigned short */
    UNSIGNED_SHORT = SbDataType::UNSIGNED_SHORT,
    /** unsigned int (32bits) */
    UNSIGNED_INT32 = SbDataType::UNSIGNED_INT32,
    /** signed byte */
    SIGNED_BYTE  = SbDataType::SIGNED_BYTE,
    /** signed short */
    SIGNED_SHORT = SbDataType::SIGNED_SHORT,
    /** signed int (32bits) */
    SIGNED_INT32 = SbDataType::SIGNED_INT32,
    /** float */
    FLOAT = SbDataType::FLOAT,
    /** double */
    DOUBLE = SbDataType::DOUBLE
  };


  /**
  * Returns the number of bytes per voxel of the specified data type.
  */
  static int dataSize( DataType dataType );

  /**
  * Returns TRUE if the given data type is a signed integer data type.
  */
  static SbBool isDataSigned( DataType dataType );

  /**
  * Returns TRUE if the given data type is a float data type.
  */
  static SbBool isDataFloat( DataType dataType );

  /**
  * Returns the number of significant bits.
  */
  int numSigBits() const;

  /**
  * Returns @B min @b and @B max @b values of the data set data.@BR
  * Returns \if_java null \else FALSE \endif if the requested data is not
  * available (for example, if no data set exists).@BR
  * @BR
  * NOTE: This method might force LDM to load the @I entire @i data set if
  * the volume reader does not respond to the getMinMax query. Normally for an
  * LDM format data set, the min and max values are stored in the LDM header.
  * For a non-LDM data set, if a filename and/or reader have been specified and
  * the data set has not yet been loaded, LDM will load the entire data set to
  * compute the min and max values. For a large data set this may take a long time.
  */
  virtual SbBool getMinMax( int64_t &min, int64_t &max);

  /**
  * Returns @B min @b and @B max @b values of the data set.@BR
  * Returns \if_java null \else FALSE \endif if the requested data is not
  * available (for example, if no data set exists).@BR
  * @BR
  * NOTE: This method might force LDM to load the @I entire @i data set if
  * the volume reader does not respond to the getMinMax query. Normally for an
  * LDM format data set, the min and max values are stored in the LDM header.
  * For a non-LDM data set, if a filename and/or reader have been specified and
  * the data set has not yet been loaded, LDM will load the entire data set to
  * compute the min and max values. For a large data set this may take a long time.
  * [OIVJAVA-WRAPPER NAME{getDoubleMinMax}]
  */
  virtual SbBool getMinMax( double &min, double &max);

  /**
   * Returns the data type.
   */
  DataType getDataType() const;

  /**
   * Returns the number of bytes per voxel in VolumeViz
   */
  int getDataSize();

  /**
   * This method allows the data to
   * be read directly from the disk using the specified subclass of
   * SoVolumeReader.
   * If the @B takeOwnership @b parameter is FALSE (default), LDM
   * will not delete the specified instance. Otherwise, LDM will automatically
   * delete the specified instance, for example, when the #SoDataSet node is deleted.
   * If filename is not set before setting the reader, LDM will display an error message
   * just to warn that the reader has not opened any file yet. LDM continues to work as expected.
   *
   * Note: When using a custom reader, any reader method that changes the volume
   * properties (dimension, size, data type, etc) should notify the SoVolumeData
   * node by calling the reader's touch() method.  If this notification is not done,
   * SoDataSet fields, for example #SoDataSet::extent, won't be updated correctly.
   *
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetReader(reader)}]
   * [OIVNET-WRAPPER-ARG IN&STRONG_REF,IN]
   * [OIVNET-WRAPPER OBSOLETE{Use Reader property instead},SINCE{9.0}]
   */
  void setReader( SoVolumeReader &reader, SbBool takeOwnership = FALSE );

   /**
   * Returns the maximum number of data sets that can be render composited on the
   * current hardware (essentially the number of texture units - 1).
   */
  static int getMaxNumDataSets() ;

  /**
   * Converts the specified point in voxel coordinates (I,J,K) to geometric coordinates (X,Y,Z).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * The left bottom coordinate of the voxel is returned.
   * Correctly converts coordinates that are outside the volume dimensions,
   * but the resulting geometric coordinate is outside the volume extent.
   *
   * @see voxelToXYZ(const SbBox3f&) const
   * @see XYZToVoxel(const SbVec3f&) const
   * @see XYZToVoxel(const SbBox3f&) const
   */
  virtual SbVec3f voxelToXYZ( const SbVec3f& dataPosition ) const;

  /**
   * Converts the specified box in voxel coordinates (I,J,K) to geometric coordinates (X,Y,Z).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume dimensions,
   * but the resulting geometric coordinate is outside the volume extent.
   *
   * @see voxelToXYZ(const SbVec3f&) const
   * @see XYZToVoxel(const SbVec3f&) const
   * @see XYZToVoxel(const SbBox3f&) const
   */
  virtual SbBox3f voxelToXYZ( const SbBox3f& box ) const;

  /**
   * Converts the specified point in geometric coordinates (X,Y,Z) to voxel coordinates (I,J,K).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbVec3f&) const
   * @see voxelToXYZ(const SbBox3f&) const
   * @see XYZToVoxel(const SbBox3f&) const
   */
  virtual SbVec3f XYZToVoxel( const SbVec3f& dataPosition ) const;

  /**
   * Converts the specified box in geometric coordinates to voxel coordinates.
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbVec3f&) const
   * @see voxelToXYZ(const SbBox3f&) const
   * @see XYZToVoxel(const SbVec3f&) const
   */
  virtual SbBox3f XYZToVoxel( const SbBox3f& xyzBox ) const;

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_METHOD(8000,"Use void readTile(tileID, SoBufferObject*, bool ) instead.")
  virtual void readTile(SoLDMTileID tileID, unsigned char* buffer, bool transform);

  /**
  * @deprecated Use #SoDataSet::dataTransform field which handle an SoLDMDataTransform fieldContainer.
  * Type declaration for the data transform function. See #SoDataSet::setLDMDataTransformFunction.
  * This function allows a tile of data to be transformed after it is loaded, but before
  * it is stored in main memory.  It is not currently possible to access other tiles of
  * data (for example using the data access API) from this function.  Note that the
  * function will be called from LDM data loader threads, so multiple threads may be
  * executing in this function at the same time (on different tiles).
  * Inputs are:
  *  - The associated data set object the function is calling for.
  *    This allows retrieving information such as data type (ds->getDataType()).
  *  - The dimensions of the tile to transform.
  *    This defines the size of the buffer and is the same for every tile in a dataset.
  *    However tiles on the "outside" of the dataset may be partial tiles and contain
  *    less than the full number of actual data values.
  *  - A buffer containing the tile data to transform.
  *    The data should be modified "in place" in this buffer.
  *  - The position and extent of the tile in data space (voxel coordinates).
  *    For lower resolution tiles (level > 0) the extent of the tile will be larger
  *    than the dimensions of the tile (number of values in the tile).
  *  - The tile resolution level.  Level 0 is full resolution data.
  *  - Optional user data specified with setLDMDataTransformFunction.
  *
  * @DEPRECATED_SINCE_OIV 8.0
  * [OIV-WRAPPER NAME{LDMDataTransformCB}]
  * [OIV-WRAPPER-ARG IN,IN,ARRAY{ds != nullptr? ds->getDataSize()*bufferDimension[0]*bufferDimension[1]*bufferDimension[2]: 0},IN,IN,IN]
  */
  typedef void SoLDMDataTransformFunction(SoDataSet* ds,
                                          const SbVec3i32& bufferDimension, //tile dim
                                          void* bufferToTransform,
                                          const SbBox3i32& dataBox,      //position of tile in data space
                                          int resolutionLevel,	  //resolution level of the tile
                                          void* userData);

  /**
   * If set, the user-defined function is called after each tile is loaded,
   * but before it is stored in main memory.
   * This allows you to modify the original data (for example, scaling or
   * filtering) before it is displayed.
   *
   * [OIV-WRAPPER EVENT_NAME{LDMDataTransformCallback}]
   */
  SoDEPRECATED_METHOD(8000,"Use the #SoDataSet::dataTransform field instead.")
  void setLDMDataTransformFunction(SoLDMDataTransformFunction* func, void* userData = NULL);

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Returns the tile border (overlap) in voxels.
   * Since OIV 9.0 always returns 0.
   */
  SoDEPRECATED_METHOD(9000,"No longer used.")
  int getOverlapping();

#endif /** @DEPRECATED_END */

SoEXTENDER_Documented public:

  /**
   * Resets previously set custom reader setup by setReader call.
   * \if_cpp
   *   Reference counting:
   *   \par
   *   Warning: This method calls ref() on the reader and then calls unref().
   *   @I If the reader has a reference count of zero (the default), the call
   *   to resetReader() will cause it to be destroyed! @i
   *   A warning message is posted in this case when using a debug build of Open
   *   Inventor.
   * \endif
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onResetReader()}]
   */
  void resetReader();

  /**
   * Returns the LDMTopoOctree used by this SoDataSet.
   * The LDMTopoOctree is initialized at the initialization of the reader.
   */
  const SoLDMTopoOctree* getLDMTopoOctree() const;

SoEXTENDER public:

  /** @copydoc SoNode::callback */
  virtual void callback( SoCallbackAction *action );
  /** @copydoc SoNode::GLRender */
  virtual void GLRender( SoGLRenderAction *action );
  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  /** @copydoc SoNode::handleEvent */
  virtual void handleEvent(SoHandleEventAction* action);
  /** @copydoc SoNode::pick */
  virtual void pick(SoPickAction *action) ;
  /** @copydoc SoNode::write */
  virtual void write(SoWriteAction *action);
  /** @copydoc SoNode::doAction */
  virtual void doAction( SoAction *action );

  /**
  * Creates fake data in buffer
  */
  inline virtual void useFakeData(SoLDMTileID /* tileID */, SoBufferObject* /* buffer */) {};

SoINTERNAL public:

  // This map is used to maintain signature of each tile in cache
  // It's used to be able to find an uncompressed version of a given tile in the cache
  // when already uncompressed tile are reloaded.
  typedef std::map<SoLDMTileID, uint64_t> tileSigInCacheMap;

  static void initClass();
  static void exitClass();

  /** Return true. if the current Dataset is of type  */
  bool isReallyOfType(const SoType& type, bool derived);

  /**
   * @brief Indicates whether the GPU DataRange shader can be used.
   *
   * Basically, GPU DataRange can be used if texturePrecision is > data num significant bit.
   */
  bool isGPUDataRange( SoState* state );

  /** 
   * Apply edition to specified tile. Return tileBuffer if tile is not edited,
   * or a new buffer corresponding to result of edition if tile is edited.
   * Returned buffer must be ref/unref.
   */
  SoBufferObject* getEditedTile( SoBufferObject* tileBuffer, const SoLDMTileID& tileId );

  /**
   * Internal helper function which return LDM tile transformed depending on state
   * see SoVolumetransform for details.
   */
  virtual SoBufferObject *getTransformedTile(const SoLDMTileID& tile,
                                             const SoLDM::DataSetIdPair& p,
                                             SoState * state=NULL,
                                             const bool useExtendedData = false);

  // return the number of SoDataSet instance created
  // this is usefull to optimize SoLdmValuationAction
  static int getNumInstance();

  /** Initialize Thread local storage structure. */
  SB_THREAD_TLS_HEADER();

  SoMultiDataSeparator *getMultiDataSeparator() { return m_MultiDataSeparator; }
  void setMultiDataSeparator(SoMultiDataSeparator *mds, SoLDMMediator* mediator = nullptr);

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  /**
   * Copy all parameters of SoLDMResourceParameters to the SoLDMResourceManager
   */
  void sendParametersToLDM(SoLdmValuationAction* action);

  //When true data set won't be used for rendering
  void ignoredByMultiDataSeparator(bool state);
  bool isIgnoredByMultiDataSeparator();

  // have the properties read ?
  virtual bool getDataCharRead() { return m_dataCharRead;};

  /* @return texture precision in Bits */
  int getIndTexPrec() const;
  bool loadInMemory(SoState* s = NULL);
  void setListChange(bool val);
  bool getListChange();

    /* 
   * Return internal state of resources. Dirty state means
   * resource field has been changed by application / user.
   * Can be cumulative value defined in 
   * See enum definition made in SoLDMResourceManager::State
   */
   unsigned short getResourcesState() const;

  /* 
   * Set internal state of resources. State set to true means
   * resource field have been changed by application / user.
   */
   void setResourcesState( const SoLDMResourceManager::State flag, bool value );

  //shall we accumulate the node in state?
  bool     isAccumulated();

  static SbBool getCurrentMatrixDifference(SoState* state, const SoLDM::DataSetIdPair& idPair, SbMatrix& geomMatrix);

  //multi-data
  void lookForSceneGraphChange(SoAction* action);

  // extract slice data
  virtual SoBufferObject* getDataOneSlice( const SbVec3i32& tileSize, SoBufferObject* tileData, int Zaxis, int slice);
  virtual void releaseDataOneSlice(SoBufferObject *sliceDataBuffer);

  //***** factory methods ****

  virtual LDMSliceAccessor* getAppropriateSliceAccessor(const SoLDM::DataSetIdPair& p);
  virtual SoLDMReader* getAppropriateLDMReader( const SbString& )
  { return NULL; };

  // logical center per id
  bool getCenter(SoLDMTileID tileID, SbVec3f& center)const;

  SoLDMMediator* getMediator() const
  {
    return m_mediator;
  };
  void setMediator(SoLDMMediator* m);

  virtual bool createTex(int Zaxis, SoState *state, const SbVec3i32 &tileSize, SoBufferObject *tileDataBuffer, SoTransferFunction *transferFunction,
                         SoBufferObject*& textureBufferObject, int alphaUse = 0);

  // TODO FIXME : should cleanup this call.
  /** @param numElem is the number of datum to convert */
   void* getTexData(const void* tileBuffer,
                    int numElem,
                    SoTransferFunction* transferFunction,
                    int alphaUse,
                    int texPrec,
                    void *textureBuffer,
                    SoDataRange* datarange,
                    SoDataCompositor* dataCompositor);

  // allocate ( if needed ) and initialize the texture buffer, according to the input buffer and the transfer function.
  // The texture buffer can be given with the tex_data argument. In this case no allocation is done and tex_data is returned.
  // Otherwise returns the initialized texture buffer.
  // Returned pointer does not need to be deleted, but reference to this buffer will be valid only until the next getTexData call.
  SoBufferObject* getTexData(SoBufferObject* tileBufferObject,
                             SoTransferFunction* transferFunction,
                             int alphaUse,
                             int texPrec,
                             SoBufferObject* textureBufferObject,
                             SoDataRange* dataRange,
                             SoDataCompositor* dataCompositor);

  SoTexture::InternalFormat getRGBAInternalFormat() const;

  SoLDMResourceManager& getResourceManager() { return *m_resourceManager; }

  /**
  * Return the precision of the texture in bits
  * in RGBA texture mode, returns a negative value
  */
  virtual int getTexPrec(SoState*);

  /** Find texture format according to colorMapType.
   * @param colorMapType Type of colormap. See SotransferFunction::GLColorMapType.
   * @param format. Out. DataFormat given to glTexSubImage.
   * @param dataType. Out. DataType given to glTexSubimage.
   * @param internalFormat. Out. Internal texture format. */
  void getTextureFormat(SoTransferFunction::GLColorMapType colorMapType,
                        SbEnums::ImageDataFormat& format,
                        SbDataType::DataType& dataType,
                        SoTexture::InternalFormat& internalFormat) const;

  //Display loaded tiles
  virtual void drawTilesInMainMemory(SoGLRenderAction* action, const SoLDM::DataSetIdPair& idPair);

  //Copy dataset's fields if needed into the given mediator
  // Notifications are disabled on #ldmResourceParameters during the process
  void copyLDMParameters(SoLDMMediator *mediator);

  /**
   * Return true if multidata is enabled
   * WARNING: This may also return true when there is an SoMultiDataSeparator on state but only
   * 1 VolumeData.
   */
  virtual bool isMultidataEnabled() const { return m_multidataEnabled; }

  // draw a colored Box
  enum BoxColor {
    GREY,
    RED,
    BLUE,
    GREEN,
    FULL_RES_GREY,
    FULL_RES_GREEN
  };

  /**
   * Return the current id. (Useful to know the previous value
   * when the field dataSetId is changed)
   */
  unsigned short getId() const 
    { return m_id; }

  /**
   * Return true if there is a read error
   */
  bool hasReadError();

  /**
   * Return true if at least one of the dataset dimensions is 0
   */
  inline bool isEmpty()
  {
    const SbVec3i32& dim = getDimension();
    return ( dim[0] == 0 || dim[1] == 0 || dim[2] == 0 );
  };

  /**
  * A datum is a series of m*n data type. e.g. 3 int followed by 4 float followed by 6 short
  * n is num in the DatumElement struct, data type is type.
  */
  struct DatumElement{
    SoDataSet::DataType type;
    int                 num;
  };

  /**
   * Convert the given data to the normalized float value a shader
   * will see on the GPU, taking into account datatype, data range,
   * texture precision, etc.
   * I.e. from [data(Type|Range)Min,data(Type|Range)Max] -> [0, 1]
   */
  float convertToNormalizedTextureSpace(SoState *state, const float data);

  /**
  * Specify the first traversal of the node has been done once.
  */
  void setFirstTraversalDone(bool flag);

  /**
  * Return true if this node has been traversed once
  */
  bool isFirstTraversalDone() const;

  virtual bool isPalettedTexture(SoState*);

  // Get a unique signature for a given tile. If tile already loaded, return the
  // previously generated signature. Otherwise, generate a new one. Each dataset
  // manages its signature list.
  virtual uint64_t getTileSignature(const SoLDMTileID& tileId);

  /**
   * Convenient method needed for .NET wrapping
   * calling setReader( reader, TRUE );
   * [OIVNET-WRAPPER VISIBILITY{Public},PROPERTY{Reader},SETTER]
   */
  inline void setOwnedReader( SoVolumeReader &reader )
    { setReader( reader, TRUE ); };

  /*
      Ref/unref mechanism on volume render used to manage
      GPU resources. Allow SoDataSet to free GPU resources when
      texture are not needed
      */
  void refVolRender( int contextId );
  void unrefVolRender( int contextId );

  /**
   * Returns the best uniform resolution which can be used regarding CPU/GPU memory.
   * Best resolution is computed during pre-rendering operations.
   * Returns -1 if has been called before computation.
   */
  int getBestEqualResolution() const;

  /** Delete the histogram */
  void invalidateHistogram();

  /** Returns the dataset's datatype MinMax values */
  SbVec2f getDataTypeMinMax() const;

  /**
   * Compute tile's bbox in data space
   * @param tile the tile
   * @param tileBox is the computed bbox
   * @param tileDim is the dimension of the tile
   * @return the texture tile associated to tile
   */
  SoLDMTileID getTileCharacteristics(const SoLDMTileID& tile, SbBox3i32& tileBox, SbVec3i32& tileDim) const;

  /**
   * Returns true if we can use undefined voxels i.e no data conversion is performed before sending it to the GPU
   */
  bool supportUndefinedVoxel(SoState* state);

  /** Returns the real reader used internally. */
  SoVolumeReader* getInternalReader() const;

  int getTexelSizeBits(SoState* state = nullptr);

  /** Returns the object-to-voxel matrix */
  SbMatrix getVoxelMatrix() const;

  /** Returns the voxel-to-object matrix */
  SbMatrix getVoxelMatrixInv() const;

protected:

  /** Default Constructor */
  SoDataSet();

  /** destructor */
  virtual ~SoDataSet();

  inline virtual void computeLighting( const SbVec3i32& /* tileSize */, SoBufferObject* /* rgbaBuffer */ ) {};

  /**
   * Puts the data set element on the state
   */
  virtual void setElement(SoState* state);

  /**
   * Returns true if the extent specified by the reader must be use
   */
  bool isUsingUserExtent();

  /*
  Allows a class inheriting from SoDataSet to decide whether multiple data should be allowed or not.
  If it is the SoDataSet will assign one mediator for all dataset.
  Else one mediator is assigned for all dataset (one LDM core per dataset).
  In vviz this method checks whether a compositor or a shader is present in the state.
  If not then multidata is not allowed.
  */
  virtual bool enableMultidata(SoState* state);

  //Return a reader according to m_filename and m_data
  virtual SoVolumeReader* getAppropriateReader();

  /** If true, tile dim is rounded to a pow 2 dim*/
  bool m_forcePow2TileDim;

  //LDM reader
  SoRef<SoFieldContainer> m_reader;

  bool load(SoState*);
  virtual void unload();
  virtual void releaseMemory();


  bool m_multidataEnabled;

  SbVec3i32     m_dimension;
  bool          m_initialized;

  SbDataType       m_type;
  int              m_numBits;

  //True if data header has already been read
  bool m_dataCharRead;

  SbString      m_filename;
  virtual void updateFilename();


  unsigned short m_id;


  /**
   *Releases memory and set the new reader
   */
  virtual void readerChanged();

  void initReader();

  virtual SoLDMMediator *createMediator();
  SoLDMMediator* m_mediator;

  SoMultiDataSeparator *m_MultiDataSeparator;

  SoLDMResourceManager* m_resourceManager;

  void getTransferFunction(SoTransferFunction *transferFunction, int alphaUse,
                           int &shift, int &offset, int &nrgba, unsigned int * &rgba);

  //Build a tile out of the original full resolution data.
  //The data is taken out from disk by the reader (vol, am,..)
  //and subsampled until the desired level.
  virtual void buildTile(SoLDMTileID tileID, SoBufferObject* buffer);

  SoVolumeHistogram *m_histogram;

  /** Implements the SoLdmValuationAction */
  virtual void ldmAction(SoLdmValuationAction* action);

  /**
   * Returns TRUE if the data represents RGBA color values.
   */
  virtual SbBool isRGBAData();

SoINTERNAL protected:

  /**
   * SoDataSet must return the SoLDMGeometryElement defining the weights of the representations
   * that are going to use it.
   */
  virtual SoLDMGeomElement* getAppropriateElement( SoLDMGeometry* SO_UNUSED_PARAM(v) )
  {
    return NULL;
  }

  /**
   * Returns texture max num bits for this type of dataset.
   */
  virtual unsigned short getMaxTexturePrecisionBits() const { return 16; }

  SoSFFieldContainer customReader;
  SoSFBool           customReaderOwnerShip;

  /**
  * editingStack used by editing. Must be SoLDMEditingStack. is NULL until the 
  * first call of startEditing.
  */
  SoSFLDMDataTransform editingStack;

private:

  /** Common code to constructors */
  void commonConstructor();

  /**
   * thread specific variables
   */
  struct MTstruct {
    SoBufferObject *tmpTexture;
    SoCpuBufferUniform *tmpTextureUniform;
    SoBufferObject *tmpColormap;
    SoBufferObject *tmpTexturePermute;
  };

#ifndef HIDDEN_FROM_DOC
   friend class SoVolumeState;
   friend class SoLDMMediator;
   friend class SoCpuBufferFromVolumeReaderImpl;
   friend class SoObliqueSliceLdm;
#endif
   friend class SoLdmValuationAction;
   friend class SoLDMGeometry;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif  // _SO_DATA_SET_


