/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/


#ifndef LDM_RESOURCE_PARAMETER_H
#define LDM_RESOURCE_PARAMETER_H

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/fields/SoFieldContainer.h>
#include <Inventor/fields/SoSubFieldContainer.h>

#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFVec3i32.h>

#include <Inventor/SbBox.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/cassert>

class SoDataSet;
class SoLDMMediator;
class SoLDMTileVisitor;
class SoLDMResourceManager;

/**
* @LDMEXT Defines resource parameters constraints of an SoDataSet object.
*
* @ingroup LDMNodes
*
* @DESCRIPTION
*   This class specifies the resource parameters of an instance of the class SoDataSet.
*   See the field SoDataset#ldmResourceParameters.
*
*   You can set global resource parameters using static methods in class SoLDMGlobalResourceParameters.
*   See the SoLDMGlobalResourceParameters class description for an overview of LDM memory management.
* @SEE_ALSO
*    SoDataSet,
*    SoVolumeData,
*    SoLDMGlobalResourceParameters
*
*/
class LDM_API SoLDMResourceParameters : public SoFieldContainer
{
  SO_FIELDCONTAINER_HEADER(SoLDMResourceParameters);

public:
  /**
   * This structure returns information to the application when #fixedResolution is TRUE and #resolution is not -1.
   */
  struct FixedResolutionReport{
    /**
     * Available report code.
     */
    enum ReportCode
    {
      /**
       * Loading in progress.
       * In this case, the progress is indicated by the two variables #numTilesToLoad
       * and #numTilesLoaded.
       * The current resources of the machine allow LDM to load
       * the data at the specified resolution. In this case, the only possible way
       * to receive an ABORT report code from the system is either if the user changes
       * the resources (i.e., decreases the main memory space), or add some new geometry
       * to the scene.
       */
      PROGRESS,
      /**
       * If LDM cannot grant a fixed resolution request, the \if_dotnet delegate \else callback \endif
       * function is called with an ABORT report code. In this case, LDM will automatically
       * disable the fixed resolution mode.
       * The FixedResolutionCB \if_dotnet delegate \else callback \endif will be called with an ABORT report code in
       * the following cases:
       * - The request to load the data at the specified resolution level (resolution
       * argument in the #enableFixedResolutionMode function) in main memory
       * for the current geometry (either SoVolumeRender, SoVolumeSkin, SoOrthoSlice, SoObliqueSlice,
       * volume geometries, or other LDM based render nodes) is not possible because the main memory space (set by the
       * setMaxMainMemory function) is not large enough. In this case, the \if_dotnet delegate \else callback \endif
       * is called with an ABORT code immediately after the first sceneGraph traversal,
       * once the fixed resolution mode has been enabled.
       * - There was a geometry change (i.e., a geometry node was added to the scene graph) and
       * there is not enough room in memory for the geometry nodes to be displayed at the current
       * resolution level.
       * In this case, the memory space needed in addition for LDM to be able to grant
       * the request, is specified by the #numTilesToAdd variable. If there is enough main memory
       * resources on the user machine, an automatic memory adjustment can then be realized
       * by calling setMaxTilesInMainMem( getMaxTilesInMainMem() + numTilesToAdd ) right before
       * #enableFixedResolutionMode is called again. Those two calls can be done within the \if_dotnet delegate \else callback \endif
       * function so LDM does not disable the fixed resolution mode after an ABORT report code
       * has been sent.
       */
      ABORT
    };
    /**
     * Report code.
     */
    ReportCode what;
    /**
     * How many more tiles need to be allowed in main memory. The value of this variable
     * is relevant only if the \if_dotnet delegate \else callback \endif is called with an ABORT report code.
     * An automatic memory adjustment is possible by calling
     * setMaxTilesInMainMem( getMaxTilesInMainMem() + numTilesToAdd ) and then reenabling the
     * fixed resolution mode within the \if_dotnet delegate \else callback \endif function.
     */
    int numTilesToAdd;
    /**
     * How many tiles need to be loaded. The value of this variable
     * is relevant only if the \if_dotnet delegate \else callback \endif is called with a PROGRESS report code.
     */
    int numTilesToLoad;
    /**
     * Number of tiles LDM has already loaded. The value of this variable
     * is relevant only if the \if_dotnet delegate \else callback \endif is called with a PROGRESS report code.
     */
    int numTilesLoaded;
  };

  /**
   * When #fixedResolution is TRUE and #resolution is not -1, this \if_dotnet delegate \else callback \endif is
   * used by the application to get a report back from LDM as far as progress achieved and requests made.
   */
  typedef void FixedResolutionCB(FixedResolutionReport&, void*);

  /**
   * Tile Load Policy.
   */
  enum LoadPolicy {
    /** Load tiles only when when there is no user interaction, i.e., application idles.
     * (Default)
     */
    NO_USER_INTERACTION,
    /** Always load tiles. */
    ALWAYS,
    /** No tile loading. Allows suspension of the loading process. */
    NEVER
  };

  /**
   * Tile cache management policy.
   * This enum is used by the field #tileCachePolicy.
   *
   * It specifies when and how the tiles are stored in CPU memory.
   *
   * For non-compressed tiled data, the cache policy has no effect.
   * The tiles are stored as-is in the first level CPU cache during data loading.
   *
   * See SoLDMGlobalResourceParameters for detail on the first and second level CPU tile cache.
   */
  enum TileCachePolicy {
    /**
    * For non-tiled data such as DICOM or SEGY or an in-memory volume, tiles are created
    * when needed (to send data to the GPU) and stored in the second level CPU tile cache.
    *
    * For tiled data that is compressed (e.g. an LDM file created with compression),
    * the compressed tiles are stored in the first level CPU tile cache "as is" to reduce the memory
    * footprint (the first level tile cache contains only compressed data).
    * When a tile must be sent to the GPU, it is uncompressed and the resulting
    * uncompressed tile is stored in the second level CPU tile cache.
    *
    * This mode saves memory but can reduce interactivity.
    */
    NONE,

    /**
    * For non-tiled data such as DICOM or SEGY or in-memory volume, tiles are created
    * when needed (to send data to the GPU): full resolution tiles are stored in the
    * second level CPU tile cache and all the other created tiles (lower resolution)
    * are stored in the first level CPU tile cache.
    *
    * For tiled data that is compressed (e.g. an LDM file created with compression),
    * the full resolution tiles are stored *compressed* in the first level CPU tile
    * cache (to reduce the CPU memory footprint) and the other created tiles are
    * stored *uncompressed* in the first level CPU tile cache. When compressed tiles
    * must be sent to the GPU, they are uncompressed and stored in the second level
    * CPU tile cache.
    *
    * This setting uses more memory but saves time when low resolution tiles need
    * to be sent to the GPU.
    */
    LOWRES_ONLY,

    /**
    * For non-tiled data such as DICOM or SEGY or in-memory volume, tiles are created
    * when needed (to send data to the GPU) and both low resolution and
    * full resolution tiles are stored in the first level CPU tile cache.
    * This mode is recommended for non-tiled volumes loaded from disk when the volume
    * will fit in memory (generally true for medical volumes).  
    * NOTE: For an in-memory volume, this setting will duplicate the data in memory.
    * For an in-memory volume it is better to use the LOWRES_ONLY or NONE setting.
    *
    * For tiled data that is compressed (e.g. an LDM file created with compression),
    * both full resolution and low resolution tiles are uncompressed when loaded
    * and stored *uncompressed* in the first level CPU tile cache.
    *
    * This setting uses the most memory, but saves time when tiles need to be sent to
    * the GPU. It does not use the second level CPU tile cache.
    */
    ALL
  };


  /**
   * Constructor.
   */
  SoLDMResourceParameters();

  /**
   * Destructor.
   */
  virtual ~SoLDMResourceParameters();

  /**
   * Maximum allowed main memory in MB that LDM is allowed to use for this data set.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation take in account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in memory allocation, and allocated CPU memory for this
   * node will be the exact value set in the field. In this case, the amount of memory is removed from Global value.
   * All other dataset will share
   * (SoLDMGlobalResourceParameters::getMaxMainMemory  - ( Sum of memory of all datasets with #maxMainMemory != -1 )) / ( number of datasets with #maxMainMemory == -1 )
   *
   */
  SoSFInt32 maxMainMemory;
 
  /**
   * Maximum allowed texture (GPU) memory in MB for this data set.
   * This field is only considered for SoVolumeRender nodes.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation take in account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in memory allocation, and allocated CPU memory for this
   * node will be the exact value set in the field. In this case, the amount of memory is removed from Global value.
   * All other dataset will share
   * (SoLDMGlobalResourceParameters::getMaxTexMemory  - ( Sum of memory of all datasets with #maxTexMemory != -1 )) / ( number of datasets with #maxTexMemory == -1 )
   */
  SoSFInt32 maxTexMemory;

  /**
   * Tile loading policy.
   * @useenum{LoadPolicy}. Default is NO_USER_INTERACTION.
   * NO_USER_INTERACTION means the asynchronous loading
   * threads will only load tiles when the user is not interacting with the scene (i.e. the
   * application is idle).
   *
   * The loading occurs as long as there is something to load if ALWAYS is passed.
   * No loading occurs if NEVER is passed, however at least the resolution 0 is loaded.
   */
  SoSFEnum loadPolicy;

  /**
  * Specify how LDM store the tiles in memory.
  * Please refer to #TileCachePolicy for the different options.
  * Default value is ALL.
  *
  *  @FIELD_SINCE_OIV 10.0
  */
  SoSFEnum tileCachePolicy;

  /**
   * Maximum number of 3D tiles to be loaded into texture memory per frame for this data set.
   * You can set this number for all data sets by calling SoLDMGlobalResourceParameters::setTex3LoadRate().
   * This field is only considered for SoVolumeRender nodes.
   *
   * Each tile needed for volume rendering (SoVolumeRender) and volume geometry must be
   * transfered to GPU memory as a 3D texture.
   * So, for example, transfering 1000 tiles at the loadRate 10 requires 100 frames.
   * (How many seconds this takes depends on the frames per second possible, which in turn
   * depends on many factors including the complexity of the scene graph.)  Increasing the
   * loadRate value reduces the number of frames required to reach maximum resolution.  But
   * the render traversal has to wait for the data transfer to finish, so each frame may take
   * longer to render, increasing the total time to reach maximum resolution.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation takes into account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in automatic allocation, and allocated 3D texture LoadRate for this
   * node will be the exact value set in the field. In this case, the amount of memory is removed from Global value.
   * All other dataset will share
   * (SoLDMGlobalResourceParameters::getTex3LoadRate  - ( Sum of memory of all datasets with #tex3LoadRate != -1 )) / ( number of datasets with #tex3LoadRate == -1 )
   */
  SoSFInt32 tex3LoadRate;

  /**
   * Maximum of 2D textures memory in MB for this data set.
   * This field is only considered for SoSlice nodes.
   *
   * Each tile needed to render a slice (SoOrthoSlice, SoVolumeSkin, etc) must be stored
   * in GPU memory as a 2D texture.  This method limits the amount of GPU memory that can
   * be used by (for example) SoOrthoSlice and SoVolumeSkin nodes. This can be useful when
   * combining slices and volume rendering because the 3D textures used by volume rendering
   * require much more GPU memory.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation takes into account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in memory allocation, and allocated CPU memory for this
   * node will be the exact value set in the field. In this case, the amount of memory is removed from Global value.
   * All other dataset will share
   * (SoLDMGlobalResourceParameters::setMax2DTexMemory - ( Sum of memory of all datasets with #max2DTexMemory != -1 )) / ( number of datasets with #max2DTexMemory == -1 )
   *
   * @FIELD_SINCE_OIV 9.6.0
   */
  SoSFInt32 max2DTexMemory;

  /**
   * Maximum number of 2D tiles to be loaded in texture memory per frame.
   * This affects rendering nodes that use 2D textures, for example SoOrthoSlice and SoVolumeSkin.
   * Default is SoLDMGlobalResourceParameters::getTex2LoadRate().
   * This field is only considered for SoSlice nodes.
   *
   * Each tile needed to render a slice (SoOrthoSlice, SoVolumeSkin, etc) must be loaded
   * into GPU memory as a 2D texture.  Increasing the loadRate value reduces the number of
   * frames required to reach maximum resolution.  But the render traversal has to wait for 
   * the data transfer to finish, so each frame may take longer to render, increasing 
   * the total time to reach maximum resolution.  
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation takes into account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in automatic allocation, and allocated 2D texture  for this 
   * node will be the exact value set in the field. In this case, the amount of LoadRate is removed from Global value.
   * All other dataset will share 
   * (SoLDMGlobalResourceParameters::getTex2LoadRate  - ( Sum of memory of all datasets with #tex2LoadRate != -1 )) / ( number of datasets with #tex2LoadRate == -1 )
   */
  SoSFInt32 tex2LoadRate;

  /**
   * Minimum loading threshold for LDM module.
   * Specifies to LDM to never unload tiles below this resolution threshold.
   * Threshold is the power of 2 of the desired subsample level.
   * For instance, passing 2 means never unload tiles of resolution 1/4.
   * Default is -1, meaning the minimum subsample level.
   *
   * If used with an ROI enabled, only tiles below the min resolution threshold
   * and within the ROI subvolume will never be unloaded. All tiles outside the ROI
   * subvolume will be unloaded. The ROI box allows you to specify where it is the
   * most important to increase the resolution within the subvolume. By default,
   * (subvolume ROI flag enabled) this would be within the box. If for example using
   * the exclusion box mode, then lower resolution tiles will be within the box and higher
   * resolution ones outside the box and within the subvolume.
   */
  SoSFInt32 minResolutionThreshold;

  /**
   * Maximum loading threshold for the LDM module.
   * LDM will never load tiles below this resolution threshold.
   * Threshold is the power of 2 of the desired subsample level.
   * For instance, passing 1 means never load tiles of full resolution.
   * Default is 0, full resolution, i.e., load everything.
   */
  SoSFInt32 maxResolutionThreshold;

  /**
   * Sets the tile size.
   *
   * VolumeViz always manages volume data in CPU memory as a hierarchy of tiles
   * and data is transferred to GPU memory as tiles.
   *
   * When using VolumeViz with in-memory data or a file format other than LDM,
   * the data is converted to tiles "on the fly".  In this case, the default tile
   * size is (128, 128, 128).
   *
   * When loading data using an LDM volume reader, e.g. loading an LDM format file
   * or using a custom volume reader that implements the readTile() method, the
   * tile size is initialized with the value returned by the reader's getTileSize()
   * method.  The application can specify a different tile size, in which case the
   * tiles are automatically generated using the data from the reader's tiles,
   * with the following limitation:  If the reader's readTile() method returns uniform
   * or compressed buffers (SoCpuBufferUniform or SoCpuBufferCompressed),
   * they are converted to regular buffers and the memory optimization is lost.
   *
   * Changing the default tile size may affects the rendering or data loading performances.
   * Basically, a larger tile size is usually better for rendering performance or to quickly
   * obtain the highest resolution thanks to more optimal GPU utilization.
   * On the other hand, it may penalize the data fetching on the CPU side from slow hard disk
   * drives or over the network. Experimentation is generally required to find a good trade-off
   * between rendering efficiency and data fetching time.
   *
   * Notes:
   *   - tileDimension must be set @I after@i setting the volume filename or volume reader on
   *     an SoVolumeData node.
   *
   *   - Tiles are stored as textures on the GPU. Therefore a tile must not exceed the maximum
   *     size allowed by the GPU.  See SbGPUCapabilities::getMax3DTextureSize().
   *
   *   - If the tile dimension is set larger than the default value and the #SoVolumeRender::subdivideTile
   *     field is set to true, then the subtileDimension should
   *     also be set to a larger value to avoid reducing performance. In general we recommend
   *     setting #subTileDimension to #tileDimension/4
   */
  SoSFVec3i32 tileDimension;

  /**
   * Size of subtiles used by some accelerated algorithms.
   * Default is (32, 32, 32).
   * This field is only considered when field #SoVolumeRender::subdivideTile is set to TRUE.
   * Notes:
   *   - The subTileDimension values must be a power-of-2, e.g. 32, 64, 128, etc.
   *   - It takes time to build subtiles, in general we recommend setting #subTileDimension to #tileDimension/4.
   *   - The default subTileDimension (32) is appropriate for the default tileDimension (128),
   *     but should be set to a larger value if the tileDimension is a larger value.  
   *     Using too small a value will reduce performance.
   */
  SoSFVec3i32 subTileDimension;

  /**
   * Tile loading notification rate.
   * Default is 50 tiles.
   * When @B rate@b tiles have been loaded in main memory, a redraw will be scheduled.
   * Which means, for the default rate, that if each voxel is one byte and the
   * tile size is 64x64x64, a redraw will be scheduled whenever 13 MB of data has been loaded.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation takes into account all nodes in
   * scenegraph.
   *
   * If value is not -1 (default) LDM will not consider this node in automatic allocation, and allocated Notification rate for this 
   * node will be the exact value set in the field. In this case, the fixed value of notification rate is then removed from Global value.
   * All other dataset will share (SoLDMGlobalResourceParameters::getLoadNotificationRate - ( Sum of memory of all
   * datasets with * #loadNotificationRate != -1 )) / ( number of datasets with #loadNotificationRate == -1 )
   */
  SoSFInt32 loadNotificationRate;

  /**
   * Tile half-life (seconds).
   * Default is 2 seconds.
   * When calling SoDataSet::getLdmDataAccess::getData(), once a tile is unlocked by
   * the application (call to SoDataSet::getLdmDataAccess::releaseData), its weight will decrease
   * until the tile is weighted again by LDM. This function allows you to set how long
   * it takes for the tile's weight to be divided by 2 (initial weight is 1). 
   * The tile weight will be set to 0 when the weight becomes less than 0.05.
   */
  SoSFFloat tileHalfLife;

  /**
   *
   * Specifies if a single resolution or several resolutions are used when the rendering is stabilized.
   *
   * When TRUE, this single resolution level is specified by the field #resolution.
   *
   * Default is FALSE, which specifies that the field #resolution is ignored.
   */
  SoSFBool fixedResolution;

  /**
   *
   * Specifies the resolution level when #fixedResolution is TRUE.
   *
   * - When the value of this field is greater or equal to 0, it specifies the volume resolution used for rendering:
   *     0 for full resolution and higher for sub-resolutions.
   *     The resolution value may be incompatible with the hardware configuration (e.g. not enough memory)
   *     or with the LDM settings (i.e. see #maxMainMemory, #maxTexMemory and SoLDMGlobalResourceParameters for more details).
   *     In this case, this field and #fixedResolution are ignored.@BR
   *     Note that there is no rendering until the given resolution is reached.
   *
   * - When this field value is set to -1, the fixed resolution is adaptive, and is computed to be the highest one according to:
   *     - the memory resources
   *     - the position of the camera
   *     - the region of interest (see SoROI for more details)
   *     .
   *   Note that several resolutions are displayed during the loading and before reaching
   *   this adaptive fixed resolution.
   *
   * Default is 1, i.e, 1 voxel out of 2 of the full resolution.
   */
  SoSFInt32 resolution;

  /**
   * Number of threads used by the tile prefetch mechanism.
   * Tile prefetch is activated when LDM_USE_PREFETCH_OPTIM is true and #tileCachePolicy is NONE.
   * Default value is -1, meaning the maximum number of threads will be used.
   */
  SoSFInt32 numPrefetchThread;

  /**
   * Defines whether or not to generate borders around tiles.
   * Borders are used to accelerate the interpolation of voxel values at the
   * boundary of adjacent tiles.
   *
   * - If set to TRUE:
   *   - Borders are generated around tiles. When interpolation between adjacent
   *     tiles is enabled (see SoVolumeShader::interpolateOnMove), a rendering
   *     performance gain can be observed (compared to when tileBorders is
   *     FALSE), at the cost of a precomputation that is done each time a tile
   *     is sent to the GPU. Using borders will also slightly increase the size
   *     of the tile textures on the GPU.
   *
   * - If set to FALSE:
   *   - No borders are generated around tiles. No precomputation is required
   *     before sending a tile to the GPU. Interpolation between adjacent tiles
   *     is computed "on the fly" (slower compared to when tileBorders is TRUE,
   *     but the cost might be negligible on modern GPUs).
   *
   * Limitations:
   * - Volume projection and rectilinear coordinates are not supported when this
   *   value is set to FALSE (artifacts appear at the intersection between
   *   tiles). Set to TRUE instead in those cases.
   *
   * The default value is TRUE.
   *
   * @FIELD_SINCE_OIV 10.9.0
   */
  SoSFBool tileBorders;

  /**
   * Minimum number of tiles that must be loaded in memory before rendering.
   *
   * For example, if this number is 50 and 40 tiles are currently loaded, then
   * at least 10 new tiles must be loaded before the next rendering.
   *
   * This can be used as a "minimum quality" parameter, where the quality is
   * defined by the number of loaded tiles in memory. No rendering will be done
   * until this minimum number of tiles is loaded in memory.
   *
   * The tiles are selected according to their weight, which depends in
   * particular on their hierarchy level in the octree topology.
   *
   * If this number of tiles requires a memory amount that exceeds
   * #maxMainMemory, then it will be internally clamped to the maximum value
   * that respects #maxMainMemory.
   *
   * The default value is 1 (root tile only).
   *
   * @FIELD_SINCE_OIV 10.9.0
   */
  SoSFInt32 minTilesToLoad;

  /**
   * Viewpoint refinement policies.
   *
   * Used by #viewpointRefinementPolicy.
   */
  enum ViewpointRefinementPolicy
  {
    /** Viewpoint will not affect tile loading priorities. */
    VIEWPOINT_REFINEMENT_OFF,

    /**
     * The refinement of tiles will depend on the viewpoint.
     * In other words, the closest tiles to the camera have the highest priority
     * and will be loaded first. This can be useful for large volumes that
     * cannot be completely loaded in memory.
     */
    VIEWPOINT_REFINEMENT_ON,

    /**
     * The refinement of tiles will depend on the viewpoint or not according to
     * the flag returned by SoLDMGlobalResourceParameters::getViewpointRefinement().
     */
    VIEWPOINT_REFINEMENT_INHERITED_FROM_GLOBAL,
  };

  /**
   * Defines the viewpoint refinement policy.
   *
   * @useenum{ViewpointRefinementPolicy}.
   * The default value is #VIEWPOINT_REFINEMENT_INHERITED_FROM_GLOBAL.
   *
   * @FIELD_SINCE_OIV 10.9.0
   */
  SoSFEnum viewpointRefinementPolicy;

  /**
   * Screen resolution culling policies.
   *
   * Used by #screenResolutionCullingPolicy.
   */
  enum ScreenResolutionCullingPolicy
  {
    /** Screen resolution culling is disabled. */
    SCREEN_RESOLUTION_CULLING_OFF,

    /**
     * Screen resolution culling is enabled.
     * Only tiles for which the projection of a voxel is greater than or equal
     * to 1 pixel on screen will be loaded. This avoids unnecessary loading of
     * high resolution data for large volumes. However if you "zoom out" from a
     * volume, it does not force lower resolution tiles to be used (high
     * resolution tiles already in memory will still be used). If necessary use
     * the #fixedResolution field to force use of lower resolution tiles.
     */
    SCREEN_RESOLUTION_CULLING_ON,

    /**
     * Screen resolution culling will be enabled or not according to the flag
     * returned by SoLDMGlobalResourceParameters::getScreenResolutionCulling()
     */
    SCREEN_RESOLUTION_CULLING_INHERITED_FROM_GLOBAL,
  };

  /**
   * Defines the screen resolution culling policy.
   *
   * @useenum{ScreenResolutionCullingPolicy}.
   * The default value is #SCREEN_RESOLUTION_CULLING_INHERITED_FROM_GLOBAL.
   *
   * @FIELD_SINCE_OIV 10.10.0
   */
  SoSFEnum screenResolutionCullingPolicy;

  /**
   *
   * Sets the amount of time (in seconds) during which LDM continues not loading once a user
   * stops interaction.
   * In other words, LDM will start loading again timeOut seconds after the
   * user stops interaction.
   * Considered only in NO_USER_INTERACTION mode.
   * Default is 0.3 seconds.
   */
  void setMovingTimeOut(float timeout);

  /**
   *
   * Returns the amount of time during which LDM continues not loading data once a user
   * stops interaction.
   * Considered only in NO_USER_INTERACTION mode.  See #setMovingTimeOut().
   */
  float getMovingTimeOut();

  /**
   * Gets the size of a voxel in bytes.
   */
  int getDataSize();

  /**
   * This method is used to specify a \if_dotnet delegate \else callback \endif that gives feedback
   * while the rendering tries to reach an explicit required fixed #resolution.
   * This method sets the field #fixedResolution to TRUE and the field #resolution to the first argument.
   *
   * Use this method only if a \if_dotnet delegate \else callback \endif is specified and if an explicit resolution
   * is required (0 or any positive value). Otherwise use the fields #fixedResolution and #resolution.
   *
   * @param resolution this value is assigned to the field #resolution
   * @param func this \if_dotnet delegate \else callback \endif is ignored if the resolution argument is -1.
   *        See #resolution and #fixedResolution for detail.
   */
  void enableFixedResolutionMode(int resolution, FixedResolutionCB* func);

  /**
   * Sets the importance of one type of geometry relative to others.
   *
   * The geometry type should be one of the VolumeViz geometry classes. For example,
   * SoOrthoSlice::getClassTypeId().
   *
   * The priority passed must be between 0 and 1, 0 being lowest priority and 1 the highest.
   * LDM will refine tiles first where geometries are the most important.
   * For example, if an SoVolumeRender node has a higher priority than slices, then the tiles required for
   * this SoVolumeRender node will be refined first.
   *
   * Default is 0.5 for a volume, 0.9 for other geometry.
   */
  void setGeometryPriority(SoType geometryType, float priority);

  /**
   * Returns the priority of the specified geometry type.
   * The function will return FALSE if the geometry is of unknown type.  See #setGeometryPriority().
   */
  SbBool getGeometryPriority(SoType geometryType, float& priority);

SoINTERNAL public:
  typedef struct GeomPriority
  {
    float   m_weight;
    SoType  m_geomId;
  } GeomPriority;

  /**
   * Sets the data set at which this node is bound.
   */
  void setDataSet(SoDataSet *ds);

  /**
   * Copies some LDM parameters which are not in fields.
   */
  void copyParameters(SoLDMMediator *mediator);

  /**
   * Prints all parameters into f.
   */
  void fprintfResourceManager(FILE *f);

  /**
   * This function allows you to synchronously unload the data if it needs to be refetched.
   */
  void invalidateData();

  /**
   * This function is used when collecting the geometry.
   * If #fixedResolution mode is enabled, each collected geometry will count how many tiles
   * are needed to render them.
   * After geometry collection, volumeData will report to the application the total count.
   */
  void setNumTiles(int numTiles);

  /**
   * Return how many tiles are needed to render at a given level of resolution
   */
  int getNumTiles();

  /**
   * Converts geometry coordinates to voxel coordinates for a geometry box.
   * (useful to figure out num tiles needed if #fixedResolution mode is enabled)
   */
  SbBox3i32 XYZToVoxel(const SbBox3f &dataBox);


  /**
   * Fixed resolution methods
   */
  bool isReportDisabled();
  void setReportDisabled(bool flag);
  FixedResolutionCB* getFixedResCB();
  int getNumTilesLoaded();
  bool toReport();
  void setToReport(bool flag);
  void increaseNumTilesLoaded();
  void decreaseNumTilesLoaded();
  void resetNumTilesLoaded();
  void resetFixedResVar();

  void sendToLdm(SoDataSet* ds);

  void enableParamsNotify(SbBool state);

  virtual void notify(SoNotList*);

  inline void setResourceManager(SoLDMResourceManager* resourceManager);

  /**
   * If true, load the maximum possible resolution.
   * This used to simulate the brute force render mode of volumeviz
   * but with LDM.
   */
  inline void forceFixedResolution(bool flag);

  /**
   * Return true if maximum possible resolution
   * must be loaded synchronously.
   */
  inline bool isFixedResolutionForced() const;

  /**
   * Internal in 9.3 for Avizo experimentation.
   * Resolution level for fixed resolution mode.
   * See #resolution field for more details.
   */
  enum Resolution {
    /**
     * Best equal resolution which can be loaded
     * according to memory parameters.
     */
    BEST = -1,
    /**
     * Full resolution.
     */
    FULL = 0
  };

private:

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Sets the tile overlap value (number of voxels) for non-LDM data.
   * 0 is the only supported value since Open Inventor 9.0.
   */
  SoDEPRECATED_FIELD(9000,"No longer used")
  SoSFInt32 overlapping;

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9100)

  /**
   * Maximum number of tiles in main memory for this data set. @BR
   * Changing this number will automatically adjust the maximum amount of memory
   * set by #maxMainMemory (memory required for each tile depends on tile size
   * and voxel size).
   * Default is (maxMainMemory/numDatasets)/(tileSize in MB).
   *
   * Only #maxMainMemory can be used. Field as no more any effect on others parameters
   */
  SoDEPRECATED_FIELD(9100,"No longer used.")
  SoSFInt32 maxTilesInMainMem;

  /**
   * Maximum number of 3D tiles in texture memory for this data set.
   * Each tile needed for volume rendering (SoVolumeRender) and volume geometry must be
   * stored in GPU memory as a 3D texture.
   * Changing this number will automatically adjust the field #maxTexMemory (maximum
   * amount of texture memory). Default is maxTexMemory/(tileSize in MB).
   *
   * Only #maxTexMemory can be used. Field as no more any effect on others parameters
   */
  SoDEPRECATED_FIELD(9100,"No longer used.")
  SoSFInt32 maxTilesInTexMem;

  /**
   * Lock parameter values.
   * Default is FALSE.
   * By default, the resource parameters are automatically recalculated if the parameters
   * for another data set change or a new data set is created (to respect the global amount of resources).
   * If this field is TRUE, the resources for this data set are not allowed to change.
   *
   * In Since OIV 9.1 memory parameters are fixed as soon as the user change the value of one of the field
   * managing memory (CPU and/or GPU). If field is set to -1, LDM is allowed to change the amount of memory
   * given to the dataset. If field is not equal to -1 it means the user define a specific value that LDM
   * won't be allowed to change
   */
  SoDEPRECATED_FIELD(9100,"No longer used.")
  SoSFBool fixedParams;

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9500)

  /**
   * Maximum number of 2D tiles allowed in texture (GPU) memory for this data set.
   * Each tile needed to render a slice (SoOrthoSlice, SoVolumeSkin, etc) must be stored
   * in GPU memory as a 2D texture.  This method limits the amount of GPU memory that can
   * be used by (for example) SoOrthoSlice and SoVolumeSkin nodes.  This can be useful when
   * combining slices and volume rendering because the 3D textures used by volume rendering
   * require much more GPU memory.
   *
   * Default value is -1, meaning LDM is in charge of the automatic sharing of a given resource.
   * Automatic sharing means LDM will compute, by itself, the amount
   * of associated resource it allocates to each dataset. The automatic allocation takes into account all nodes in
   * scenegraph.
   * 
   * If value is not -1 (default) LDM will not consider this node in memory allocation, and allocated CPU memory for this 
   * node will be the exact value set in the field. In this case, the amount of memory is removed from Global value.
   * All other dataset will share 
   * (SoLDMGlobalResourceParameters::getMax2DTextures  - ( Sum of memory of all datasets with #max2DTextures != -1 )) / ( number of datasets with #max2DTextures == -1 )
   */
  SoDEPRECATED_FIELD(9500,"No longer used.")
  SoSFInt32 max2DTextures;

  // TODO: the following friend dependencies should be removed once internal dependencies are removed
  friend class SoLDMResourceManager;
#endif /** @DEPRECATED_END */

  // Deprecated in 9700 and removed (hidden) in 10000.
  // No longer used by the default heightfield rendering algorithm( SoHeightFieldRender )
  // Use SoComplexity instead.
  SoSFInt32 maxNumTriangles;

  bool m_forceFixedResolution;

  /**
   * Forbid recopy constructors
   */
  SoLDMResourceParameters& operator=(const SoLDMResourceParameters& ldmrp);
  SoLDMResourceParameters(const SoLDMResourceParameters& );

  //Methods used in ldmParameterChangedCB
  void updateTileHalfLife(float timeInSec);
  void updateTileSize();
  void updateMaxResolutionThreshold( int threshold );
  void updateMinResolutionThreshold( int threshold );
  void updateLoadPolicy( LoadPolicy loadPolicy );
  void updateEnableFixedResolutionMode(int resolution, FixedResolutionCB* func);
  void updateDisableFixedResolutionMode();

  SoDataSet* m_ds;
  SoLDMResourceManager* m_resourceMgr;

  FixedResolutionCB* m_fixedResCB;
  int  m_numTilesLoaded;
  int  m_numTilesNeeded;
  bool m_toReport;
  bool m_reportDisable;
  bool m_isFixedRes;
  bool m_bLdmParamDebug;

  /** used to avoid calling m_resourceMgr->getMinResolutionThreshold */
  int m_previousMinThreshold;

  //to save current system state before using fixed res mode
  LoadPolicy         m_policyUsed;
  int                m_resolution;
  SoLDMTileVisitor*  m_fixedResVisitor;
  SoLDMTileVisitor*  m_beforeFixedResVisitor;
  bool m_beforeFixedResVisitorFlag;

  float m_movingTimeOut; //keep not loading for timeOut sec after the user stopped interacting.

  std::vector<GeomPriority> m_geomPriority;
  int getGeometryIndex(SoType geometryType);

  static bool s_artForceFixed;
};

/***************************************************************************/
inline bool
SoLDMResourceParameters::isReportDisabled()
{
  return m_reportDisable;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::setReportDisabled(bool flag)
{
  m_reportDisable = flag;
}

/***************************************************************************/
inline SoLDMResourceParameters::FixedResolutionCB*
SoLDMResourceParameters::getFixedResCB()
{
  return m_fixedResCB;
}

/***************************************************************************/
inline int
SoLDMResourceParameters::getNumTilesLoaded()
{
  return m_numTilesLoaded;
}

/***************************************************************************/
inline bool
SoLDMResourceParameters::toReport()
{
  return m_toReport;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::setToReport(bool flag)
{
  m_toReport = flag;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::increaseNumTilesLoaded()
{
  m_numTilesLoaded++;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::decreaseNumTilesLoaded()
{
  m_numTilesLoaded--;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::resetNumTilesLoaded()
{
  m_numTilesLoaded = 0;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::setResourceManager(SoLDMResourceManager* resourceManager)
{
  m_resourceMgr = resourceManager;
}

/***************************************************************************/
inline SoLDMResourceParameters&
SoLDMResourceParameters::operator=(const SoLDMResourceParameters& )
{
  assert(0);
  return *this;
}

/***************************************************************************/
inline void
SoLDMResourceParameters::forceFixedResolution(bool flag)
{
  m_forceFixedResolution = flag;
}

/***************************************************************************/
inline bool
SoLDMResourceParameters::isFixedResolutionForced() const
{
  return m_forceFixedResolution || s_artForceFixed;
}

/***************************************************************************/
extern LDM_API std::ostream& operator << (std::ostream& os, const SoLDMResourceParameters& p);

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif


