/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef  _SO_ROI_
#define  _SO_ROI_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFBox3i32.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/sensors/SoNodeSensor.h>


class SoGLRenderAction;
class SoCallbackAction;
class SoWriteAction;

/**
* @LDMEXT Region of Interest (subvolume) node.
*
* @ingroup LDMNodes
*
* @DESCRIPTION
*   The ROI ("region of interest") node allows you to specify a sub-region of the
*   volume that will be rendered. Voxels within the ROI are rendered; voxels outside
*   the ROI are not rendered.  This provides a simple and very efficient form of
*   volume clipping (much more efficient than using SoClipPlane nodes).
*
*   The SoVolumeData node on which the ROI will be applied can be specified with #dataSetId.
*   When this field is set to 0, the last SoVolumeData node on state is used.
*
*   In addition, for large volumes that cannot be fully loading into CPU and/or GPU
*   memory, SoROI can improve both data loading performance and image quality.
*   VolumeViz will always try to load the highest resolution data possible for the voxels
*   inside the region defined by the ROI. VolumeViz is not @I required@i to load data in
*   tiles that are completely outside the ROI. VolumeViz may load tiles outside the ROI
*   if there is sufficient memory (this is useful in cases where the user can move or
*   resize the ROI). But in general, for large volumes, using SoROI reduces the amount of
*   data that must be loaded and increases image quality for the voxels inside the ROI.
*
*   Note that the above discussion only applies in multi-resolution mode (the default).
*   In fixed resolution mode (see SoLDMResourceParameters::fixedResolution), VolumeViz
*   will try to load all the tiles in the volume at the specified resolution level
*   regardless of the SoROI settings.
*
*   The ROI can be a simple box (default) or a complex shape defined by (up to) three
*   box shapes (X, Y, and Z slabs) and logic flags that specify how the slabs are to
*   be combined.
*
*   SoROI is commonly used with an opaque color map to provide a "volume probe" in
*   seismic data applications.  However it is useful with any kind of volume data to
*   limit the data displayed and to improve loading and image quality as discussed above.
*
*   The #box and #subVolume fields are specified in voxel coordinates. The limits are included in the
*   ROI. A value of 0,0,0 0,0,0 (min and max) in the #subVolume field means that
*   this field should be ignored. This is the default.
*
*   This node acts on the rendering shape nodes of VolumeViz (SoVolumeRender, SoOrthoSlice,
*   SoObliqueSlice, SoVolumeSkin, etc.)
*
*   To define a simple ROI, set the limits of the ROI in the #box field
*   (and do not set the #subVolume field). The same result is obtained by
*   setting the #box and #subVolume fields to the same value, but this is not necessary.
*   But note that the default ROI box is @I not @i automatically the
*   full volume. You must initialize the ROI box to something, for example
*   the volume dimensions minus one.
  *  \if_cpp
  *    \code
  *      // Initialize ROI box to full volume
  *      SoVolumeData* volData = new SoVolumeData();
  *      . . .
  *      SoROIManip*   roiManip = new SoROIManip();
  *      roiManip->box.setValue( SbVec3i32(0,0,0), volData->data.getSize() - SbVec3i32(1,1,1) );
  *      root->addChild( roiManip );
  *    \endcode
  *   \endif
  *   \if_dotnet
  *    \code
  *      // Initialize ROI box to full volume
  *      SoVolumeData volData = new SoVolumeData();
  *      . . .
  *      SoROIManip roiManip = new SoROIManip();
  *      roiManip.box.SetValue( new SbVec3i32(0, 0, 0), volData.data.GetSize() - new SbVec3i32(1, 1, 1));
  *      root.AddChild( roiManip );
  *    \endcode
  *   \endif
  *   \if_java
  *     \code
  *       // Initialize ROI box to full volume
  *       SoVolumeData volData = new SoVolumeData();
  *       . . .
  *       SoROIManip roiManip = new SoROIManip();
  *       roiManip.box.setValue( new SbVec3i32(0, 0, 0), volData.data.getSize().minus( new SbVec3i32(1, 1, 1)));
  *       root.addChild( roiManip );
  *     \endcode
  *   \endif
*
*   For a complex ROI,
*   the region defined by the SoROI #box field always acts upon the region
*   defined by the SoROI #subVolume field, not the entire volume. For example,
*   in EXCLUSION_BOX mode, the visible portion of the volume is the #subVolume
*   region minus the #box region. You are allowed to set the #box region larger
*   than (or completely outside) the #subVolume region, but only the intersection
*   of the two regions is significant.
*
*   You can also use the convenient manipulator class SoROIManip to allow your
*   users to interactively move and resize the region of interest.
*
*The figures below show the #subVolume field used to limit the
*             visible portion of the volume (left) and the ROI used as an
*             "exclusion box" to cut away part
*             of the subvolume (right).
*
*      @TABLE_0B
*         @TR Example: #box field used to limit visible portion of the volume
*         @TR @IMAGE subvolume.jpg
*      @TABLE_END
*
*      @TABLE_0B
*         @TR Example: #box field + #subVolume field used as an exclusion box
*         @TR @IMAGE roiexclusion.jpg
*      @TABLE_END
*
*   @B The Crop Box and Cropping Process. @b
*
*   The crop box is defined by three sets of parallel planes that define three
*   slabs:
*
*   - The xmin and xmax planes define the X slab.
*   - The ymin and ymax planes define the Y slab.
*   - The zmin and zmax planes define the Z slab.
*
*   After these three planes have been specified, cropping is done in four stages.
*
*   Stage 1 determines which voxels to include relative to one or more of the slabs.
*   Classification can be enabled or disabled relative to the X slab, Y slab, or Z
*   slab. This classification is performed three separate times, resulting in three
*   terms: Term 0, Term 1, and Term 2. Each term has its own logic flags that enable
*   the three slabs, independent of the other terms. The flags are as follows:
*
*   - Enable or disable classification relative to the X slab.
*   - Enable or disable classification relative to the Y slab.
*   - Enable or disable classification relative to the Z slab.
*
*   Stage 2 determines whether to invert the values obtained in Stage 1 so that the
*   voxels @I outside @i a slab are selected. This determination is made for each
*   of the terms (Term 0, Term 1, Term 2). As in Stage 1, each term has its own
*   inversion flag and is independent of the other terms.
*
*   Stage 3 creates either the @I union @i or the @I intersection @i of Term 0,
*   Term 1, and Term 2. This is specified using a logic flag.
*
*   Stage 4 determines whether to @I invert @i the result of Stage 3 and provides
*   the final ROI. Again, the inversion (if any) is specified using a logic flag.
*
*
*
*   @B Example of Cropping Process @b
*
*   (Figures courtesy Real Time Visualization)
*
*   The following example will show you how the cropping process works.
*
*   Start with a volume that is 100x200x100 (x, y, z). The final cropped shape will
*   be the union of a 20x200x20 bar and a 100x25x100 box.
*
*   Here is the initial volume:
*
*   @IMAGE ROI_a.jpg
*
*   Here is the bar:
*
*   @IMAGE ROI_b.jpg
*
*   The bar can be formed in Term 0 of Stage 1 by using the intersection of the X and
*   Z slabs, each with min and max values set to 40 and 60, respectively.
*
*   Here is the box:
*
*   @IMAGE ROI_c.jpg
*
*   The box can be formed in Term 1 of Stage 1 by using just the Y slab, with min and
*   max values set to 125 and 150.
*
*   Term 2 of Stage 1 can be set to be identical to Term 0 or Term 1, or it can be
*   set to include no samples by setting no enable flags (no slabs selected, so
*   entire volume is selected) and setting the invert flag of Stage 2 so that the
*   entire volume is deselected. To get the union of these terms in Stage 3,
*   OR_SELECT is set. That results in the desired cropping, so Stage 4, in which the
*   results are inverted, is not used.
*
*   Here is the code to set the box dimensions and the logic flags:
*
*  \if_cpp
*   \code
*      SoROI* myROI = new SoROI();
*      int xmin =  40; int xmax =  60;
*      int ymin = 125; int ymax = 150;
*      int zmin =  40; int zmax =  60;
*      myROI->box.setValue( xmin, ymin, zmin, xmax, ymax, zmax );
*      myROI->flags = SoROI::ENABLE_X0 | SoROI::ENABLE_Z0 | SoROI::ENABLE_Y1 | SoROI::INVERT_2 | SoROI::OR_SELECT;
*   \endcode
*  \endif
*  \if_dotnet
*   \code
*      SoROI myROI = new SoROI();
*      int xmin =  40; int xmax =  60;
*      int ymin = 125; int ymax = 150;
*      int zmin =  40; int zmax =  60;
*      myROI.box.SetValue( xmin, ymin, zmin, xmax, ymax, zmax );
*      myROI.flags.Value = SoROI.FlagsType.ENABLE_X0 | 
*                          SoROI.FlagsType.ENABLE_Z0 |
*                          SoROI.FlagsType.ENABLE_Y1 |
*                          SoROI.FlagsType.INVERT_2  |
*                          SoROI.FlagsType.OR_SELECT;
*   \endcode
*  \endif
*  \if_java
*   \code
*      SoROI myROI = new SoROI();
*      int xmin =  40; int xmax =  60;
*      int ymin = 125; int ymax = 150;
*      int zmin =  40; int zmax =  60;
*      myROI.box.SetValue( xmin, ymin, zmin, xmax, ymax, zmax );
*      myROI.flags.setValue( SoROI.FlagsType.ENABLE_X0.getValue() |
*                            SoROI.FlagsType.ENABLE_Z0.getValue() |
*                            SoROI.FlagsType.ENABLE_Y1.getValue() |
*                            SoROI.FlagsType.INVERT_2.getValue()  |
*                            SoROI.FlagsType.OR_SELECT.getValue()  );
*   \endcode
*  \endif
*   Here is the resulting complex crop box:
*
*   @IMAGE ROI_d.jpg
*
*
*   @B Additional Examples @b
*
*      @TABLE_0B
*         @TR @IMAGE ROI_ex1.jpg
*           @TD @B Example 1: @b flags = SUB_VOLUME
*
*                  Alternate setting:flags = ENABLE_X0 | ENABLE_Y1 | ENABLE_Z2
*         @TR @IMAGE ROI_ex2.jpg
*           @TD @B Example 2: @b flags = FENCE
*
*                  Alternate setting:flags = ENABLE_X0 | ENABLE_Y1 |
*                                            ENABLE_Z2 | OR_SELECT
*         @TR @IMAGE ROI_ex3.jpg
*           @TD @B Example 3: @b flags = FENCE
*
*                  Note that example 2 and 3 have the same flags set;
*                  but in example 2, xmin, ymin, and zmin values are set to zero. Setting its xmax,
*                  ymax, and zmax values to the maximum value produces a similar crop.
*         @TR @IMAGE ROI_ex4.jpg
*           @TD @B Example 4: @b flags = FENCE_INVERT
*
*                  Alternate setting:flags = ENABLE_X0 |
*                                            ENABLE_Y1 | ENABLE_Z2 | OR_SELECT | INVERT_OUTPUT
*         @TR @IMAGE ROI_ex5.jpg
*           @TD @B Example 5: @b flags = CROSS
*
*                  Alternate setting:flags = ENABLE_X0 | ENABLE_Y0 |
*                                            ENABLE_Y1 | ENABLE_Z1 | ENABLE_X2 | ENABLE_Z2 | OR_SELECT
*         @TR @IMAGE ROI_ex6.jpg
*           @TD @B Example 6: @b flags = CROSS_INVERT
*
*                  Alternate setting:flags = CROSS | INVERT_OUTPUT
*      @TABLE_END
*
* @FILE_FORMAT_DEFAULT
*    ROI {
*    @TABLE_FILE_FORMAT
*       @TR dataSetId   @TD 0
*       @TR subVolume   @TD 0, 0, 0, 0, 0, 0
*       @TR box         @TD 0, 0, 0, 1, 1, 1
*       @TR flags       @TD 7
*       @TR relative    @TD FALSE
*    @TABLE_END
*    }
*
* @B LIMITATIONS @b
* Only one ROI per volume data can be used at same time.
*
* @SEE_ALSO
*    SoVolumeRender,
*    SoOrthoSlice,
*    SoObliqueSlice,
*    SoROIManip
*
* @ACTION_BEHAVIOR
* SoCallbackAction,
* SoGLRenderAction,
* SoGetBoundingBoxAction,
* SoPickAction,
* SoWriteAction
*
*
*/
class LDM_API SoROI : public SoNode {
  SO_NODE_HEADER( SoROI );

 public:
  /**
   * Specifies which SoVolumeData node to use.
   *
   * This is useful when datasets of different dimensions are present in the scene graph.
   * Please see SoMultiDataSeparator for more details.
   *
   * When set to 0, the last SoVolumeData node on state is used.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 10.11.0
   */
  SoSFInt32 dataSetId;

  /**
   * Specifies the bounds of the region of interest. If #subVolume is non-empty,
   * the box is clamped at the bounds of the subvolume. Depending on the
   * #relative field, the bounds may be relative to the subvolume. Default is
   * an (essentially) empty box.
   */
  SoSFBox3i32 box;

  /**
   * Specifies how the bounds of the box are used. Default is SUB_VOLUME. The region of
   * interest is the box itself.
   */
  SoSFBitMask flags;

  /**
   * Flag value mask
   */
  enum Flags {
    /** Enable cropping with X slab for term 0 (stage 1) */
    ENABLE_X0 = 0x1,
    /** Enable cropping with Y slab for term 0 (stage 1) */
    ENABLE_Y0 = 0x2,
    /** Enable cropping with Z slab for term 0 (stage 1) */
    ENABLE_Z0 = 0x4,
    /** Invert result for this term 0 (stage 2) @BR */
    INVERT_0  = 0x8,

    /** Enable cropping with X slab for term 1 (stage 1) */
    ENABLE_X1 = 0x10,
    /** Enable cropping with Y slab for term 1 (stage 1) */
    ENABLE_Y1 = 0x20,
    /** Enable cropping with Z slab for term 1 (stage 1) */
    ENABLE_Z1 = 0x40,
    /** Invert result for this term 1 (stage 2) @BR */
    INVERT_1  = 0x80,

    /** Enable cropping with X slab for term 2 (stage 1) */
    ENABLE_X2 = 0x100,
    /** Enable cropping with Y slab for term 2 (stage 1) */
    ENABLE_Y2 = 0x200,
    /** Enable cropping with Z slab for term 2 (stage 1) */
    ENABLE_Z2 = 0x400,
    /** Invert result for this term 2 (stage 2) @BR */
    INVERT_2  = 0x800,

    /**
     * Stage 3: If set, the result is the union (OR) of term 0, term 1, and term 2. If
     * clear, the result is the intersection (AND) of the three terms @BR
     */
    OR_SELECT     = 0x1000,

    /**
     * Stage 4: If enabled, the result of stage 4 (union or intersection) is inverted
     * as the last step in cropping @BR
     */
    INVERT_OUTPUT = 0x2000,

    /** Convenient enums */
    SUB_VOLUME    = ENABLE_X0 | ENABLE_Y0 | ENABLE_Z0,
    EXCLUSION_BOX = SUB_VOLUME | INVERT_OUTPUT,
    CROSS         = ENABLE_X0 | ENABLE_Y0 | ENABLE_Y1 | ENABLE_Z1 | ENABLE_X2 | ENABLE_Z2 | OR_SELECT,
    CROSS_INVERT  = CROSS | INVERT_OUTPUT,
    FENCE         = ENABLE_X0 | ENABLE_Y1 | ENABLE_Z2 | OR_SELECT,
    FENCE_INVERT  = FENCE | INVERT_OUTPUT
  };

  /**
   * Specifies the bounds of the subvolume that will be rendered. By default, it is an
   * empty box and has no effect. The subvolume is always a simple box.
   */
  SoSFBox3i32 subVolume;

  /**
   * Specifies whether the box bounds are relative to the subvolume or the full volume
   * (i.e., are specified in absolute slice coordinates). TRUE means that if the
   * subvolume is non-empty, moving the subvolume through the volume also moves the
   * ROI box through the volume. Default is FALSE.
   */
  SoSFBool  relative;

  /**
   * Constructor.
   */
  SoROI();

 SoEXTENDER public:
  /** @copydoc SoNode::doAction */
  virtual void doAction( SoAction *action );
  /** @copydoc SoNode::callback */
  virtual void callback( SoCallbackAction *action );
  /** @copydoc SoNode::GLRender */
  virtual void GLRender( SoGLRenderAction *action );
  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  /** @copydoc SoNode::pick */
  virtual void pick(SoPickAction *action) ;
  /** @copydoc SoNode::write */
  virtual void write(SoWriteAction *action);
  
  /** @copydoc SoNode::getAlternateRep */
  virtual SoNode* getAlternateRep( SoAction* action );

  //------------------------------------------------------------------------------

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /**
   * Return the list of all visible boxes
   * @param boxes is the list of visible boxes
   * @param dim is the volume ijk dimension
   * @return the nimber of element in boxes
   */
  int getVisibleBoxes(SbBox3i32* &boxes, const SbVec3i32 &dim);

  /**
   * Compute the bounding box enclosing all visible boxes
   * @param box is the bounding box enclosing all visible boxes
   * @param dim is the volume ijk dimension
   */
  void getVisibleBoundingBox(SbBox3i32& box, const SbVec3i32 &dim);

 protected:
  // Destructor
  virtual ~SoROI();

 private:
  SoNodeSensor  *m_FiedsNS;
  int            m_NumVisBoxes;
  SbBox3i32     *m_VisBoxes;
  SbVec3i32      m_dim;
  unsigned int   getVisibleParts();
  static void    fieldsChangedCB(void *, SoSensor *);

};
//--------------------------------------------------------------------------------

#endif // _SO_ROI_


