/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef  _SO_TRANSFER_FUNCTION_
#define  _SO_TRANSFER_FUNCTION_

#include <LDM/SoLDM.h>

#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoMFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/algorithms/SoConversion.h>
#include <Inventor/SbVec.h>

#include <Inventor/SbPImpl.h>

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SoGLTexture;
class SoVolumeState;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoTransferFunction)

/**
@LDMEXT Describes the association between data set values and colors.

@ingroup LDMNodes

@DESCRIPTION
  This node defines a mapping from a range of scalar values to a
  set of color and alpha (inverse of transparency) values.

  SoTransferFunction has no effect on rendering an RGBA volume
  (SoVolumeData::dataRGBA field contains TRUE). In this case the voxel
  values are explicitly color values already.

#  Data range

  The range of scalar values is specified by an SoDataRange node.
  If the application does not provide an SoDataRange node, then the full
  range of the volume's data type is mapped into the color map.
  This can be appropriate and convenient for 8-bit (byte) data, but is
  usually incorrect for larger data types, especially float values.
  For example, DICOM data is normally stored in 16-bit voxels, which would
  have a default data range of 0..65535 (or -32767..32767 if signed).
  To get the correct image in this case the application should create an
  an SoDataRange node and set the data range to the actual range of data
  values, for example 0..4095. If the actual range is not known, the min
  and max values can be queried using the SoVolumeData getMinMax methods.
  Note that the
  minValue and maxValue fields of this node have a different meaning that
  is described later. An example data-to-color-mapping looks like this:

  @IMAGE colormapremap.jpg

#  Color maps

  If no transfer function has been specified, VolumeViz rendering
  nodes automatically use the predefined GRAY color map.
  To use one of the predefined color maps, set the #predefColorMap field
  to a valid value other than NONE, for example:
  - GRAY  
    In this color map all the color values are 1,1,1 (full white) and the alpha
    values are a linear ramp from 0 to 1.  This can be a useful starting point
    for volume rendering.
  - INTENSITY  
    In this color map the color values are a linear ramp from 0 to 1 and all alpha values
    are 1 (full opaque). This can be a useful starting point for slice rendering.

  To create a custom color map, set the #predefColorMap field to NONE,
  set the #colorMapType field to the appropriate type (RGBA is the default
  and most commonly used), then set the color map values in the #colorMap
  field. The #colorMap field
  contains an array of float values in the range [0,1]. The number of float
  values needed depends on the colorMapType. It is equal to the number of
  colors in the color map multiplied by the number of components in each color.
  An ALPHA color map has one component per color, a LUM_ALPHA (Luminance plus
  Alpha) color map has two components per color and an RGBA color map has four
  components per color. For example, for an RGBA color map of length N, there
  must be 4*N float values in the field.

  As usual with SoMF fields, memory is automatically allocated as needed if
  you set the values one at a time using the \if_dotnet Item property. \else
  set1Value() method. \endif This is convenient, but very inefficient.  If you
  must set values one at a time, use the \if_dotnet Count property \else
  setNum() method \endif to pre-allocate enough memory for all the float values.
  There is another reason not to set values one at a time. Changing a value in
  the colorMap field automatically updates the #actualColorMap field on each change.
  To avoid performance problems, we recommend using the setValues() or
  \if_cpp setValuesPointer() \else setValuesBuffer() \endif method to create the color map.
  \if_cpp
  \par
  \code
    // Create a "reversed" INTENSITY color map using RGBA values
    const int numColors = 256;
    SbColorRGBA* colors = new SbColorRGBA[numColors];
    for (int i = 0; i < numColors; ++i) {
      float intensity = (numColors - i)/(float)numColors;
      colors[i].setValue( intensity, intensity, intensity, 1 );
    }
    SoTransferFunction* tf = new SoTransferFunction();
    tf->predefColorMap = SoTransferFunction::NONE;
    tf->colorMap.setValues( 0, 4*numColors, (float*)colors );
    delete [] colors;
  \endcode
  \endif
  \if_dotnet
  \par
  \code
    // Create a "reversed" INTENSITY color map using RGBA values
    const int numColors = 256;
    float[] colors = new float[4numColors];
    for (int i = 0; i < numColors; ++i)
    {
        float intensity = (numColors - i) / (float)numColors;
        int index = i4;
        colors[index] = intensity;
        colors[index + 1] = intensity;
        colors[index + 2] = intensity;
        colors[index + 3] = 1;
    }
    SoTransferFunction tf = new SoTransferFunction();
    tf.predefColorMap.Value = SoTransferFunction.PredefColorMaps.NONE;
    tf.colorMap.SetValues(0, colors);
  \endcode
  \endif
  \if_java
  \par
  \code
    // Create a "reversed" INTENSITY color map using RGBA values
    int numColors = 256;
    float[] colors = new float[4numColors];
    for (int i = 0; i < numColors; ++i) {
      float intensity = (numColors - i) / (float)numColors;
      int index = i4;
      colors[index ] = intensity;
      colors[index+1] = intensity;
      colors[index+2] = intensity;
      colors[index+3] = 1;
    }
    SoTransferFunction tf = new SoTransferFunction();
    tf.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.NONE );
    tf.colorMap.setValues( 0, colors );
  \endcode
  \endif

  It is also possible to load a color map from a file using the #loadColormap()
  method. Currently the only supported file format is Avizo color map (.am or .col).

Note that the meaning of the min and max fields in SoDataRange is quite different than
the meaning of the #minValue and #maxValue fields in SoTransferFunction. The fields
in SoDataRange specify the range of voxel values that will be mapped into the full
color map.  But changing the minValue and maxValue fields in SoTransferFunction "remaps"
the color map, compressing the original list of color values into the specified range of
entries.  Entries outside this range are set to full transparent black (all components zero).
This does not affect the mapping of scalar values to the color map, it just modifies
the color map.  Scalar values in the current data range are still mapped to the full range
of color map entries.  This is mainly useful if you are using one of the
predefined color maps and want to quickly remove low value voxels that represent "noise"
in the data or to use the first or last entry in the color map for "undefined" values.
Visually the effect can be similar to changing the data range using SoDataRange, and
it is generally faster, but note that remapping effectively reduces the resolution of the
color map. In general it's better to change the range of data values mapped into the color
map using SoDataRange.

To modify an existing color map, use the setValues() method (as discussed above) or
use the start/finish editing methods. For example:@BR
\if_cpp
\par
\code
  SoTransferFunction* tf . . .
  float* data = tf->colorMap.startEditing();
  // Change data values
  data[0] = 1;
  . . .
  tf->colorMap.finishEditing();
\endcode
\endif
\if_dotnet
\par
\code
  SoTransferFunction tf . . .
  SbNativeArray<float> data = tf.colorMap.StartEditing();
  // Change data values
  data[0] = 1;
  . . .
  tf.colorMap.FinishEditing();
\endcode
\endif
\if_java
\par
\code
  SoTransferFunction tf . . .
  FloatBuffer data = tf.colorMap.StartEditing();
  // Change data values
  data.put( 1 );
  . . .
  tf.colorMap.FinishEditing();
\endcode
\endif

# Rendering

  For scalar volumes (non RGBA), the current data range and transfer function specify
  a color and an alpha value for each voxel of the volume. These colors and alpha are
  multiplied by the current base color on the state before applying the lighting effects.
  See the definition of base color @ref SoLightModel-base-color-def "here". The base color
  and alpha can be set for example with an SoMaterial or an SoPhysicalMaterial node
  in the scene graph.

  The transfer function is associated with the rendering node(s),
  not the volume data node.  So for example it is possible to use
  one transfer function for volume rendering (SoVolumeRender) and
  a different transfer function for slices (e.g. SoOrthoSlice).

  When rendering a single volume, there is only one transfer function
  active at a time and each rendering node uses the current (i.e. last
  traversed) transfer function in the traversal state, similar to the
  way materials apply to polygonal geometry.

  The application may switch from one transfer function to a different
  one.  For example, by putting multiple transfer functions under an
  SoSwitch node.  The application may also dynamically modify the
  contents of a transfer function, even if using a predefined color map.
  See the #colorMap and #actualColorMap fields.

  When combining multiple volumes using SoMultiDataSeparator and
  SoVolumeShader it is possible to specify multiple transfer functions.
  All the transfer functions under the SoMultiDataSeparator will be
  available to the shader program based on their #transferFunctionId.
  The application should assign a unique id to each transfer function.
  See the description of the #transferFunctionId field for more details.

  When using volume masks (SoVolumeMask and SoVolumeMaskGroup), the
  #transferFunctionId is also very important.  In this case multiple
  transfer functions, with different id values, may exist in the
  traversal state simultaneously. Each mask region is only visible if
  there exists a transfer function with the same id as that region.
  Traversal order of the transfer functions is not important.
  Each region, including the original (unmasked) volume, is colored using
  the transfer function (if any) that has the same id value.
  See SoVolumeMask for more details.

  It can be convenient to use the same color map for volume rendering
  and for slice rendering. But typically the volume rendering color map
  should have transparency (alpha values less than one) and the slice
  rendering color map should be opaque.  It's not actually necessary
  to create two separate color maps.  Set the *alphaUse* field in the
  slice node to ALPHA_OPAQUE and the slice will ignore the transparency
  values in the color map.

  Applications are not limited to using an SoTransferFunction node to
  specify the data to color mapping. The application can implement its
  own data to color mapping function in GLSL (OpenGL shader language).
  Use an SoVolumeShader node and replace the VVizComputeFragmentColor
  function.  In this case SoTransferFunction can still be used to load
  a color map (or maps) on the GPU, if desired, but the actual assignment
  of a color is done by the application's shader function.

# More about data mapping

  We have discussed above the basic concept of mapping scalar values to
  color values. Here we present more details, which may be important in
  some cases, for example when rendering a label volume.
  The way scalar values are mapped to indices in the color map depends on:
  - The volume data type (SoVolumeData),

  - The data range (SoDataRange),

  - The texture precision (SoVolumeData), and

  - The color map size (SoTransferFunction).

  We have discussed all of these previously except for texture precision.
  The texturePrecision field of SoVolumeData effectively specifies the GPU
  data type, in other words the size (in bits) of the data values stored on
  the GPU. The default is 8 (1 byte), but VolumeViz also supports 16.
  Effectively a volume data value is normalized in the specified data range
  (SoDataRange) then converted to a GPU data value in the range of values
  supported by the GPU data type (default 0..255).  For 8-bit (byte) volume
  data this is a one-to-one mapping.  For larger data types it is
  possible that more than one volume data value will map to the same GPU data
  value (value aliasing). During rendering the GPU data value is normalized
  to the number of entries in the color map then converted to an index into
  the list of colors. For 8-bit (byte) data, if the color map contains 256
  entries, this is also a one-to-one mapping.  But note that the GPU data
  values are usually interpolated before this step (see the interpolation
  field in, for example, SoVolumeRender).

  The data values may also be explicitly rescaled before mapping onto the color
  map, using the #shift and #offset fields.

# Label volumes

  A label volume, also known as a label field, is usually the result of doing
  some sort of segmentation on a data volume.  Each voxel value is an integer
  label (id) identifying which material, object, etc. that the voxel belongs to.
  There could be 100's or 1000's of labels, but there might be as few as 8 label
  values. For example, a simple label volume might have 7 opaque materials plus
  plus an "exterior" material that is completely transparent.
  Conceptually, there is one big difference between a (typical) data volume and
  a label volume.  A data volume is conceptually a set of discrete samples taken
  from a continuous scalar field.  So we know the exact value at the center of
  each voxel and interpolate between those values to get the value at any position
  in between voxels.  In a label volume we normally consider each voxel to belong
  completely to one material, so the value is constant until we cross the boundary
  into the next voxel. Therefore we do not want to interpolate the label values.
  When rendering a label volume with volume rendering, set field SoVolumeShape#interpolation
  to NEAREST and leave the field SoVolumeRenderingQuality#preIntegrated to FALSE.
  If rendering isosurfaces, set the field SoVolumeRenderingQuality#segmentedInterpolation
  to TRUE.

  It is also important to set the data range, texture precision and color map
  size carefully. In particular, take care that the number of values in the data
  range and the number of entries in the color map are exactly equal and a "power
  of 2" (8, 16, 32, ...). For example, if you have 6 label values, you might (logically)
  set the data range to 0..5 (min = 0 and max = 5) and create a label color map with
  6 color values. That will work most of the time, but in a some cases (that are
  difficult to predict), loss of precision in the mapping arithmetic can cause
  a data value to map to the wrong color.  In the previous example, with 6 label values,
  we recommend setting the data range to 0..7 and creating a color map with 8 color values
  (the last two values can be anything since they will not be used).

  If you need more than 256 label values (this is not common, but it is possible),
  follow the above recommendation and also set the texturePrecision field to 16
  (the default is 8). This ensures that each label value maps to a unique GPU data value.
  For example, if you have 1000 labels, we recommond setting the data
  range to 0..1024, setting texturePrecision to 16 and creating a color map with 1024 color
  values.

  Finally, if you're rendering a label volume and you want to "remove" one of the materials
  (i.e. make it invisible), you cannot do this in the usual data volume way by modifying the
  data range in the SoDataRange node or changing the min/max values in the SoTransferFunction
  node.  Doing either of those change will cause label values to map to different colors.
  Instead you should directly modify the values in the color map. In other words, for the color
  entry corresponding to the material id, change the Alpha value to zero.

# Faux shading

  This node also supports a "faux shading" technique that can produce
  results somewhat similar to surface shading without the performance
  penalty for computing lighting.  Faux shading modifies the lower
  portion of the color map so that color and opacity ramp down to zero.
  Faux shading is controlled by the #fauxShadingLength, #fauxShadingStrength,
  and #fauxShadingDarkenThreshold fields.
     @TABLE_1B
        @TR No Faux Shading @TD Faux Shading
        @TR     @IMAGE volume_noedges.jpg
            @TD @IMAGE fauxshading.jpg
     @TABLE_END

  The #actualColorMap field always contains the actual color values (after
  remapping, faux shading, etc have been applied) that will be used for rendering.

# LIMITATIONS

 - Maximum color map size.  
   In the most common case the volume contains scalar data values, scalar values
   are loaded on the GPU and the color map is applied on the GPU.  In this case
   the color map size (number of colors) cannot be larger than the OpenGL maximum
   texture size. (Because the color map is stored in an OpenGL texture.)
   This value is generally 16384 on modern GPUs. Use the OpenGL function
   glGetIntegerv(GL_MAX_TEXTURE_SIZE) to get this value if necessary.

   If it is necessary to use a larger color map, set the field
   SoVolumeData#usePalettedTexture to false. In this case the color map is applied on
   the CPU and RGBA values are loaded on the GPU.  The color map can be any size.
   However rendering may be different because the GPU will interpolate between
   color values rather than data values and modifying the color map will be much
   slower because data must be reloaded on the GPU.

 - Node order:  
   When using an SoVolumeGroup, the SoTransferFunction node must
   be inserted after the SoVolumeData, not before.

 - Shape nodes:  
   This node only affects VolumeViz shape nodes (SoVolumeRender, SoOrthoSlice,
   SoObliqueSlice, SoVolumeSkin, etc).  For Open Inventor nodes see SoColorMap.

 - Multi-volume combining  
   When using SoMultiDataSeparator, all transfer functions must have the same
   'colorMapType' and must have the same number of entries.

 - Old graphics boards:  
   On very old graphics boards, modifying the color map when using
   RGBA (not paletted) textures will usually force the textures to be recreated,
   which may be slow. See the field SoVolumeData#usePalettedTexture.

@FILE_FORMAT_DEFAULT
   TransferFunction {
   @TABLE_FILE_FORMAT
      @TR shift            @TD 0
      @TR offset           @TD 0
      @TR predefColorMap   @TD NONE
      @TR colorMapType     @TD RGBA
      @TR colorMap         @TD 0
      @TR actualColorMap   @TD 0
      @TR transferFunctionId   @TD 0
      @TR minValue @TD 0
      @TR maxValue @TD 255
      @TR fauxShadingStrength @TD 1
      @TR fauxShadingLength @TD 0
      @TR fauxShadingDarkenThreshold @TD 1
   @TABLE_END
   }

@ACTION_BEHAVIOR

SoCallbackAction,
SoGLRenderAction   
     Sets transfer function parameters in the traversal state.

@SEE_ALSO
   SoDataRange,
   SoVolumeRender,
   SoOrthoSlice,
   SoObliqueSlice
 */
class LDM_API SoTransferFunction : public SoNode {

  SO_NODE_HEADER( SoTransferFunction );

  SO_PIMPL_BASE_PUBLIC_HEADER(SoTransferFunction);

 public:
  /**
   * This field allows the use of multiple transfer functions.
   *
   * By default all transfer function nodes are initialized to a transfer function id of 0.
   * If you want to use multiple transfer functions, a different id must be assigned to each
   * transfer function.
   * The transfer function id can be used in shader programs (see SoVolumeShader) to access
   * one of multiple transfer funcions and can also be used to associate a transfer function
   * with a volume mask (see SoVolumeMask).
   * Note that, unlike SoDataSet/SoVolumeData, this value is NOT a GL texture unit number.
   * Therefore it is possible (but not recommended) to use id values starting at 0.  The
   * SoVolumeData dataSetId values should start at 1, so generally it's best to be
   * consistent and use the same value as the data set that this TF corresponds to, which
   * usually means starting the numbering at 1. Note that whatever values you assign, if you
   * create a custom GLSL shader function, you @I must@i use the same values when you call
   * VVizTransferFunction() in your shader function.
   *
   * All active transfer functions are combined into a single 2D texture by "stacking" them
   * in order according to their ids. By default this texture is bound to OpenGL texture
   * unit 0. It is possible to change the texture unit used for transfer
   * functions by setting the preferences variable IVVR_TF_TEX_UNIT (see SoPreferences).
   *
   * Since Open Inventor 8.1, functions are provided which allow convenient access to a
   * specific transfer function in a GLSL fragment shader.  Although direct access
   * is still possible, we strongly recommend using the GLSL convenience functions
   * provided by VolumeViz to ensure compatibility with future rendering features:
   *
   * - vec4 VVizTransferFunction( float pos, int tfId ) @BR
   *   Returns the color at the normalized position @I pos @i in the transfer function @I tfId @i.
   *
   * - vec4 VVizTransferFunction( float frontVoxel, float backVoxel, int tfId ) @BR
   *   Returns the preintegrated color between @I frontVoxel @i and @I backVoxel @i in the transfer
   *  function @I tfId @i.
   *
   * For direct access, the texture @I S @i coordinate specifies the entry in the color map
   * and the texture @I T @i coordinate specifies which color map to use.
   * If N is the number of color maps used, the rules are:
   *
   *   - Like all texture coordinates, the @I s @i and @I t @i coordinates range from 0 to 1.
   *   - The transferFunctionId ranges from 0 to N-1.
   *   - To access a specific transfer function, the @I t @i value passed must be between
   *     [ transferFunctionId / N, ( transferFunctionId + 1 ) / N ]
   *
   * @FIELD_SINCE_OIV 6.0
   */
  SoSFInt32 transferFunctionId;

  /**
   * Used for rescaling the input data values.
   * Before color association, the data value is re-scaled as follows:
   * \verbatim
          newvalue = value << #shift + #offset
     \endverbatim
   * By default, #shift = #offset = 0.
   */
   SoSFInt32 shift;

  /**
   * Used for rescaling the input data values.
   * See #shift.
   */
  SoSFInt32 offset;

  /**
  * Remaps the defined color map between @B minValue @b and @B maxValue @b indices. Entries
  * less than @B minValue @b and greater than @B maxValue @b are set to full transparent
  * black (all components set to zero).
  *
  * Setting minValue to 64 and maxValue to 192 while the predefined STANDARD color map is selected
  * makes the color map as shown:
  *
  * @IMAGE standardReMap.jpg
  *
  * @FIELD_SINCE_OIV 7.0
  */
  SoSFInt32 minValue;

  /**
   * See #minValue field.
   * The value is automatically set to the number of colors in colorMap
   * minus 1 as long an explicit value is not specified.
   *
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFInt32 maxValue;

  /**
   * Controls how many color values will be affected by faux shading.
   *
   * Valid values are between 0.0 and 1.0 (in other words a percentage of
   * the color map).  0.0 (the default) means no faux shading.
   * Faux shading is applied between minValue and minValue + ( maxValue - minValue ) * fauxShadingLength.
   * Faux shading darkens the color and reduces the opacity of the affected color map entries.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat fauxShadingLength;

  /**
   * Controls how much faux shading will darken
   * the color values of the transfer function.
   *
   * Any value greater than 0.0 is valid.
   * 1.0 (the default) means a linear darkening progression.
   * Values greater than 1.0 darken the color map faster.
   * Values less than 1.0 darken the color map more slowly.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat fauxShadingStrength;

 /**
   * Opacity threshold for darkening edges.
   *
   * Valid values are between 0.0 and 1.0.  The default is 1.0.
   * Only colors with opacity lower than
   * fauxShadingDarkenThreshold will be darkened.
   * Use this to prevent the volume from being globally darkened.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat fauxShadingDarkenThreshold;

  /**
   * Predefined color maps. See the #predefColorMap field for images.
   */
  enum PredefColorMap {
    /** No color map */
    NONE,
    /** Grey (Default). This is constant intensity (white) with a 0..1 alpha ramp.
     *                  A good initial color map for volume rendering. */
    GREY,
    /** Gray (Synonym of grey) */
    GRAY = GREY,
    /** Temperature (opaque). */
    TEMPERATURE,
    /** Physics (opaque). */
    PHYSICS,
    /** Standard (opaque). */
    STANDARD,
    /** Glow (opaque). This is similar to "hot iron" type color maps. */
    GLOW,
    /** Blue red (opaque). */
    BLUE_RED,
    /** Seismic */
    SEISMIC,
    /** Blue white red (opaque). */
    BLUE_WHITE_RED,
    /** Intensity (opaque). This is an intensity ramp (gray scale) with constant alpha (1).
     *                      A good initial color map for slice rendering.
     *                      Appropriate for DICOM Photometric Interpretation mode 'MONOCHROME2' */
    INTENSITY,
    /** 256 labels (opaque). A good color map for rendering label volumes. */
    LABEL_256,
    /** VolRenRed (opacity ramp). Suitable for volume rendering. **/
    VOLREN_RED,
    /** VolRenGreen (opacity ramp). Suitable for volume rendering. **/
    VOLREN_GREEN,
    /** Airway.
     * This colormap is especially adapted to display airways of CT Scan datasets.
     * Ranges of data may have to be adapted depending on your acquisition device's calibration.
     * Please refer to SoDataRange node for details. */
    AIRWAY,
    /** Airway surfaces.
     * This colormap is especially adapted to display airways and soft tissues of CT Scan datasets.
     * Ranges of data may have to be adapted depending on your acquisition device's calibration.
     * Please refer to SoDataRange node for details. */
    AIRWAY_SURFACES,
    /**
     * Intensity reversed (opaque). This is a reversed intensity ramp (gray scale) with constant alpha (1).
     *                      Appropriate for DICOM Photometric Interpretation mode 'MONOCHROME1' */
     INTENSITY_REVERSED
  };

  /**
   * Predefined color map to use.
   * @useenum{PredefColorMap}. Default is NONE (GRAY will be used if no values are set in the #colorMap field).
   * All predefined color maps have 256 entries.
   *
   *    @TABLE_0B
   *       @TR GREY or GRAY       @TD @IMAGE grayColorMap.jpg
   *       @TR TEMPERATURE        @TD @IMAGE temperatureColorMap.jpg
   *       @TR PHYSICS            @TD @IMAGE physicsColorMap.jpg
   *       @TR STANDARD           @TD @IMAGE standardColorMap.jpg
   *       @TR GLOW               @TD @IMAGE glowColorMap.jpg
   *       @TR BLUE_RED           @TD @IMAGE blue_redcolormap.jpg
   *       @TR SEISMIC            @TD @IMAGE seismicColorMap.jpg
   *       @TR BLUE_WHITE_RED     @TD @IMAGE blue_white_redcolormap.jpg
   *       @TR INTENSITY          @TD @IMAGE intensityColorMap.jpg
   *       @TR INTENSITY_REVERSED @TD @IMAGE intensityReversedColorMap.jpg
   *       @TR LABEL_256          @TD @IMAGE label_256ColorMap.jpg
   *       @TR VOLREN_RED         @TD @IMAGE volrenRedColorMap.jpg
   *       @TR VOLREN_GREEN       @TD @IMAGE volrenGreenColorMap.jpg
   *       @TR AIRWAY             @TD @IMAGE airwayColorMap.jpg
   *       @TR AIRWAY_SURFACES    @TD @IMAGE airwaySurfaceColorMap.jpg
   *    @TABLE_END
   *
   * NOTE: The checkerboard pattern shows through where the color map alpha (opacity)
   * value is less than 1.0.
   *
   * If this field is explicitly set to NONE, then the #colorMap and #colorMapType fields should be set.
   */
  SoSFEnum predefColorMap;

  /** Available color map type */
  enum ColorMapType {
    /** (1 floats) */
    ALPHA,
    /** (2 floats) */
    LUM_ALPHA,
    /** (4 floats) */
    RGBA
  };

  /**
   * ColorMap type (number of color components).
   * @useenum{ColorMapType}. Default is RGBA.
   *
   * ALPHA means one alpha component per color.
   *
   * LUM_ALPHA means two components per color, luminance and alpha. In such a case the
   * colorMap float array contains a list of two floats. Index 0 is luminance, index
   * 1 is alpha, index 2 is luminance, index 3 is alpha and so on.
   *
   * RGBA means four components per color, ordered red, green, blue, then alpha.
   *
   * An alpha value equal to zero means "fully transparent", an alpha value equal to
   * one means "completely opaque" (exactly the inverse of transparency in SoMaterial).
   */
  SoSFEnum  colorMapType;

  /**
   * Array of floats in the range [0,1] defining a color map.
   * The number of floats needed depends on #colorMapType.
   * It is equal to the number of colors defined multiplied by the number of
   * components per color. For example, for an RGBA color map of length n,
   * there should be 4*n float values in the field.
   *
   * Note that changing a value in the colorMap field automatically updates the #actualColorMap field.
   * Thus it can be very inefficient to set the color map entries one at a time.
   * To avoid a performance decrease we recommend using one of the
   * following techniques, particularly when changing many values:
   *
   * \if_cpp
   * - Use the setValues() or setValuesPointer() method.
   * \endif
   * \if_dotnet
   * - Use the SetValues() or SetValuesBuffer() method.
   * \endif
   * \if_java
   * - Use the setValues() or setValuesBuffer() method.
   * \endif
   *
   * - Use the start/finish editing methods.
   *
   * See the examples in the class description.
   */
  SoMFFloat colorMap;

  /**
   * This field contains the actual color map used.
   * It is set automatically according to the value of the other fields.
   * The method reMap() modifies this field as well.
   * You may modify this field, and your changes will be taken into account during rendering.
   * However these changes will be lost if the other fields are changed.
   *
   * @FIELD_SINCE_OIV 6.0
   */
  SoMFFloat actualColorMap;

  /**
   * Constructor
   */
  SoTransferFunction();

  /**
   * Returns TRUE if the current color map contains alpha values less than 1.
   * FALSE means the color map is completely opaque.
   */
  SbBool hasTransparency() const;

  /**
   * Loads a colormap from a file.
   * Returns true if successful.  The specified file name must have an
   * extension of either ".am" or ".col".
   *
   * Note: This method loads RGBA values into the colorMap field, but
   * the application must still set the predefColorMap field to NONE
   * for the custom colormap values to be used.
   * 
   * Supported formats:
   * - .am  : Avizo Colormap v1.0
   * - .col : Avizo Colormap v2.0
   *
   */
  virtual SbBool loadColormap(const SbString& filename);

 SoEXTENDER public:
  virtual void doAction( SoAction *action );
  virtual void callback( SoCallbackAction *action );
  virtual void GLRender( SoGLRenderAction *action );
  virtual void pick(SoPickAction *action) ;
  virtual void write(SoWriteAction *action);

  //------------------------------------------------------------------------------
 SoINTERNAL public:

  /**
   *  Define how to map data range values across the colormap
   */
  enum MappingMethod 
  {
    /**
     * First dataRange value will be mapped to the center of first colormap texel.
     * Last dataRange value will be mapped to the center of last colormap texel.
     * Other values are mapped linearly. 
     *
     * @IMAGE map_to_center.png
     */
    MAP_TO_CENTER = SoConversion::MAP_TO_CENTER,

    /**
     * First dataRange value will be mapped to the left of first colormap texel.
     * Last dataRange value will be mapped to the right of last colormap texel.
     * Other values are mapped linearly. 
     *
     * @IMAGE map_to_border.png
     */
    MAP_TO_BORDER = SoConversion::MAP_TO_BORDER
  };

  /**
   * This field precise how to map color across the colormap.
   * Please refer to #MappingMethod for details.
   *
   * @Note: workaround for bug #30927. Not exposed for now.
   */
  SoSFEnum mappingMethod;

  typedef float OpacityCorrectionFunction(float alpha, void* userData);

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  /**
   * Returns TRUE, to trigger TraversalPass update when TrnasferFunction
   * changes in order to manage tf that goes from transparent to non-trnasparent.
   */
  virtual bool affectsPath() const;

  static void initClass();
  static void exitClass();

  int getPackedColorMap(unsigned int * &, int);
  static unsigned int  getDefaultPackedColor(unsigned int);
  enum GLColorMapType {
    NO_COLORMAP = 0,
    TEX_FRAGMENT_PROGRAM = 2
  };

  GLColorMapType installColorMap(SoState*, int alphaUse);

  /**
   * Apply f on all alpha values before sending the colormap to GL
   */
  GLColorMapType installColorMap(SoState*, int, OpacityCorrectionFunction* f, void* userData);
  int m_colorMapMin;
  int m_colorMapMax;

  /**
   * Do opacity correction on the buffer allRgba of size numRgba
   */
  void applyOpacityCorrection(unsigned int* allRgba, size_t numRgba,
                              OpacityCorrectionFunction* f, void* userData);

  /**
   * Returns TRUE if the colormap is fully transparent between the given min max values.
   */
  SbBool isFullyTransparent(int64_t min, int64_t max, int alphaUse = 0);

  //for cluster command, list of method to sync
  enum Cmd{
    REMAP = 0
  };

  void  getCMBuffer( int alphaUse, int nrgba, unsigned int *rgba );

  /**
   * Return true if all value between min and max are transparent.
   * min and max are in [0, 1].
   */
  bool isTransparent(const float min, const float max, const float materialTransparency) const;

  /**
   * Return the palette texture
   */
  SoRef<SoTexture> getTexture() const;

  /**
   * Resture the texture size
   */
  SbVec2i32 getTextureSize() const;

  /**
   * Return the number of color in thetransfer function.
   * A color is a alpha value, a alpha-lum pair or a RGBA quad according to colorMapType
   */
  unsigned int getNumColors() const;

  /**
   * True if there is only 0/1 transparency
   */
  bool isBinary() const;

  /**
   * Fill and return texture with a set of pre-integrated transfer function on each depth
   */
  static void createPreIntegratedTexture(SoState* state, SoGroup* placeHolder);

  /**
   * Return the number of component according to colormapType
   */
  size_t getNumComponents() const;

  /**
   * Return position of the transfer function in the texture
   * Will be a row number for 2D tf, a depth number for 3D
   */
  int getTransferFunctionPosInTexture(int maxTfId);

private :

  // Function to factorise constructors code
  void construct();

  friend class SoVolumeState;
};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif // _SO_TRANSFER_FUNCTION_


