// DICOM annotation utility class

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _DICOM_INFO_H_
#define _DICOM_INFO_H_

#include <Medical/InventorMedical.h>
#include <Medical/nodes/TextBox.h>

#include <Inventor/SbString.h>
#include <Inventor/fields/SoSFFilePathString.h>

class SoVRDicomData;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Text node to display DICOM information on sceen.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node is a convenience class derived from TextBox for displaying
 *   DICOM annotation text on screen.
 *
 *   This node displays a text box at a fixed location in the viewer window.
 *   The position is specified in normalized device coordinates -1 to 1.
 *   The DICOM file is specified using the 'fileName' field.
 *
 *   Alignment options allow the box to be positioned, for example, in a
 *   corner of the window. For example position -0.98, -0.98, 0 with default
 *   alignment is the upper left corner of the window.
 *   The text box automatically expands or shrinks when lines of text are
 *   added to or deleted from the box.  Positioning a text box in the lower
 *   left corner creates a sort of 'console' output overlaying the scene.
 *   Note that the alignment options control the positioning of the text box.
 *   The text inside the box is currently always left justified.
 *
 *   In order to have convenient default values for font size, the font
 *   settings are @I not@i inherited from the scene graph. By default the text
 *   is rendered using an SoText2 node with font name 'Arial', font size 16 and
 *   line spacing 1.1. The application can modify these values using the
 *   getFontNode() and getTextNode() methods. Lighting and picking are disabled.
 *   Text will be rendered on top of whatever is rendered in the main scene graph.
 *   The application can modify the text strings directly, but this class also
 *   provides some convenience methods that are very useful. For example, the
 *   addLine() method appends a new string to the end of the list.
 *
 * @FILE_FORMAT_DEFAULT
 *    DicomInfo {
 *    @TABLE_FILE_FORMAT
 *      @TR fileName      @TD ""
 *      @TR position      @TD 0 0 0
 *      @TR alignmentH    @TD LEFT
 *      @TR alignmentV    @TD TOP
 *      @TR textAlignH    @TD LEFT
 *      @TR fontName      @TD Arial:Bold
 *      @TR fontSize      @TD 15
 *      @TR border        @TD FALSE
 *      @TR borderColor   @TD 1 1 1
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, TextBox, Gnomon, Magnifier, Ruler
 * 
 * @PREVIEWFEATURES
 */ 

class INVENTORMEDICAL_API DicomInfo : public TextBox {

  SO_NODE_HEADER(DicomInfo);

public:
  /**
  * Path to the DICOM file used to extract DICOM info.
  * Default is empty.
  */
  SoSFFilePathString  fileName;

  /**
  * Add a line to the DICOM annotation.
  * - title: a string to prefix the DICOM info
  * - group / element : tag of requested DICOM info
  *
  * Returns : index of the next insertion
  */
  int displayDicomInfo( SbString title, unsigned short group, unsigned short element );

  /**
  * Add a line to the DICOM annotation
  * - title: a string to prefix the DICOM info
  * - infoString : a string to define miscellaneous info
  *
  * Returns : index of the next insertion
  */
  int displayDicomInfo( SbString title, SbString infoString );

  /**
  * Returns the string representing the specified DICOM tag (empty if not present).
  */
  SbString getDicomInfo(unsigned short group, unsigned short element);

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

  /** Constructor. */
  DicomInfo();

protected:
  /** Destructor */
  virtual ~DicomInfo();

  SbString       m_filename;
  SoVRDicomData* m_dicomReader;

  bool checkFilename();
};

#endif