///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _MEDICAL_GNOMON_H
#define _MEDICAL_GNOMON_H

#include <Medical/InventorMedical.h>

#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFVec2i32.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSwitch.h>

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node to display a medical gnomon (compass) on the screen.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node displays a medical specific "gnomon" (compass) on the screen
 *   that shows the user the current orientation of the volume in 3D space.
 *
 *   Visibility of the gnomon is controlled by the #isDisplayed field.
 *
 *   Position and size of the gnomon are specified in pixels using the
 *   #position, #width and #height fields.
 *
 * @FILE_FORMAT_DEFAULT
 *    Gnomon {
 *    @TABLE_FILE_FORMAT
 *      @TR isDisplayed     @TD TRUE
 *      @TR position        @TD 0 0
 *      @TR width           @TD 100
 *      @TR height          @TD 100
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, DicomInfo, Magnifier, Ruler
 * 
 * @PREVIEWFEATURES
 */ 

class INVENTORMEDICAL_API Gnomon : public SoAnnotation {

  SO_NODE_HEADER(Gnomon);

public:

  /** Controls whether the gnomon is visible (default is true). */
  SoSFBool  isDisplayed;

  /** Position of the gnomon viewport in points (default is 0,0). */
  SoSFVec2i32 position;

  /** Specifies the width of the gnomon viewport in points (default is 100). */
  SoSFInt32 width;

  /** Specifies the height of the gnomon viewport in points (default is 100). */
  SoSFInt32 height;

  /** Constructor. */
  Gnomon();

  /** Initialize class (called automatically by InventorMedical::init()). */
  static void initClass();

  /** Finish using class (called automatically by InventorMedical::finish()). */
  static void exitClass();

  /** Scale the size of the gnomon (default is 1,1,1). */
  void setScale( SbVec3f scale ) { m_gnomonScale->scaleFactor.setValue( scale ); };

  /** Replace the gnomon geometry (default is basic medical gnomon).
   *  Gnomon geometry should have a 3D extent of -1 to 1.
   */
  void setGeometry( SoNode* geometrySceneGraph );

SoINTERNAL public:

  /** Specifies the distance from the camera to the gnomon geometry.
   *  Default is 3.5.  Applications should not need to modify this value. */
  SoSFFloat cameraDistance;

  /** Returns the internal gnomon camera (used internally in callback). */
  SoOrthographicCamera* getCamera();

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

protected:
  virtual ~Gnomon();
  
  void commonConstructor();

  /** Build the SceneGraph representing the Gnomon */
  void buildGnomon();

  // Managed isDisplayed Field
  void updateGnomonVisibility();

  SoRef<SoCallback>           m_callback;
  SoRef<SoOrthographicCamera> m_camera;
  SoRef<SoSwitch>             m_displaySwitch;
  SoRef<SoScale>              m_gnomonScale;
  SoRef<SoSeparator>          m_geometry;
};

#endif
