// Define a view (within a render area)

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _SCENE_VIEW_H_
#define _SCENE_VIEW_H_

// Note: Full header files are required to use SoRef on member variables.
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoViewport.h>

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFVec2f.h>
#include <Inventor/SbViewportRegion.h>

#include <Medical/InventorMedical.h>

class SoAction;
class SoNodeSensor;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Defines a view for use with ViewManager.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *  This class defines a "view" consisting of a viewport (subregion) inside
 *  the render area, a scene graph to be rendered and an optional border.
 *
 *  This class is conceptually similar to the older class PoSceneView, but
 *  is designed to be used with the Viewer Component classes, specifically
 *  SceneExaminer, and with the ViewManager class.
 *
 *  Usually the specified scene graph will contain its own camera node and one
 *  or more light nodes.  If the user should be able to manipulate the camera,
 *  we recommend adding a SceneExaminer node (which automatically creates a
 *  camera, headlight and event handlers) and then setting the actual scene to be
 *  rendered as a child of the SceneExaminer.
 *
 *  Normally the SceneView node(s) should be added as children of a ViewManager
 *  node.  This is not required if no nodes in the view need to handle events,
 *  for example if the view is an orientation indicator or bird's eye view and
 *  its camera is linked to a main camera. However the ViewManager node is needed
 *  to deliver events to its SceneView children based on the position of the event
 *  and the order of the SceneView children.  The SoHandleEventAction will only
 *  be allowed to traverse a SceneView node's scene graph if the event position is
 *  inside that SceneView's viewport or the SceneView has "captured" events.
 *  See ViewManager for more details.
 *
 *  Note that the specified viewport is @I relative@i to the node's inherited viewport.
 *  For the "first level" of SceneView nodes the viewport is relative to the entire
 *  render area, as expected. However this feature allows the powerful capability to
 *  "nest" SceneView nodes.  If a SceneView node is the child of another SceneView
 *  node, the child's viewport is relative to the viewport set by the parent node.
 *  For example, one quadrant of the render area could be subdivided into multiple
 *  smaller viewports to create a "light box" view.
 *  The viewport definition may be dynamically changed to create a different
 *  layout of views.  For example four equal size viewports can be dynamically
 *  changed to one large viewport plus three small viewports.  The convenience
 *  methods in the ViewManager node make it easy to modify the view layout.
 *
 *  The #active field can be used to prevent a view from being rendered.  This
 *  is convenient, for example, when switching from a four view layout to a one
 *  or two view layout.  In this case it is not necessary to remove the inactive
 *  SceneView nodes from their parent ViewManager, just set 'active' to false.
 *
 *  Data nodes, e.g. SoVolumeData, that are shared by multiple views may be
 *  placed in the scene graph @I above@i the ViewManager node.  But note that
 *  in most cases Open Inventor cannot compute the correct bounding box without
 *  traversing the data node.  This means that calling viewAll() on the SceneExaminer
 *  inside a view will compute the correct view volume.  You can work around this
 *  by placing an SoBBox node under the SceneView and setting its 'boundingBox'
 *  field to (for example) the extent of the SoVolumeData node.
 *  Alternatively you can create one instance of the data node and add that
 *  instance under the SceneExaminer in each view that needs that data. In this
 *  case the viewAll() should work as expected.  Do not create a separate instance
 *  of the data node for each view as this will cause the data to be loaded multiple
 *  times.
 *
 *  Tiling: 
 *  Even when working in normalized coordinates it is necessary to take some care
 *  about the size of the viewports in pixels.  For example, if the render area
 *  is 1024 pixels wide, then creating two tiles (viewports) with width 0.5 will
 *  result in two non-overlapping viewports, each 512 pixels wide as expected.
 *  But 1024 is not evenly divisible by 3, so creating three tiles with width 0.3333
 *  means that each tile will be 341 pixels wide and 3 * 341 = 1023, leaving a
 *  1 pixel gap.  In this case it is necessary to make one of the viewports 1 pixel
 *  larger than the others. The width of 1 pixel in normalized coordinates is
 *  simply 1 / widthOfParentViewport (in pixels).
 *
 *  Limitations:
 *  - To define its subregion, SceneView uses a SoViewport node that simply
 *    overwrites the value of the SoViewportRegionElement, but does not manage
 *    SoLogicalViewportElement or SoModifyViewVolumeElement. This means that
 *    SceneView is not compatible with very large offscreen rendering that
 *    creates viewport "tiles" at a higher level using the previously
 *    mentioned elements. See more details about this limitation in
 *    SoOffscreenRenderArea.
 *
 * @FILE_FORMAT_DEFAULT
 *    SceneView {
 *    @TABLE_FILE_FORMAT
 *      @TR viewportOrigin  @TD 0 0
 *      @TR viewportSize    @TD 1 1
 *      @TR scene           @TD NULL
 *      @TR active          @TD TRUE
 *      @TR drawBorder      @TD TRUE
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  ViewManager,
 *  SceneExaminer
 *
 * @PREVIEWFEATURES 
 */ 

class INVENTORMEDICAL_API SceneView : public SoSeparator {

  SO_NODE_HEADER(SceneView);

public:

  /** Viewport origin (normalized device coordinates 0..1).
   *  Default is 0,0.
   *  Illegal values, e.g. < 0 or > 1, are clamped to 0..1.
   */
  SoSFVec2f viewportOrigin;

  /** Viewport size (normalized device coordinates 0..1).
   *  Default is 1,1.
   *  Illegal values, e.g. < 0 or > 1, are clamped to 0..1.
   */
  SoSFVec2f viewportSize;

  /** Enable rendering the view. Default is true. */
  SoSFBool active;

  /** Enable drawing a border around the view's viewport.
   *  Default is true.
   *  - The border is 1 pixel wide and uses the outermost pixels of the viewport,
   *    effectively reducing the size of the viewport by two pixels in each direction.
   *  - If two adjacent views have their border enabled the result will usually be
   *    a two pixel wide line (1 pixel in each view).  However, see the discussion
   *    about "tiling" above. If the pixel size of the views are not taken into
   *    account then border lines may overlap or show a one pixel gap.
   */
  SoSFBool drawBorder;


  /** Convenience method to set viewport in normalized coordinates with one call.
   *  Modifies the 'viewportOrig' and 'viewportSize' fields. */
  void setViewport( float origX, float origY, float sizeX, float sizeY );

  /** Convenience method to set viewport in normalized coordinates with one call.
   *  Modifies the 'viewportOrig' and 'viewportSize' fields. */
  void setViewport( const SbVec4f& viewport );

  /** Returns the current viewport as a convenient single object. */
  const SbVec4f& getViewport() const;

  /** Returns the actual viewport in pixels.
   *  Convenient to use with (for example) viewAll(). */
  const SbViewportRegion& getViewportRegion() const;

  /** Returns the actual viewport in pixels. */
  const SbVec4f& getViewportPixels() const;

  /** Enable handling events.
   *  Default is true. This method is used by the ViewManager node to temporarily
   *  disable event handling without triggering notification.  Generally the
   *  application should not call this method directly. */
  void enableEventHandling( SbBool onoff );

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

  /** Constructor. */
  SceneView();

 SoEXTENDER public:

  /** Override generic child methods so we can manage our internal scene graph. */
  virtual void addChild( SoNode* child );
  virtual int  findChild( const SoNode* child ) const;
  virtual SoNode* getChild( int index ) const;
  virtual void insertChild( SoNode* child, int newChildIndex );
  virtual void removeChild( SoNode* child );
  virtual void removeChild( int index );
  virtual void removeAllChildren();
  virtual void replaceChild( SoNode* oldChild, SoNode* newChild );
  virtual void replaceChild( int index, SoNode* newChild );
  virtual int getNumChildren() const;

protected:
  /** Destructor */
  virtual ~SceneView();

  // Internal helpers
  void viewportFieldChanged();
  void viewportParentChanged( const SbViewportRegion& vpregion );
  void updateOurViewport();
  static void callback(void* userData, SoAction* action);

  // Internal state
  SbVec4f            m_curViewport;     // Current viewport in normalized coords
  SbViewportRegion   m_vportRegion;     // Used to answer queries

  SbVec4f            m_viewport;        // What we actually set (pixels) as float
  SbVec4i32          m_viewportPixels;  // What we actually set (pixels) as int
  SbVec2i32          m_viewportOrig;    // What we actually set (pixels)
  SbVec2i32          m_viewportSize;    // What we actually set (pixels)
  
  SbVec2i32          m_parentOrig;      // What we inherited (pixels)
  SbVec2i32          m_parentSize;      // What we inherited (pixels)

  SoRef<SoSwitch>    m_masterSwitch;    // Internal scene graph
  SoRef<SoViewport>  m_vportNode;
  SoRef<SoSeparator> m_sceneNode;
  SoRef<SoSwitch>    m_borderSwitch;
  SoRef<SoLineSet>   m_borderLine;

  bool               m_handleEvents;    // See enableEventHandling() method.

  SoNodeSensor*  m_sensor;
  bool m_ignoreSensor;                  // Used to temporarily disable sensor
  static void sensorCB( void* data, SoSensor* sensor );
};

#endif