// Slice orientation markers utility class

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _SLICE_ORIENTATION_MARKERS_H_
#define _SLICE_ORIENTATION_MARKERS_H_

#include <Inventor/nodes/SoAnnotation.h>

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFString.h>

class SoFont;
class SoFieldSensor;
class SoText2;
class SoTranslation;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node to display slice orientation markers in window coordinates.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node displays slice orientation markers at a fixed location in the viewer window.
 *   Slice orientation markers are useful when viewing images in a "2D" mode.
 *   The orientation markers are a set of four letters.  Each letter is displayed
 *   at the mid-point of one edge of the window and offset inward by the value
 *   specified in the #offset field.  The letters displayed depend on the value
 *   of the #axis field and correspond to the DICOM radiological viewing convention.
 *   This node is intended to be used with SoOrthoSlice and a "plane" viewer.
 *
 *   The application can connect the #axis field from the 'axis' field of the
 *   SoOrthoSlice node and the correct orientation markers will be displayed
 *   automatically if the slice axis is changed.
 *
 *   For:
 *
 *   - AXIAL display
 *     - Left/Right: R(ight) / L(eft)
 *     - Top/Bottom: A(nterior) / P(osterior)
 *
 *   - CORONAL display
 *     - Left/Right: R(ight) / L(eft)
 *     - Top/Bottom: S(uperior) / I(nferior)
 *
 *   - SAGITTAL display
 *     - Left/Right: A(nterior) / P(osterior)
 *     - Top/Bottom: S(uperior) / I(nferior)
 *
 * @FILE_FORMAT_DEFAULT
 *    SliceOrientationMarkers {
 *    @TABLE_FILE_FORMAT
 *      @TR offset      @TD 0.05
 *      @TR axis        @TD AXIAL
 *      @TR fontName    @TD Arial:Bold
 *      @TR fontSize    @TD 17
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical,
 *  DicomInfo,
 *  Gnomon,
 *  Magnifier,
 *  Ruler,
 *  SliceScaleBar,
 *  TextBox
 * @PREVIEWFEATURES 
 */ 

class INVENTORMEDICAL_API SliceOrientationMarkers : public SoAnnotation {

  SO_NODE_HEADER(SliceOrientationMarkers);

public:
  /** Axis (default is AXIAL).
    * Use enum MedicalHelper::Axis. */
  SoSFEnum  axis;

  /** Distance from edge of window to letter.
   *  Specified in normalized screen coordinates (-1 to 1). Default is 0.05. */
  SoSFFloat offset;

  /** Specify the font name (default is "Arial:Bold").
   *  See SoFont::name for details. */
  SoSFString fontName;

  /** Specify the font size in pixels (default is 17). */
  SoSFFloat fontSize;

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

  /** Constructor. */
  SliceOrientationMarkers();

protected:
  /** Destructor */
  virtual ~SliceOrientationMarkers();

  /** Update letters when axis changes. */
  void updateLetters();

  SoRef<SoFont>  m_fontNode;

  SoText2*       m_textNodes[4];
  SoTranslation* m_tranNodes[4];

  SoFieldSensor* m_axisSensor;
  SoFieldSensor* m_offsetSensor;
  static void fieldSensorCB( void* data, SoSensor* sensor );
};

#endif
