////////////////////////////////////////////////////////////////////////
//
// SliceScaleBar utility class
//
// Mike Heck, VSG Inc, December 2011
//
////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _SLICE_SCALE_BAR_H
#define _SLICE_SCALE_BAR_H

#include <Inventor/nodes/SoAnnotation.h>

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFVec2f.h>

class SoAction;
class SoDrawStyle;
class SoFont;
class SoLineSet;
class SoMaterial;
class SoSeparator;
class SoText2;
class SoTranslation;
class SoVertexProperty;
class SoNodeSensor;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node to display a dynamic scale bar in window coordinates.
 * 
 * @ingroup MedicalNodes
 * 
 * @DESCRIPTION
 * This class displays a 2D scale bar.
 * The #position of the scale bar is specified in normalized screen
 * coordinates (-1 to 1). By default, the #length of the scale bar is a fixed
 * distance in normalized screen coordinates.  However if the #trackedCamera
 * field is set, then #length is a distance in 3D world coordinates and the
 * the size of the scale bar on screen is computed based on that distance.
 * For example, if the application is viewing DICOM data measured in
 * millimeters (mm), then setting #length to 100 displays a scale bar that
 * shows the distance 10 cm on screen.  The scale bar size will change if the
 * camera is zoomed in or out (camera height field changes).
 *
 * The distance computation is based on "horizontal" or "vertical" relative
 * to the camera, so the scale bar adjusts automatically if the camera is
 * rotated to view a different volume axis, e.g. Coronal vs Axial.
 *
 * This class is not intended to replace the MeshViz axis classes for general
 * 2D data plotting.  This class has only been tested with SoOrthoSlice
 * and an SoOrthographicCamera (the usual case for medical image viewing).
 *
 * @FILE_FORMAT_DEFAULT
 *    SliceScaleBar {
 *    @TABLE_FILE_FORMAT
 *      @TR position         @TD 0 0 0
 *      @TR length           @TD 1
 *      @TR numTickIntervals @TD 0
 *      @TR trackedCamera    @TD null
 *      @TR orientation      @TD HORIZONTAL
 *      @TR alignment        @TD CENTER
 *      @TR label            @TD ""
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, DicomInfo, Gnomon, Magnifier, Ruler, SliceOrientationMarkers
 *
 * @PREVIEWFEATURES
 */

class INVENTORMEDICAL_API SliceScaleBar : public SoAnnotation {

  SO_NODE_HEADER(SliceScaleBar);

public:

  /** Position in normalized screen coordinates (-1 to 1). Default is (0,0,0). */
  SoSFVec2f position;

  /** Length in normalized screen coordinates (-1 to 1) if not tracking,
   *  else length in 3D world coordinates. Default is 1.
   *
   *  For example, if the application is viewing DICOM data measured in
   *  millimeters (mm), then setting length to 100 displays a 10 cm scale
   *  bar on screen.
   */
  SoSFFloat length;

  /** Number of tick intervals (default is 0).
   *  If numTickIntervals is 0, no tick marks are drawn.
   *  For example, if the length is set to 100 mm, then set numTickIntervals
   *  to 10 to get a tick mark every 10 mm (1 cm).
   */
  SoSFInt32 numTickIntervals;

  /** Tracked camera (default is null).
   *  This should be the camera that is viewing the tracked scene.
   *  It will be used to determine the length of the axis in NDC
   *  based on the specified length in 3D world coordinates. 
   */
  SoSFNode trackedCamera;

  /** Scale bar orientation (default is HORIZONTAL). */
  SoSFEnum orientation;

  /** Scale bar alignment (default is CENTER). */
  SoSFEnum alignment;

  /** Label (default is empty string). */
  SoSFString label;

  /** Scale bar orientation. */
  enum Orientation {
    /** Horizontal */
    HORIZONTAL = 0,
    /** Vertical */
    VERTICAL = 1
  };

  /** Scale bar alignment. */
  enum Alignment {
    /** Left (for horizontal orientation). */
    LEFT = 0,
    /** Bottom (for vertical orientation). */
    BOTTOM = 0,
    /** Center (for either orientation). */
    CENTER = 1,
    /** Right (for horizontal orientation). */
    RIGHT = 2,
    /** Top (for vertical orientation). */
    TOP = 2
  };

  /** Constructor. */
  SliceScaleBar();

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

protected:
  /** Destructor. */
  virtual ~SliceScaleBar();

  /** Build internal scene graph. */
  void buildSceneGraph();

  /** Compute end-points of scale bar. */
  void computeEndPoints( SbVec3f& p0, SbVec3f& p1 );

  SoRef<SoSeparator>      m_lineSep;
  SoRef<SoLineSet>        m_axisLineSet;
  SoRef<SoVertexProperty> m_vertProp;

  SoRef<SoSeparator>      m_textSep;
  SoRef<SoTranslation>    m_labelPos;
  SoRef<SoFont>           m_labelFont;
  SoRef<SoText2>          m_labelText;

  // TODO: Add title and labels
  //SoRef<SoFont>           m_fontNode;

  bool  m_fieldsChanged; // Set when fields have changed
  float m_ndcLength;     // Same as length field if not tracked
  SbVec3f m_p0;          // Current end-points in NDC
  SbVec3f m_p1;          //   (they depend on orientation and alignment)

  int       m_tickLenPix;  // Target tick line length in pixels
  float     m_tickLenNdc;  // Tick line length in NDC (depends on viewport)
  SbVec2i32 m_winSizePix;  // Window size in pixels
  SbVec2f   m_pixelPerNdc; // Converts 1 ndc unit to pixels (but not a coordinate)

  SoNodeSensor* m_sensor; // Monitors changes to our fields
  static void sensorCB( void* data, SoSensor* sensor ); 
  
  static void staticCB( void* data, SoAction* action ); // Called during traversal
  void renderCB( SoAction* action );

  void resetLines();                        // Remove all lines
  void addLine( SbVec3f& p0, SbVec3f& p1 ); // Add a new line (e.g. tick)

  void updateAxis(); // Recreate/reposition axis geometry
};

#endif