// TextBox utility class

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _TEXT_BOX_H_
#define _TEXT_BOX_H_

#include <Medical/InventorMedical.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoTextProperty.h>

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/SbViewportRegion.h>

// Note we need the actual header files to use SoRef (not just just a 'class' declaration).
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>

class SoNodeSensor;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node to display a text box in window coordinates.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node displays a text box at a fixed location in the viewer window.
 *
 *   The #position is specified in normalized device coordinates -1 to 1.
 *   Horizontal #alignmentH and vertical #alignmentV can be specified.
 *   For example position -0.98, -0.98, 0 with TOP/LEFT alignment (the
 *   default) puts a text box in the upper left corner of the window.
 *   The text box automatically expands or shrinks when lines of text are
 *   added to or deleted from the box.  Positioning a text box in the lower
 *   left corner creates a sort of 'console' output overlaying the scene.
 *   Note that the alignment options control the positioning of the text box.
 *   The text can be left, center or right justified inside the box using
 *   the #textAlignH field.
 *
 *   In order to have convenient default values for the font properties, by
 *   default the font properties are @I not@i inherited from the scene graph.
 *   By default the text is rendered using an SoText2 node with font name
 *   'Arial:Bold', font size 15 and line spacing 1.1. The application can modify
 *   the #fontName and #fontSize fields or modify text properties directly
 *   using the getFontNode() and getTextNode() methods.
 *
 *   Lighting and picking are disabled.  A #border can be drawn around the box.
 *   Text will be rendered on top of whatever is rendered in the main scene graph.
 *   The application can modify the text strings directly, but this class also
 *   provides some convenience methods that are very useful. For example, the
 *   addLine() method appends a new string to the end of the list.
 *
 *   The bounding box of this node is ignored, i.e. it does not contribute to an
 *   SoGetBoundingBoxAction traversal and it does not affect a "viewAll" call on
 *   the camera or viewer.
 *
 *   @B Note:@b @BR
 *   If you are using one of the Open Inventor viewer classes (e.g. SoWinExaminerViewer).
 *   You must explicitly create a camera node (e.g. SoPerspectiveCamera) and add it to
 *   the scene graph. The viewer will @I not@i automatically create a camera, because it
 *   will find the camera in this node's internal scene graph.
 *
 * @FILE_FORMAT_DEFAULT
 *    TextBox {
 *    @TABLE_FILE_FORMAT
 *      @TR position      @TD 0 0 0
 *      @TR alignmentH    @TD LEFT
 *      @TR alignmentV    @TD TOP
 *      @TR textAlignH    @TD LEFT
 *      @TR fontName      @TD Arial:Bold
 *      @TR fontSize      @TD 15
 *      @TR border        @TD FALSE
 *      @TR borderColor   @TD 1 1 1
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, DicomInfo, Gnomon, Magnifier, Ruler
 *
 * @PREVIEWFEATURES 
 */ 

class INVENTORMEDICAL_API TextBox : public SoAnnotation {

  SO_NODE_HEADER(TextBox);

public:
  /** Position of the text box in normalized screen coordinates (-1 to 1).
   *  Default is 0,0,0. */
  SoSFVec3f position;

  /** Horizontal alignment of the text box (default is LEFT). */
  SoSFEnum  alignmentH;

  /** Vertical alignment of the text box (default is TOP). */
  SoSFEnum  alignmentV;

  /** Horizontal alignment of the text inside the box (default is LEFT). */
  SoSFEnum  textAlignH;

  /** Specify the font name (default is "Arial:Bold").
   *  See SoFont::name for details. */
  SoSFString fontName;

  /** Specify the font size in pixels (default is 15). */
  SoSFFloat fontSize;

  /** Enable drawing a border around the text box (default is FALSE). */
  SoSFBool  border;

  /** Border color (default is 1,1,1). */
  SoSFColor borderColor;

  /** Horizontal alignment values. */
  enum AlignmentH {
    /** Left edge */
    LEFT   = SoTextProperty::LEFT,
    /** Center */
    CENTER = SoTextProperty::CENTER,
    /** Right edge */
    RIGHT  = SoTextProperty::RIGHT,
  };

  /** Vertical alignment values. */
  enum AlignmentV {
    /** Top edge */
    TOP   = SoTextProperty::TOP,
    /** Middle */
    MIDDLE = SoTextProperty::HALF,
    /** Bottom edge */
    BOTTOM  = SoTextProperty::BOTTOM,
  };

  /** Set the contents of the specified line of text (convenience method). */
  void setLine( const SbString& text, int line = 0 );

  /** Get the contents of the specified line of text. */
  const SbString& getLine( int line );

  /** Add a line of text at the bottom of the box. */
  void addLine( const SbString& text );

  /** Delete one or more lines of text.
   *  'startLine' is the first line to delete. @BR
   *  'numToDelete' is the number of lines to delete.
   *    Default (1) is to delete one line. -1 means delete all lines starting with 'startLine'.
   */
  void deleteLines( int startLine, int numToDelete = 1 );

  /** Delete all lines of text. */
  void deleteAll();

  /** Get number of lines of text currently in the box. */
  int getNumLines() const;

  /** Get the internal font node (allows to modify font parameters directly). */
  SoFont*  getFontNode();

  /** Get the internal text node (allows to modify line spacing directly). */
  SoText2* getTextNode();

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

  /** Constructor. */
  TextBox();

protected:
  /** Destructor */
  virtual ~TextBox();

  SoRef<SoFont>        m_fontNode;
  SoRef<SoLineSet>     m_borderGeom;
  SoRef<SoSwitch>      m_fontSwitch;
  SoRef<SoSwitch>      m_borderSwitch;
  SoRef<SoText2>       m_textNode;
  SoRef<SoTranslation> m_tranNode;
  SoRef<SoSeparator>   m_textSep;

  bool                 m_isModified;
  SbViewportRegion     m_curViewport;
  SbBox3f              m_curBBox;

  SoNodeSensor*        m_nodeSensor;
  static void nodeSensorCB( void* data, SoSensor* sensor );

  static void callbackNodeCB( void* data, SoAction* action );

  void updatePosAndGeom( SoState* state = NULL );
};

#endif