/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _PO_MESHCROSSSECTION_
#define  _PO_MESHCROSSSECTION_

#include <MeshViz/3Ddata/PoMesh3D.h>
#include <Inventor/fields/SoSFPlane.h>

#include <Inventor/STL/vector>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class PbIndexedMesh3D;
class PbTetrahedronMesh3D;
class PbHexahedronMesh3D;
class PbCartesianGrid3D;
class PbParalCartesianGrid3D;
class PbRegularCartesianGrid3D;
class PbMesh;

class SoMFInt32;
class SoMFVec3f;
class SoCoordinate3;
class SoMaterial;

/**
 * @DTEXT  Representation of cross section of a volume mesh.
 * 
 * @ingroup MeshRepresentation
 * 
 * @DESCRIPTION
 *    Class to build a filled representation of a cross-section in a volume mesh. 
 * 
 *    If the field PoMesh::valuesIndex is not defined (or if coloringType = 
 *    COLOR_INHERITED), the representation uses only one inherited color. If 
 *    the field PoMesh::valuesIndex is defined (and if coloringType != 
 *    COLOR_INHERITED), the representation is colored by using the current data 
 *    mapping applied to each mesh node value. (see PoMesh for more explanations 
 *    about the current data-mapping).
 *    
 *    The vector data of the mesh for this representation is unused.
 *
 *    When using the #ISOSURFACE mode, the cross section can be generated by using
 *    either a triangle strip set or an indexed face set depending on the preference
 *    variable SoPreference#OIV_3DDATA_USE_TRIANGLE_STRIP_SET
 * 
 * @FILE_FORMAT_DEFAULT
 *    PoMeshCrossSection {
 *    @TABLE_FILE_FORMAT
 *       @TR plane @TD Z=0 plane
 *       @TR valuesIndex @TD -1
 *       @TR coloringType @TD COLOR_INHERITED
 *       @TR vecsIndex @TD -1
 *       @TR valuesIndexForCellFilter  @TD -1
 *       @TR moduleDataMapping @TD NULL
 *    @TABLE_END
 *    }
 * 
 * @CATALOG_PARTS_INCLUDE pomeshcrosssection.cat.html
 * 
 * 
 */
class MESHVIZ_API PoMeshCrossSection : public PoMesh3D {

  SO_KIT_HEADER(PoMeshCrossSection) ;

  // Define fields for new parts
  SO_KIT_CATALOG_ENTRY_HEADER(groupCrossSection) ;


 public:

  /**
   * Constructor.
   */
  PoMeshCrossSection() ;

  /**
   * Forces node-kit rebuilding. See PoBase.
   */
  virtual void   rebuild() ;

  /**
   * Position of the cross section. Z=0 plane by default.
   */
  SoSFPlane  plane;


  /** 
   * Specifies the method used to compute intersections.
   * @see setCrossSectionMethod().
   */
  enum CrossSectionMethod {
    /** 
     * Depending on the mesh type the cross section is computed either 
     * like an isosurface or by doing classic intersection.
     * Classic intersections are done for PbParalCartesianGrid3D and PbRegularCartesianGrid3D.
     * An isosurface is computed for all other mesh types.
     */
    DEFAULT,
    /** The cross section is computed like an isosurface */
    ISOSURFACE,
    /** 
     * The cross section is computed by doing classic intersections 
     * Be careful, this method doesn't support cell filtering (PoCellFilter).
     */
    INTERSECTION
  };

  /**
   * Specifies the method used to build this shape.
   * By default PoMeshCrossSection builds a shape by computing
   * an isosurface on a non regular mesh. The node's data are the distances
   * to the plane. This method is safe but it uses a significant 
   * amount of memory for each instance of PoMeshCrossSection.
   * 
   * Calling #setCrossSectionMethod(#INTERSECTION) forces the use of the classic method.
   * The classic method computes the shapes by intersecting each cell with
   * the plane. However it can be risky as it assumes that each mesh's cell is 
   * convex and therefore that the intersection of a plane and a cell's
   * face can produce only one edge. Using the classic method with concave
   * faces may produce unpredictable results or an infinite loop.
   * 
   * The environment variable OIV_3DDATA_CROSS_SECTION_METHOD can also be set 
   * to specify the method used to build this shape. It can take the values
   * INTERSECTION or ISOSURFACE. However, calling #setCrossSectionMethod 
   * overrides this environment variable.
   *
   * @param method #DEFAULT by default.
   */
  void setCrossSectionMethod(CrossSectionMethod method=DEFAULT) 
    { m_crossMethod = method; }
  /*----------------------------------------------------------------------------*/
 PoINTERNAL public:
  static void initClass() ;
	static void exitClass() ;

  void indexedMeshCrossSection(PbIndexedMesh3D *mesh) ;
  void tetrahedronMeshCrossSection(PbTetrahedronMesh3D *mesh) ;
  void hexahedronMeshCrossSection(PbHexahedronMesh3D *mesh) ;
  void cartGrid3DCrossSection(PbCartesianGrid3D *mesh);
  void paralCartGrid3DCrossSection(PbParalCartesianGrid3D *mesh);
  void regularGrid3DCrossSection(PbRegularCartesianGrid3D *mesh);

  const float *getIsosurfDataSet();
  float getIsosurfValue() const;

  /**
   * Gets the number of bytes allocated by this instance
   */
  virtual size_t getMemorySize();

 protected:
  // Destructor
  virtual ~PoMeshCrossSection() ;  
  virtual void   setDefaultOnNonWritingFields();
  virtual SbBool setUpConnections(SbBool onOff, SbBool doItAlways = FALSE) ;
  virtual void addElementsAllCaches() ;
  virtual void preRebuild();

  virtual int getIsosurfDataSetId() const;
  virtual SbVec3f getOverallNormal() const;

  /*----------------------------------------------------------------------------*/
 private:
  
  FieldSensorList m_fieldSensorList ;
  // Sensors

  SoFieldList     m_fieldList ;
  // List of fields defined in this class

  SoCoordinate3 *crossSectionCoord;

  void orientPolygon(std::vector< int > &coordIndex, int start_ind, float xn, float yn, float zn);

  void crossPlaneAndTetrahedron(int cell_index, const float *x,const float *y, const float *z, const float *v, const int *tetra_nodes, 
                                float xn,float yn,float zn, float dist_plane_orig, 
                                int &num_cross_pt, SbBool at_vertex=TRUE);
  void crossPlaneAndPyramid(int cell_index, const float *x,const float *y, const float *z, const float *v, const int *pyra_nodes, 
                            float xn,float yn,float zn, float dist_plane_orig, 
                            int &num_cross_pt, SbBool at_vertex=TRUE);
  void crossPlaneAndPentahedron(int cell_index, const float *x,const float *y, const float *z, const float *v, const int *penta_nodes, 
                                float xn,float yn,float zn, float dist_plane_orig, 
                                int &num_cross_pt, SbBool at_vertex=TRUE);
  void crossPlaneAndHexahedron(int cell_index, const float *x,const float *y, const float *z, const float *v, 
                               const int *hexa_nodes, const int *hexa_ind, 
                               float xn,float yn,float zn, float dist_plane_orig, 
                               int &num_cross_pt, SbBool at_vertex=TRUE);
  SbBool isFaceAlreadyFound(int num_new_points, int nb_pt_inters, int pt_inters_sorted[4]);
  
  void tetrahedronMeshCrossSectionInhColr(PbTetrahedronMesh3D *mesh, const float *v, SbBool at_vertex=TRUE) ;
  void tetrahedronMeshCrossSectionMapColr(PbTetrahedronMesh3D *mesh, const float *v) ;
  void tetrahedronMeshCrossSectionContColr(PbTetrahedronMesh3D *mesh, const float *v) ;
  void tetrahedronMeshCrossSectionTexContColr(PbTetrahedronMesh3D *mesh, const float *v) ;
  void tetrahedronMeshCrossSectionPerCellColr(PbTetrahedronMesh3D *mesh, const float *v) ;

  void hexahedronMeshCrossSectionInhColr(PbHexahedronMesh3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void hexahedronMeshCrossSectionMapColr(PbHexahedronMesh3D *mesh, const float *v) ;
  void hexahedronMeshCrossSectionContColr(PbHexahedronMesh3D *mesh, const float *v) ;
  void hexahedronMeshCrossSectionTexContColr(PbHexahedronMesh3D *mesh, const float *v) ;
  void hexahedronMeshCrossSectionPerCellColr(PbHexahedronMesh3D *mesh, const float *v) ;

  void indexedMeshCrossSectionInhColr(PbIndexedMesh3D *mesh, const float *v, SbBool at_vertex=TRUE) ;
  void indexedMeshCrossSectionMapColr(PbIndexedMesh3D *mesh, const float *v) ;
  void indexedMeshCrossSectionContColr(PbIndexedMesh3D *mesh, const float *v) ;
  void indexedMeshCrossSectionTexContColr(PbIndexedMesh3D *mesh, const float *v) ;
  void indexedMeshCrossSectionPerCellColr(PbIndexedMesh3D *mesh, const float *v) ;

  void cartGrid3DCrossSectionInhColr(PbCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void cartGrid3DCrossSectionMapColr(PbCartesianGrid3D *mesh, const float *v);
  void cartGrid3DCrossSectionContColr(PbCartesianGrid3D *mesh, const float *v);
  void cartGrid3DCrossSectionTexContColr(PbCartesianGrid3D *mesh, const float *v);
  void cartGrid3DCrossSectionPerCellColr(PbCartesianGrid3D *mesh, const float *v);

  void paralCartGrid3DXCrossSectionInhColr(PbParalCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void paralCartGrid3DYCrossSectionInhColr(PbParalCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void paralCartGrid3DZCrossSectionInhColr(PbParalCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void paralCartGrid3DCrossSectionInhColr(PbParalCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void paralCartGrid3DCrossSectionMapColr(PbParalCartesianGrid3D *mesh, const float *v);
  void paralCartGrid3DCrossSectionContColr(PbParalCartesianGrid3D *mesh, const float *v);
  void paralCartGrid3DCrossSectionTexContColr(PbParalCartesianGrid3D *mesh, const float *v);
  void paralCartGrid3DCrossSectionPerCellColr(PbParalCartesianGrid3D *mesh, const float *v);

  void regularGrid3DXCrossSectionInhColr(PbRegularCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void regularGrid3DYCrossSectionInhColr(PbRegularCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void regularGrid3DZCrossSectionInhColr(PbRegularCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void regularGrid3DCrossSectionInhColr(PbRegularCartesianGrid3D *mesh, const float *v, SbBool at_vertex=TRUE);
  void regularGrid3DCrossSectionMapColr(PbRegularCartesianGrid3D *mesh, const float *v);
  void regularGrid3DCrossSectionContColr(PbRegularCartesianGrid3D *mesh, const float *v);
  void regularGrid3DCrossSectionTexContColr(PbRegularCartesianGrid3D *mesh, const float *v);
  void regularGrid3DCrossSectionPerCellColr(PbRegularCartesianGrid3D *mesh, const float *v);

  int m_allocElementFaces,m_numElementFaces;
  int (*m_elementFaces)[5];

  SbBool       m_distNodes2PlaneUpdated;
  std::vector<float> m_distNodes2Plane;
  int          m_distNodes2PlaneId; // state id of the data set (m_distNodes2Plane)
  float        m_crossValue;

  SbVec3f      m_lastPlaneNormal;
  SbBool       m_crossMethod;

} ;

/*----------------------------------------------------------------------------*/

#define P_PLANE_EPS 0.0001

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _PO_MESHCROSSSECTION  _ */



