/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _MBVEC3_H
#define  _MBVEC3_H

#ifdef _WIN32
#  pragma warning( push )
#  pragma warning(disable:4250)
#endif


#include <Inventor/STL/iostream>
#include <Inventor/STL/limits>
#include <Inventor/SbMathHelper.h>

#include <cmath>

/**
 * @DTEXT Class defining a vector or a point of 3 coordinates.
 * 
 * @ingroup MeshVizXLM
 * 
 * @DESCRIPTION
 * 3D vector class used to store 3D vectors and points.
 * This class is similar to the standard Open Inventor SbVec3 classes,
 * but exists to allow MeshViz Extraction to be used independently of
 * Open Inventor.
 * 
 */
template <typename _T>
class MbVec3
{
public:
  typedef _T ValueType;

  /** Returns the maximum value for a vector */
  static MbVec3 numeric_limit_max()
  {
    return MbVec3<_T>((std::numeric_limits<_T>::max)());
  }

  /** Constructor (vector is initialized to zero values) */
  MbVec3() 
  { 
    vec[0] = vec[1] = vec[2] = 0; 
  }
  /** Constructor */
  MbVec3(const _T v[3])
  { 
    vec[0] = v[0]; vec[1] = v[1]; vec[2] = v[2]; 
  }

  /** Constructor */
  MbVec3(_T x, _T y, _T z)
  { 
    vec[0] = x; vec[1] = y; vec[2] = z; 
  }

  template < typename _Type >
  MbVec3(_Type x, _Type y, _Type z)
  { 
    vec[0] = (_T) x; vec[1] = (_T) y; vec[2] = (_T) z; 
  }

  /** Constructor */
  explicit MbVec3(char v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(unsigned char v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(short v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(unsigned short v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(int v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(size_t v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(long v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

#if defined(_WIN32)
  /** Constructor */
  explicit MbVec3(unsigned long v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }
#endif

  /** Constructor */
  explicit MbVec3(float v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(double v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  explicit MbVec3(long double v)
  { 
    vec[0] = (_T)v; vec[1] = (_T)v; vec[2] = (_T)v; 
  }

  /** Constructor */
  template<typename _MbVec3T>
  explicit MbVec3(const _MbVec3T& v)
  {
    vec[0] = (_T)v[0];
    vec[1] = (_T)v[1];
    vec[2] = (_T)v[2];
  }

  /**
  * Returns right-handed cross product of vector and another vector.
  */
  MbVec3  cross(const MbVec3<_T> &v) const
  {
    return MbVec3<_T>(vec[1] * v.vec[2] - vec[2] * v.vec[1],
      vec[2] * v.vec[0] - vec[0] * v.vec[2],
      vec[0] * v.vec[1] - vec[1] * v.vec[0]);
  }

  /**
  * Returns dot (inner) product of vector and another vector.
  */
  _T  dot(const MbVec3<_T> &v) const 
  {
    return (vec[0] * v.vec[0] + vec[1] * v.vec[1] + vec[2] * v.vec[2]);
  }

  /**
  * Returns pointer to vector components.
  */
  const _T *getValue() const { return vec; }

  /**
  * Returns vector components.
  */
  void getValue(_T &x, _T &y, _T &z) const
  {
    x = vec[0]; y = vec[1]; z = vec[2];
  }

  /**
  * Returns geometric length of vector.
  */
  _T length() const
  {
    return _T(sqrt(double(vec[0] * vec[0] + vec[1] * vec[1] + vec[2] * vec[2])));
  }

  /**
  * Changes vector to be unit length, returning the length before normalization.
  */
  _T  normalize();

  /**
  * Negates each component of vector in place.
  */
  void  negate()
  {
    vec[0] = -vec[0]; vec[1] = -vec[1]; vec[2] = -vec[2];
  }

  /**
  * Component-wise vector multiplication.
  */
  MbVec3 mult(const MbVec3<_T> &v) const 
  {
    return MbVec3<_T>(vec[0] * v.vec[0], vec[1] * v.vec[1],vec[2] * v.vec[2]) ;
  }

    /**
  * Component-wise vector division.
  */
  MbVec3 div(const MbVec3<_T> &v) const 
  {
    return MbVec3<_T>(vec[0] / v.vec[0], vec[1] / v.vec[1],vec[2] / v.vec[2]) ;
  }

  /**
  * Sets the vector components.
  */
  MbVec3<_T>& setValue(const _T v[3])
  { 
    vec[0] = v[0]; vec[1] = v[1]; vec[2] = v[2]; return *this; 
  }

  /**
  * Sets the vector components.
  */
  MbVec3<_T> &setValue(_T x, _T y, _T z)
  { 
    vec[0] = x; vec[1] = y; vec[2] = z; return *this; 
  }

  //@{
  /**
  * Accesses indexed component of vector.
  */
  _T& operator [](int i) { return (vec[i]); }
  const _T & operator [](int i) const  { return (vec[i]); }
  _T& operator [](size_t i) { return (vec[i]); }
  const _T & operator [](size_t i) const  { return (vec[i]); }
//@}

  /**
  * Component-wise setting. All components are set to d.
  */
  MbVec3 & operator =(_T d)
  {
    vec[0] = vec[1] = vec[2] = d;
    return *this;
  }

  /**
  * Component-wise scalar multiplication operator.
  */
  MbVec3 & operator *=(_T d)
  {
    vec[0] *= d;
    vec[1] *= d;
    vec[2] *= d;
    return *this;
  }

  /**
  * Component-wise scalar division operator.
  */
  MbVec3 & operator /=(_T d)
  { 
    vec[0] /= d;
    vec[1] /= d;
    vec[2] /= d;
    return *this; 
  }

  /**
  * Component-wise vector addition operator.
  */
  MbVec3 & operator +=(const MbVec3 &v)
  {
    vec[0] += v.vec[0];
    vec[1] += v.vec[1];
    vec[2] += v.vec[2];
    return *this;
  }

  /**
  * Component-wise vector subtraction operator.
  */
  MbVec3 & operator -=(const MbVec3 &v)
  {
    vec[0] -= v.vec[0];
    vec[1] -= v.vec[1];
    vec[2] -= v.vec[2];
    return *this;
  }


  /**
  * Nondestructive unary negation - returns a new vector.
  */
  MbVec3 operator -() const
  {
    return MbVec3<_T>(-vec[0],-vec[1],-vec[2]);
  }

  /**
  * Component-wise vector multiplication operator.
  */
  friend MbVec3 operator *(const MbVec3 &v1, const MbVec3 &v2)
  {
    return MbVec3(v1.vec[0] * v2.vec[0], v1.vec[1] * v2.vec[1], v1.vec[2] * v2.vec[2]);
  }

  /**
  * Component-wise binary scalar multiplication operator.
  */
  friend MbVec3 operator *(const MbVec3 &v, _T d)
  {
    return MbVec3<_T>(v.vec[0] * d, v.vec[1] * d, v.vec[2] * d);
  }

  /**
  * Component-wise binary scalar multiplication operator.
  */
  friend MbVec3 operator *(_T d, const MbVec3 &v)
  { return v * d; }

  /**
  * Component-wise binary scalar division operator.
  */
  friend MbVec3 operator /(const MbVec3 &v, _T d)
  { 
    return MbVec3<_T>(v.vec[0] / d, v.vec[1] / d, v.vec[2] / d);
  }

  /**
  * Component-wise vector division operator.
  */
  friend MbVec3 operator /(const MbVec3 &v1, const MbVec3 &v2)
  {
    return MbVec3(v1.vec[0] / v2.vec[0], v1.vec[1] / v2.vec[1], v1.vec[2] / v2.vec[2]);
  }

  /**
  * Component-wise binary vector addition operator.
  */
  friend MbVec3 operator +(const MbVec3 &v1, const MbVec3 &v2)
  {
    return MbVec3(v1.vec[0] + v2.vec[0], v1.vec[1] + v2.vec[1], v1.vec[2] + v2.vec[2]);
  }

  /**
  * Component-wise binary scalar addition operator.
  */
  friend MbVec3 operator +(const MbVec3 &v, _T d)
  {
    return MbVec3(v.vec[0] + d, v.vec[1] + d, v.vec[2] + d);
  }

  /**
  * Component-wise binary scalar addition operator.
  */
  friend MbVec3 operator +(_T d, const MbVec3 &v)
  {
    return MbVec3(v.vec[0] + d, v.vec[1] + d, v.vec[2] + d);
  }

  /**
  * Component-wise binary vector subtraction operator.
  */
  friend MbVec3 operator -(const MbVec3 &v1, const MbVec3 &v2)
  {
    return MbVec3(v1.vec[0] - v2.vec[0], v1.vec[1] - v2.vec[1], v1.vec[2] - v2.vec[2]);
  }

  /**
  * Component-wise binary scalar subtraction operator.
  */
  friend MbVec3 operator -(const MbVec3 &v, _T d)
  {
    return MbVec3(v.vec[0] - d, v.vec[1] - d, v.vec[2] - d);
  }

  /**
  * Component-wise vector square root operator.
  */
  friend MbVec3 sqrt(const MbVec3 &v)
  {
    return MbVec3(sqrt(v.vec[0]), sqrt(v.vec[1]), sqrt(v.vec[2]));
  }

  /**
  * Equality comparison operator.
  */
  friend bool operator ==(const MbVec3 &v1, const MbVec3 &v2)
  {
    return (v1.vec[0] == v2.vec[0] && v1.vec[1] == v2.vec[1] && v1.vec[2] == v2.vec[2]);
  }

  /**
  * Inequality comparison operator.
  */
  friend bool operator !=(const MbVec3 &v1, const MbVec3 &v2)
  { 
    return !(v1 == v2); 
  }

  /**
  * Length comparison operator.
  */
  friend bool operator <(const MbVec3 &v1, const MbVec3 &v2)
  { 
    return v1.length() < v2.length(); 
  }

  /**
  * Length comparison operator.
  */
  friend bool operator <=(const MbVec3 &v1, const MbVec3 &v2)
  { 
    return v1.length() <= v2.length(); 
  }

  /**
  * Length comparison operator.
  */
  friend bool operator >(const MbVec3 &v1, const MbVec3 &v2)
  { 
    return v1.length() > v2.length(); 
  }

  /**
  * Length comparison operator.
  */
  friend bool operator >=(const MbVec3 &v1, const MbVec3 &v2)
  { 
    return v1.length() >= v2.length(); 
  }

  /**
  * Equality comparison within given tolerance - the square of the length of the
  * maximum distance between the two vectors.
  */
  bool equals(const MbVec3 &v, _T tolerance) const
  {
    MbVec3	diff = *this - v;
    return diff.dot(diff) <= tolerance;
  }

  /**
  * Equality comparison within the machine epsilon given by the numeric_limits<_T> 
  * Square of the length of the maximum distance between the two vectors.
  */
  bool equals(const MbVec3 &v) const
  {
    return this->equals(v,std::numeric_limits<_T>::epsilon());
  }

  /** 
  * Returns true if vectors are collinear within the given tolerance.
  */
  friend bool areCollinear(const MbVec3<_T> &v1, const MbVec3<_T> &v2, _T tol) 
  {
    return SbMathHelper::isCoinc(v1[0]*v2[1],v1[1]*v2[0],tol) && SbMathHelper::isCoinc(v1[0]*v2[2],v1[2]*v2[0],tol)
        && SbMathHelper::isCoinc(v1[1]*v2[2],v1[2]*v2[1],tol);
  }

  /** 
  * Returns true if vectors are collinear within the machine epsilon
  * given by the numeric_limits<_T>
  */
  friend bool areCollinear(const MbVec3<_T> &v1, const MbVec3<_T> &v2) 
  {
    return areCollinear(v1,v2,std::numeric_limits<_T>::epsilon());
  }

  /**
  * Writes the vector to the specified output stream.
  */
  friend inline std::ostream& operator << (std::ostream& os, const MbVec3& v)
  {
    return os << "(" << v.vec[0] << "," << v.vec[1] << "," << v.vec[2] << ")";
  }

protected:
  _T vec[3];
};

template <typename _T>
_T  
MbVec3<_T>::normalize()
{
  _T len = length();

  if (len != 0)
    (*this) /= len;
  else
    setValue(0,0,0);
  return len;
}

/**
* Vector of 3 unsigned int.
* @ingroup MeshVizXLM
* @see MbVec3
*/ 
typedef MbVec3<size_t> MbVec3ui;

/**
* Vector of 3 float coordinates.
* @ingroup MeshVizXLM
* @see MbVec3
*/ 
typedef MbVec3<float> MbVec3f;
/**
* Vector of 3 double coordinates.
* @ingroup MeshVizXLM
* @see MbVec3
*/ 
typedef MbVec3<double> MbVec3d;

#ifdef _WIN32
#  pragma warning( pop )
#endif

#endif 







