/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MiPointProbeHexahedronIjk_h
#define _MiPointProbeHexahedronIjk_h

#include <MeshVizXLM/extractors/MiPointProbeIjk.h>
#include <MeshVizXLM/mesh/MiVolumeMeshHexahedronIjk.h>
#include <MeshVizXLM/mesh/cell/MiVolumeCell.h>


#include<Inventor/STL/cassert>

/**
 * @DTEXT Probe at a given location in an hexahedron IJK volume mesh
 * 
 * @ingroup MeshVizXLM_Extract_Probe
 * 
 * @DESCRIPTION This class can be used to evaluate the value of a data set in the given mesh at any location.
 * The probe is internally building an octree onto the mesh. This octree can be optimized by setting three 
 * environment variables (see MiPointProbeUnstructured and SoPreferences). 
 * 
 */
class MESHVIZXLM_EXTR_API MiPointProbeHexahedronIjk : virtual public MiPointProbeIjk
{
public:

  /**
  * Factory method returning a new instance of a class implementing this abstract class.
  * @param mesh The input mesh 
  * @param parallel When true, tries to create an extract module using a parallel algorithm to speed up the extraction. 
  * @note Only some of the extraction algorithms have a parallel implementation.
  */ 
  static MiPointProbeHexahedronIjk* getNewInstance(const MiVolumeMeshHexahedronIjk& mesh, bool parallel = true);

  /**
  * Gets the value at the current probe location in the given PER_NODE dataset.
  * The returned value is meaningless if the probe location is outside the
  * mesh (when isFound() is false).
  */
  template <typename _T>
  _T getValue(const MiDataSetI<_T>& dataset) const;

  /**
  * Gets the value at the current probe location in the given PER_CELL dataset.
  * The returned value is meaningless if the probe location is outside the
  * mesh (when isFound() is false).
  */
  template <typename _T>
  _T getValue(const MiDataSetIjk<_T>& dataset) const;

protected://PROTECTED_TO_DOCUMENT
  /**
  * Gets the cell of the mesh containing the current location of the probe.
  * This method is protected to have a public api the most similar to 
  * the MiPointProbeIjk.
  * 
  * @returns NULL if the current location of the probe is outside the mesh.
  */
  virtual const MiVolumeCell* getCell() const = 0;

};

//-----------------------------------------------------------------------------
template <typename _T>
inline _T MiPointProbeHexahedronIjk::getValue(const MiDataSetI<_T>& dataset) const
{
  _T val(0);
  if (isFound())
  {
    assert(dataset.getBinding()==MiDataSetI<_T>::PER_NODE);
    const MiCell *cell = getCell();
    std::vector<double> weight(cell->getNumNodes());
    getWeight(weight);
    for (size_t i=0; i<cell->getNumNodes(); ++i)
      val += weight[i]*dataset.get(cell->getNodeIndex(i));
  }
  return val;
}

//-----------------------------------------------------------------------------
template <typename _T>
inline _T MiPointProbeHexahedronIjk::getValue(const MiDataSetIjk<_T>& dataset) const
{
  _T val(0);
  if (isFound())
  {
    assert(dataset.getBinding()==MiDataSetIjk<_T>::PER_CELL);
    size_t icell,jcell,kcell;
    getCellId(icell,jcell,kcell);
    val = dataset.get(icell,jcell,kcell);
  }
  return val;
}

#endif


