/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MiCell_H
#define _MiCell_H

#include <MeshVizXLM/mesh/geometry/MiGeometryI.h>
#include <MeshVizXLM/MxMeshVizXLM.h>
#include <MeshVizXLM/MiAbstractMethodError.h>

#include <MeshVizXLM/MbVec3.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/ostream>

/**
* @DTEXT  Defines an abstract cell of a mesh.
* 
* @ingroup MeshVizXLM_Mesh_Cell
* 
* @DESCRIPTION
*    A cell is a basic volume, surface or edge element that 
*    connects adjacent nodes of a mesh. The MiCell interface 
*    is necessary to define the topology of unstructured 1D, 2D or 3D meshes.
*    
*    MiCell is the base class that defines topological characteristics
*    of a cell (characteristics not depending on the cell dimension). 
*      - the number of nodes: see getNumNodes()
*      - the node index: see getNodeIndex().

*    These methods must be overridden by the appplication as they are 
*    pure virtual.
*    The characteristics that depend on the cell dimension are
*    defined in the derived interfaces MiLineCell for 1D cells,
*    MiSurfaceCell for 2D cells, and MiVolumeCell for 3D cells.
*
*   MiCell also defines additional methods with a default implementation.
*   A default implementation is provided in 2 cases:
*      - A default behavior is defined. The application should
*        override the method if this default behavior is not satisfying.
*        getRelativeSize() is an example of such a method.
*      - The method throws an exception as it is only used by some
*        extractors. When an application uses such extractors the method
*        must be overridden. getWeight() is an example of such a method.
*
*   The implementation of all the methods of this class must be thread-safe because it might be called concurrently by multiple threads.
*   If the implementation can't be thread-safe, MeshViz XLM must be forced to use only 1 thread.
*   It can be done by calling MiMeshViz::init(1) or MoMeshViz::init(1) or setting the parallel argument
*   to false for each extractor instantiation. For instance by calling MiSkinExtractUnstructured::getNewInstance(&mesh, false);
*
*  @see MiLineCell, MiSurfaceCell, MiVolumeCell
*  @see MiTopologyExplicitI, MiVolumeTopologyExplicitI, MiSurfaceTopologyExplicitI, MiLineTopologyExplicitI
*
*/
class MESHVIZXLM_EXTR_API MiCell
{
public:

  /**
   * Virtual destructor
   */
  virtual ~MiCell() {}

  /**
  * Gets the number of nodes. @BR This number must include the additional nodes
  * of non linear cells. For instance getNumNodes() for a quadratic triangle
  * cell should return 6.
  */
  virtual size_t getNumNodes() const = 0; 

  /**
  * Gets the index (in the mesh) of a node.
  * @param[in] node Node number (where: 0 <= node < getNumNodes()).
  */
  virtual size_t getNodeIndex(size_t node) const = 0;

  /**
  * Gets the relative size of the cell. @BR
  *
  * This method is used only by the MoMeshCellShape to display node names
  * if using the relative offset feature and
  * the streamline extraction to adapt the step integration to the size 
  * of the cell. getRelativeSize() should return a value depending on 
  * the size of the cell even if the default implementation returns 1. 
  * For instance, getRelativeSize() can return the length of the longest 
  * edge of this cell, or the length of its diagonal.
  *
  * If getRelativeSize() returns a constant value for 
  * any cell, the stream line extraction uses a constant number of 
  * step integration anywhere in the mesh. Returning a constant
  * implies that all the cells are supposed to have the same size.
  * getRelativeSize() must not return 0.
  *
  * @param[in] meshGeometry the geometry of the mesh. Used to retrieve the node coordinates
  * of this cell.
  * @return 1 by default.
  *
  */
  virtual double getRelativeSize(const MiGeometryI* SO_UNUSED_PARAM(meshGeometry)) const 
  {
    return 1;
  }

  /**
  * Gets the weights of a point defined by its iso parametric coordinates.@BR
  * Only classes defining non linear cells must implement this method.
  *
  * A weight value (aka shape function) must be given for each node of the cell.
  * The weight must respect the following rules:
  *   - The sums of the weights must be equal to 1. 
  *   - When ipcoord matches the i-th node of the cell,
  *     weight[i] must be 1 and weight[j] must be 0 for any j!=i.
  *   - The weight depends only on pcoord[0] for all types of 1D cells.
  *   - The weight depends only on pcoord[0] and pcoord[1] for all types of 2D cells.
  *
  * See chapter "Properties of shape functions" in the User's Guide for more details.
  * This method is not pure virtual because it is not used by all 
  * extraction classes. However using an extraction class that uses this
  * method will generate an exception. An application
  * does not need to override this method if no such extraction class is used.
  *
  * 
  * @param[in] ipcoord iso parametric coordinates of the points where the weights must be evaluated. 
  * @param[out] weight Must contain at least as many elements as the number of nodes in this cell.
  * @note For extraction purpose only, it is not nessessary to resize the weight vector array since 
  *     extraction classes are optimized such as weight vectors passed to this method are
  *     already allocated and large enough to retrieve all the computed weights.
  *     Thus the following assignment is then sufficient:
  * @verbatim
  weight[i] = wi
  @endverbatim
  *  for each i with 0 <= i < getNumNodes() 
  *      
  */
  virtual void getWeight(const MbVec3d& SO_UNUSED_PARAM(ipcoord), std::vector<double>& SO_UNUSED_PARAM(weight)) const 
  {
    throw MiAbstractMethodError("MiCell::getWeight(const MbVec3d &, std::vector<double>&)");
  }

  /**
  * Gets the iso parametric coordinates of a cell's node.@BR
  * Only classes defining non linear cells must implement this method.
  *
  * The interval value of parametric coordinates are usually [0,1] or [-1,+1].
  * If the application uses interval [0,1], the parametric coordinates returned
  * by this method for linear cell must be either 0 or 1. For quadratic cell, the
  * parametric coordinates must be either 0, 0.5 or 1. (0.5 for intermediate nodes).
  * 
  * See the following image as example of parametric coordinates values for linear 
  * and quadratic quadrangle.
  * 
  * @IMAGE ParamCoordCenter.png parametric coordinates in [-1,+1]
  * @IMAGE ParamCoordUncenter.png parametric coordinates in [0,1]
  *
  * @note 1D cells use only 1 parametric coordinate (MbVec3d[0]); 2D cells use only 
  * 2 parametric coordinates (MbVec3d[0-1]); 3D cells use 3 parametric coordinates.
  *
  * This method is not pure virtual because it is not used by all 
  * extraction classes. However using an extraction class that uses this
  * method will generate an exception. An application
  * does not need to override this method if no such extraction class is used.
  * 
  * 
  * @param[in] nodeIndex The index of the node.
  */
  virtual MbVec3d getIsoParametricCoord(size_t SO_UNUSED_PARAM(nodeIndex)) const 
  {
    throw MiAbstractMethodError("MiCell::getIsoParametricCoord(size_t nodeIndex)");
  }

  /**
  * Gets the center of the cell.@BR
  *
  * This method is not pure virtual because it is not used by all 
  * extraction classes. The default implementation provided computes
  * the average of the coordinates of each cell's node.
  * @param[in] geometry The geometry of the mesh. Used to retrieve the node coordinates
  * of this cell.
  * @return The center of the cell.
  */
  virtual MbVec3d getCenter(const MiGeometryI& geometry) const;


  /**
  * Checks if a point is inside or outside a cell.@BR
  *
  * This method is not pure virtual because it is not used by all 
  * extraction classes. However using an extraction class that uses this
  * method (i.e. streamline) will generate an exception. An application
  * does not need to override this method if no such extraction class is used.
  *
  * @param[in] meshGeometry The geometry of the mesh. Used to retrieve the node coordinates
  * of this cell.
  * @param[in] point The point to be checked.
  * @param[out] weights Must contain at least  as many elements as the number of 
  * nodes in this cell (see getWeight()).
  * @return True if the point is inside the cell.
  * @note For extraction purpose only, it is not nessessary to resize the weight vector array since 
  *     extraction classes are optimized such as weight vectors passed to this method are
  *     already allocated and large enough to retrieve all the computed weights.
  *     Thus the following assignment is then sufficient:
  * @verbatim
  weight[i] = wi
  @endverbatim
  *  for each i with 0 <= i < getNumNodes() 
  */
  virtual bool isPointInsideCell(const MiGeometryI& SO_UNUSED_PARAM(meshGeometry), 
                                 const MbVec3d& SO_UNUSED_PARAM(point), 
                                 std::vector<double>& SO_UNUSED_PARAM(weights)) const
  {
    throw MiAbstractMethodError("MiCell::isPointInsideCell(const MbVec3d &point)");
  }

  /**
  * @copydoc MiMesh::operator<<()
  */
  friend std::ostream& operator<< (std::ostream& s, const MiCell& cell);

protected: //PROTECTED_TO_DOCUMENT
  /**
  * @copydoc MiMesh::toStream(std::ostream& s) const
  */
  virtual std::ostream& toStream(std::ostream& s) const;

};

//-----------------------------------------------------------------------------
inline MbVec3d
MiCell::getCenter(const MiGeometryI& geometry) const
{
  MbVec3d center(0);
  for (size_t i=0; i<getNumNodes(); ++i) 
    center += geometry.getCoord(getNodeIndex(i));
  return ( center / (double) getNumNodes() );
}

//-----------------------------------------------------------------------------
inline std::ostream& 
MiCell::toStream(std::ostream& s) const
{
  s << "# num cell's nodes" << std::endl;
  s << getNumNodes() << std::endl;
  s << "[";
  for (size_t i=0; i<getNumNodes(); ++i)
    s << getNodeIndex(i) << " ";
  s << "]";

  return s;
}

//-----------------------------------------------------------------------------
inline std::ostream& 
operator << (std::ostream& s, const MiCell& cell)
{
  return cell.toStream(s);
}


#endif 







