/*=======================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef _SO_REMOTEVIZ_CLIENT_
#define _SO_REMOTEVIZ_CLIENT_

#include <Inventor/sys/port.h>
#include <RemoteViz/Clients/nodes/LibHelper.h>

#include <Inventor/SbPImpl.h>
#include <Inventor/SbString.h>
#include <Inventor/SbVec.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoMFVec2String.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoMFEnum.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/SbEventHandler.h>
#include <Inventor/SbEventArg.h>
#include <Inventor/devices/SoCpuBufferObject.h>

class SoLocation2Event;
class SoMouseButtonEvent;
class SoMouseWheelEvent;
class SoKeyboardEvent;
class SoTouchEvent;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoRemoteVizClient);

/**
* @RVEXT Client node enables to integrate RemoteViz into Open Inventor applications. 
*
* @ingroup RemoteViz
*
* @DESCRIPTION
*  This node is a RemoteViz client enabling software developers to easily integrate remote 3D interaction and visualization into their 
*  Open Inventor applications. It enables to establish a connection to a RemoteViz service so that display frames into an Open Inventor viewer.\n
*
*  \if_cpp
*  NOTE: Before using this node, initialize the RemoteVizClient module database by calling RemoteVizClient::init.
*  \endif
*
*  @SEE_ALSO
*    RemoteVizClient
*
*/
class RVCLIENT_API SoRemoteVizClient : public SoNode
{

  SO_NODE_HEADER(SoRemoteVizClient);

  SO_PIMPL_BASE_PUBLIC_HEADER(SoRemoteVizClient);

public:

  /** Display mode value */
  enum DisplayMode {
    /**
     *  Stretches the image to fill the container. The renderArea aspect ratio is not kept.
     */
    STRETCH    ,
    /**
     *  The image is cropped more or less (depending on the container size and the renderArea size) to keep the same scale as the renderArea.
     */
    CROP    ,
    /**
     *  This mode resizes the image to fit the container by keeping the same aspect ratio as the renderArea.
     */
    FIT
  };

  /** Video decoder values */
  enum VideoDecoder {
    /**
     * The video decoder is not yet initialized or the streaming mode is not StreamingMode#VIDEO.
     */
    NONE,
    /**
     *  OpenH264 is a H.264 software decoding implementation.\n
     *  It does not require specific hardware.
     */
    OPENH264,
    /**
     *  Intel QuickSync enables to use an Intel CPU or GPU for H.264 and VP9 hardware accelerated decoding.\n
     *  It requires compatible hardware and Intel Media SDK. See https://ark.intel.com/ to check the hardware compatibility.\n
     *  This decoder is not supported on macOS.
     */
    INTEL_QUICKSYNC,
    /**
    *  NVIDIA Video Codec enables to use an NVIDIA GPU for H.264 and VP9 hardware accelerated decoding.\n
    *  It requires compatible hardware and CUDA toolkit 10.1 or higher.
    *  See https://developer.nvidia.com/nvidia-video-codec-sdk to check the hardware compatibility.\n
    *  This decoder is not supported on macOS.
    */
    NVDECODE,
    /**
      *  libvpx is a VP9 software decoding implementation.\n
      *  It does not require specific hardware.
      */
    VPX
  };

  /** Streaming mode values */
  enum StreamingMode
  {
    /* The streaming mode is not yet known. */
    UNKNOWN,
    /* The client uses image streaming. */
    IMAGE,
    /* The client uses video streaming. */
    VIDEO
  };
 
  /**
  * Default constructor
  */
  SoRemoteVizClient();

  /**
  * Default destructor
  */
  virtual ~SoRemoteVizClient();

  // Fields:

  /**
  * Opens/Closes a connection between the client and the RemoteViz service.\n 
  * After being connected, the server-side onPendingCreateRenderArea callback will be triggered if the renderArea
  * does not exist otherwise the server-side onPendingShareRenderArea callback will be triggered.\n
  */
  SoSFBool connect;

  /**
  * Specifies the IP address of the RemoteViz Service to which the client will connect.\n 
  *
  */
  SoSFString ipAddress;

  /**
  * Specifies the port of the RemoteViz Service to which the client will connect.\n 
  *
  */
  SoSFUShort port;

  /**
  * Specifies the renderArea name to which the client will connect.\n 
  *
  */
  SoSFString renderAreaName;

  /**
  * This setting allows you to specify a preference list of video decoders.\n
  * The first decoder (among the decoders of the list) successfully initialized will be used to decode the video stream.\n
  * If all decoders of the list fail to be initialized, the decoder VideoDecoder#OPENH264 will be used.\n
  * @useenum{VideoDecoder}. Default is OPENH264.
  */
  SoMFEnum videoDecoders;

  /**
  * Connection parameters that will be sent to the RemoteViz service during the connection establishment.
  * The container size will be sent to the service as RenderArea requested size. This behavior can be overriden by 
  * including the requested height ("requestedHeight") and the requested width ("requestedWidth") in the parameters. 
  * If the connection is accepted, a server-side onRequestedSize callback will be triggered.
  *
  */
  SoMFVec2String connectionParameters;

  /**
  * This setting allows you to specify how to map the renderArea image (server-side) into the container (client-side).\n
  * @useenum{DisplayMode}. Default is FIT.
  */
  SoSFEnum displayMode;

  /**
  *  Sends a text message to the server. After being sent, the server-side onReceivedMessage callback will be triggered.
  *
  *  \param message : message to be sent
  */
  void sendMessage(const SbString& message);

  /**
  *  Sends a binary message to the server. After being sent, the server-side onReceivedMessage callback will be triggered.
  *
  *  \param buffer : buffer to be sent
  */
  void sendMessage(SoCpuBufferObject* buffer);

  /**
  *  Secures the connection between the client and the server.
  *
  *  \param publicCertificateFilePath : Path to a file containing the public certificate used to authenticate the server to the clients.
  *  The certificate has to be a PEM ("Privacy Enhanced Mail") encoded certificate.
  *
  *  \param privateKeyFilePath : Path to a file containing the private key used to sign the server key exchange between the client and the server.
  *
  *  \param enabledSecurityProtocols : Defines the security protocols used to secure the exchange between the client and the server.
  *  By default, TLSv1.1, TLSv1.2 and TLSv1.3 protocols are enabled.
  *  A bitmask of the security protocols is used. Many security protocols can be enabled by using the logical OR operator. \n 
  * [OIV-WRAPPER-ARG IN,IN,WRAP_AS{#SecurityProtocols}]
  *
  *  \param privateKeyPassphrase : The passphrase that protect the private key.
  *  Security warning !! DO NOT hard-code the passphrase !!\n
  *  Read it from a SECURE location on your system.\n\n
  */
  void enableSecureConnection(const SbString& publicCertificateFilePath, const SbString& privateKeyFilePath, unsigned int enabledSecurityProtocols = TLSv1_1 | TLSv1_2 | TLSv1_3, const SbString& privateKeyPassphrase = "");

  /**
  *  Each enumeration represents a security protocol. Use enableSecureConnection to enable them.\n
  */
  enum SecurityProtocols
  {
    SSLv2 = 0x1, // 2^0, bit 0
    SSLv3 = 0x2, // 2^1, bit 1
    TLSv1 = 0x4, // 2^2, bit 2
    TLSv1_1 = 0x8, // 2^3, bit 3
    TLSv1_2 = 0x10, // 2^4, bit 4
    TLSv1_3 = 0x20 // 2^5, bit 5
  };

  /**
  *  Gets the security procotols used in RemoteViz.\n
  *
  *  \return the bitmask of used security protocols.
  * [OIV-WRAPPER-RETURN-TYPE WRAP_AS{#SecurityProtocols}]
  */
  unsigned int getEnabledSecurityProtocols() const;

  /**
  *  Gets the SSL engine activation.
  *
  *  \return true if the secure connection is enabled.
  */
  bool isSecureConnection() const;

  /**
  *  Sends a request to resize the renderArea associated with the connection. A server-side onRequestedSize event will be triggered. \n
  *  If the renderarea size is modified in the server-side onRequestSize callback, the client-side resize event will be triggered.
  *
  *  \param size : renderArea size
  */
  void requestRenderAreaSize(const SbVec2s& size);

  /**
  *   Returns the size of the renderArea managed by the RemoteViz service.
  *
  *  \return the renderArea size
  */
  SbVec2s getRenderAreaSize() const;

  /**
  *  Returns the size of the renderArea container.
  *
  *  \return the renderArea container size
  */
  SbVec2s getContainerSize() const;

  /** 
  *  Checks if the client supports image streaming.
  *
  *  \return true if the client supports image streaming
  */
  bool isImageStreamingSupported() const;

  /**
  *  Checks if the client supports video streaming.
  *
  *  \return true if the client supports video streaming
  */
  bool isVideoStreamingSupported() const;

  /**
  * Returns the video decoder that has been successfully initialized and currently used by the client to decode video stream. \n
  * This information is known after receiving the first frame. For instance, you can retrieve this information in the event callback onDecodedFrame. \n
  * If this method is called before receiving the first frame, it will return VideoDecoder#NONE.
  * If the streaming mode is not StreamingMode#VIDEO (@see getStreamingMode), this method will return VideoDecoder#NONE.
  * @see VideoDecoder
  *  \return video decoder used by the client.
  */
  VideoDecoder getVideoDecoder() const;

  /**
  * Returns the type of streaming currently used by the client. \n
  * This information is known after receiving the first frame. For instance, you can retrieve this information in the event callback onDecodedFrame.\n
  * If this method is called before receiving the first frame, it will return StreamingMode#UNKNOWN.
  * @see StreamingMode
  *  \return streaming mode used by the client.
  */
  StreamingMode getStreamingMode() const;

  /** 
  * Structure given when a ServiceMessage event is raised.
  * Please refer to #onServiceMessage event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API ServiceMessageEventArg : public SbEventArg
  {
    enum State
    {
      /** Connected to the service. */
      CONNECTED,
      /** Disconnected from the service. */
      DISCONNECTED,
      /** The network calibration is started. */
      START_NETWORK_CALIBRATION,
      /** The network calibration is finished. */
      FINISH_NETWORK_CALIBRATION,
      /** The network bandwidth calibration is started. */
      START_BANDWIDTH_CALIBRATION,
      /** The network bandwidth calibration is finished. */
      FINISH_BANDWIDTH_CALIBRATION,
      /** The network latency calibration is started. */
      START_LATENCY_CALIBRATION,
      /** The network latency calibration is finished. */
      FINISH_LATENCY_CALIBRATION,
      /** The network calibration is pending. Another connection is currently performing it. */
      PENDING_NETWORK_CALIBRATION
    };

    /** Default constructor */
    ServiceMessageEventArg(SoRemoteVizClient* source, State state, SbVec2s renderAreaSize, const SbString& details)
      : m_source( source ), m_state( state ), m_renderAreaSize( renderAreaSize ), m_details( &details ) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the connection state with the service. */
    State getState() const { return m_state; }

    /** Returns the renderArea size. */
    const SbVec2s& getRenderAreaSize() const { return m_renderAreaSize; }

    /** 
     * Returns details of the event.
     * If State is equal to DISCONNECTED:
     *   This parameter is the disconnected reason. It can take the following values "DISPOSED", "REFUSED", "KICKED" and "NETWORKFAILURE".
     * If State is equal to FINISH_BANDWIDTH_CALIBRATION:
     *   This parameter is the network bandwidth calibration value (in bit/s).
     * Else if State is equal to FINISH_LATENCY_CALIBRATION:
     *   This parameter is the network latency calibration value (in ms).
     * Else this parameter is empty.
     */
    const SbString& getDetails() const { return *m_details; }

  private:
    SoRemoteVizClient* m_source;
    State m_state;
    SbVec2s m_renderAreaSize;
    const SbString* m_details;
  };

  /** 
  * Structure given when a received text message event is raised.
  * Please refer to #onReceivedMessage event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API MessageEventArg : public SbEventArg
  {
    /** Default constructor */
    MessageEventArg(SoRemoteVizClient* source, const SbString& message)
      : m_source(source), m_message(&message) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the received message. */
    const SbString& getMessage() const { return *m_message; }

  private:
    SoRemoteVizClient* m_source;
    const SbString* m_message;
  };

  /**
  * Structure given when a received binary message event is raised.
  * Please refer to #onReceivedMessage event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API BinaryMessageEventArg : public SbEventArg
  {
    /** Default constructor */
    BinaryMessageEventArg(SoRemoteVizClient* source, SoCpuBufferObject* buffer)
      : m_source(source), m_buffer(buffer) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const {return m_source;}

    /** Returns the received message. */
    SoCpuBufferObject* getBuffer() const {return m_buffer;}

  private:
    SoRemoteVizClient* m_source;
    SoCpuBufferObject* m_buffer;
  };

  /** 
  * Structure given when a renderArea resize event is raised.
  * Please refer to #onRenderAreaResize event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API RenderAreaSizeEventArg : public SbEventArg
  {
    /** Default constructor */
    RenderAreaSizeEventArg(SoRemoteVizClient* source, SbVec2s size)
      : m_source(source), m_size(size) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the renderArea size. */
    const SbVec2s& getSize() const { return m_size; }

    private:
      SoRemoteVizClient* m_source;
      SbVec2s m_size;
  };
  
  /** 
  * Structure given when a received and decoded frame event is raised.
  * Please refer to #onReceivedFrame, #onDecodedFrame event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API FrameEventArg : public SbEventArg
  {
    /** Default constructor */
    FrameEventArg(SoRemoteVizClient* source, uint64_t id, uint64_t dataLength, bool isLossless, const SbString& message)
      : m_source(source), m_id(id), m_dataLength(dataLength), m_isLossless(isLossless), m_message(&message) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the frame ID. */
    uint64_t getId() const { return m_id; }

    /** Returns the frame data length. */
    uint64_t getDataLength() const { return m_dataLength; }

    /** Returns if the frame encoding is lossless. */
    bool isLossless() const { return m_isLossless; }

    /** Returns the attached message. */
    const SbString& getMessage() const { return *m_message; }

    private:
      SoRemoteVizClient* m_source;
      uint64_t m_id;
      uint64_t m_dataLength;
      bool m_isLossless;
      const SbString* m_message;
  };

  /** 
  * Structure given when a mouse location event is raised.
  * Please refer to #onMouseLocationEvent event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API MouseLocationEventArg : public SbEventArg
  {
    /** Default constructor */
    MouseLocationEventArg(SoRemoteVizClient* source, SoLocation2Event* mouseLocationEvent)
      : m_source(source), m_mouseLocationEvent(mouseLocationEvent) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the mouse location event. */
    SoLocation2Event* getMouseLocationEvent() const { return m_mouseLocationEvent; }

    private:
      SoRemoteVizClient* m_source;
      SoLocation2Event* m_mouseLocationEvent;
  };

  /** 
  * Structure given when a mouse button event is raised.
  * Please refer to #onMouseButtonEvent event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API MouseButtonEventArg : public SbEventArg
  {
    /** Default constructor */
    MouseButtonEventArg(SoRemoteVizClient* source, SoMouseButtonEvent* mouseButtonEvent)
      : m_source(source), m_mouseButtonEvent(mouseButtonEvent) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the mouse button event. */
    SoMouseButtonEvent* getMouseButtonEvent() const { return m_mouseButtonEvent; }

    private:
      SoRemoteVizClient* m_source;
      SoMouseButtonEvent* m_mouseButtonEvent;
  };

  /** 
  * Structure given when a mouse wheel event is raised.
  * Please refer to #onMouseWheelEvent event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API MouseWheelEventArg : public SbEventArg
  {
    /** Default constructor */
    MouseWheelEventArg(SoRemoteVizClient* source, SoMouseWheelEvent* mouseWheelEvent)
      : m_source(source), m_mouseWheelEvent(mouseWheelEvent) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the mouse wheel event. */
    SoMouseWheelEvent* getMouseWheelEvent() const { return m_mouseWheelEvent; }

    private:
      SoRemoteVizClient* m_source;
      SoMouseWheelEvent* m_mouseWheelEvent;
  };

  /** 
  * Structure given when a keyboard event is raised.
  * Please refer to #onKeyboardEvent event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API KeyboardEventArg : public SbEventArg
  {
    /** Default constructor */
    KeyboardEventArg(SoRemoteVizClient* source, SoKeyboardEvent* keyboardEvent)
      : m_source(source), m_keyboardEvent(keyboardEvent) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the keyboard event. */
    SoKeyboardEvent* getKeyboardEvent() const { return m_keyboardEvent; }

    private:
      SoRemoteVizClient* m_source;
      SoKeyboardEvent* m_keyboardEvent;
  };

  /** 
  * Structure given when a touch event is raised.
  * Please refer to #onTouchEvent event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct RVCLIENT_API TouchEventArg : public SbEventArg
  {
    /** Default constructor */
    TouchEventArg(SoRemoteVizClient* source, SoTouchEvent* touchEvent)
      : m_source(source), m_touchEvent(touchEvent) {}

    /** Returns the SoRemoteVizClient node that triggered the event. */
    SoRemoteVizClient* getSource() const { return m_source; }

    /** Returns the touch event. */
    SoTouchEvent* getTouchEvent() const { return m_touchEvent; }

    private:
      SoRemoteVizClient* m_source;
      SoTouchEvent* m_touchEvent;
  };

  /**
  *  Triggered when a service message is received from the RemoteViz service.\n
  */
  SbEventHandler<ServiceMessageEventArg&> onServiceMessage;

  /**
  *  Triggered when a text message is received from the RemoteViz service.\n
  */
  SbEventHandler<MessageEventArg&> onReceivedMessage;

  /**
  *  Triggered when a binary message is received from the RemoteViz service.\n
  */
  SbEventHandler<BinaryMessageEventArg&> onReceivedBinaryMessage;

  /**
  *  Triggered when the renderArea has been resized.\n
  */
  SbEventHandler<RenderAreaSizeEventArg&> onRenderAreaResize;

  /**
  *  Triggered when a frame is received from the RemoteViz service.\n
  */
  SbEventHandler<FrameEventArg&> onReceivedFrame;

  /**
  *  Triggered when a frame is decoded and displayed.\n
  */
  SbEventHandler<FrameEventArg&> onDecodedFrame;

  /**
  *  Triggered when a mouse location event is fired on the client.\n
  */
  SbEventHandler<MouseLocationEventArg&> onMouseLocationEvent;

  /**
  *  Triggered when a mouse button event is fired on the client.\n
  */
  SbEventHandler<MouseButtonEventArg&> onMouseButtonEvent;

  /**
  *  Triggered when a mouse wheel event is fired on the client.\n
  */
  SbEventHandler<MouseWheelEventArg&> onMouseWheelEvent;

  /**
  *  Triggered when a keyboard event is fired on the client.\n
  */
  SbEventHandler<KeyboardEventArg&> onKeyboardEvent;

  /**
  *  Triggered when a touch event is fired on the client.\n
  */
  SbEventHandler<TouchEventArg&> onTouchEvent;

  SoINTERNAL public:
  // Initializes this class
  static void		initClass();
  static void		exitClass();

protected:
  /**
  * Detect changes to fields.
  */
  virtual void notify(SoNotList *list);

  /**
  * This method performs the "typical" operation of a node for any action.
  */
  virtual void doAction(SoAction *action);

  /**
  * Implement render action.
  */
  virtual void GLRender(SoGLRenderAction *action);

  /**
  * Traversal for handle event.
  */
  virtual void handleEvent(SoHandleEventAction *action);


private:

  void construct();
};

#endif // _SO_REMOTEVIZ_CLIENT_

