/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <memory>

namespace RemoteViz
{
  namespace Rendering
  {

    class FrameEncodersImpl;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Defines a pair of encoders that are used to encode still and interactive frames.
     *
     *  Default value for both still and interactive is NONE when a FrameEncoders object
     *  is created. But when a client connects to the service, RemoteViz will set PNG
     *  for still frames and JPEG for interactive frames. Frames are considered
     *  "interactive" when the user is interacting with the scene, for example dragging
     *  the mouse to move the camera. In this case it may be acceptable to use (for
     *  example) lossy JPEG encoding to maximize performance, then switch to
     *  loss-less PNG encoding for the "still" frame when the user interaction is finished.
     *
     *  Video encoding (H.264 or VP9) generally provides better performance (frames
     *  per second) than image encoding (JPEG and PNG). However this depends on many
     *  factors. Video encoded frames usually require less bandwidth, but the encoding
     *  of each frame can take more time. See the hardware and software dependencies for
     *  each encoder below.
     *
     *  A FrameEncoders object is passed to the RenderAreaListener methods
     *  onOpenedConnection(), onInitializedConnection() and onRefusedEncoder().
     *  - In the onOpenedConnection() method, use the FrameEncoders object to request
     *    non-default encoders. Use the ConnectionSettings#isSupportedEncoders() method
     *    to query if the proposed encoders are available.
     *
     *  - onRefusedEncoder() will be called if one of the encoders is not supported in
     *    the current environment or the combination of encoders is not supported.
     *    Use the FrameEncoders object to determine which encoder is invalid
     *    (see #getStillEncoderStatus() and #getInteractiveEncoderStatus()), then
     *    set new encoders. onRefusedEncoder() will be called again if the requested
     *    encoders are not valid.
     *
     *  - In the onInitializedConnection() method, use the FrameEncoders
     *    object to query which encoders will actually be used.
     *
     * The current frame encoders can also be queried using ConnectionSettings#getFrameEncoders().
     *
     *  See also ConnectionSettings::setFrameEncodingPolicy().
     *
     *  See the examples HelloConeH264 and HelloConeVP9.
     *
     *  Limitations:
     *  - Currently it is not possible to "mix" video and image encoding. @BR
     *    In other words, the interactive and still encoders must be both
     *    video encoders (H264, VP9, ...) or both image encoders (JPEG, PNG, ...).
     *  - Currently, when using video encoding, the interactive and still encoders must
     *    be the same, e.g. both H264_NVENC.
     *
     * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{false},SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API FrameEncoders
    {
      /*! \cond PRIVATE */
      friend class ConnectionSettingsImpl;
      friend class FrameEncoderManager;
      /*! \endcond */

    public:
      /**
       *  Frame encoder.\n
       */
      enum Encoder
      {
        /** Internal usage */
        NONE,
        /** JPEG encoder */
        JPEG,
        /** PNG encoder */
        PNG,
        /** H.264 encoder using Nvidia Video Codec.
            This encoder requires an NVIDIA GPU Kepler or higher, CUDA Toolkit 11 or higher and driver
            version 451.22 or higher on Windows / driver version 450.36.06 or higher on Linux.
            On Windows, this encoder requires the library nvEncodeAPI(64).dll.
            On Linux, this encoder requires the libraries libnvidia-encode and libva. */
        H264_NVENC,
        /** H.264 encoder using OpenH264.
            On Linux, this encoder requires the library libva. */
        H264_OPENH264,
        /** VP9 encoder using libvpx.
            On Linux, this encoder requires the library libva. */
        VP9_VPX
      };

      /**
       *  Frame encoder status.\n
       */
      enum Status
      {
        /** The encoder can be loaded and used by the service. */
        SUPPORTED,
        /** Reasons:
            - The encoder cannot be loaded by the service: the RemoteViz codec library (fei_remoteViz_*_codec) is missing 
              or at least one dependency is not satisfied (On Linux, the command ldd can be used to check if dependencies are satisfied).
            - The encoder is not supported by the system (Example: no Nvidia GPU to use H264_NVENC). 
            Enabling debugging will output more verbose information.
        */
        UNSUPPORTED,
        /** The pair (still encoder, interactive encoder) is incompatible. */
        INCOMPATIBLE,
        /** The codec format is not supported by the client (Example: the client does not support H.264 decoding). */
        CLIENT_UNSUPPORTED
      };

      //@{
      /**
       *  Constructor. \n
       *  
       */
      FrameEncoders();

      /**
       *  Copy constructor
       */
      FrameEncoders(const FrameEncoders& obj);

      /**
       *  Copy assignment operator
       */
      FrameEncoders& operator=(const FrameEncoders& obj);
      //@}

      //@{
      /**
       *  Destructor. \n
       *
       */
      ~FrameEncoders();
      //@}

      //@{
      /**
       *  Sets the interactive encoder. \n
       *  This encoder will be used to encode interactive frames.
       *  Default value is NONE when a FrameEncoders object is created.
       *  But note that, by default, RemoteViz will set this value to JPEG
       *  when a connection is initialized.
       *
       *  \param value : interactive encoder.
       */
      void setInteractiveEncoder(Encoder value);
      /**
       *  Gets the interactive encoder. \n
       *  Default value is NONE when a FrameEncoders object is created.
       *  But note that, by default, RemoteViz will set this value to JPEG
       *  when a connection is initialized.
       *
       *  \return the interactive encoder.
       */
      Encoder getInteractiveEncoder() const;
      //@}

      //@{
      /**
       *  Sets the still encoder. \n
       *  This encoder will be used to encode still frames. \n
       *  Default value is NONE when a FrameEncoders object is created.
       *  But note that, by default, RemoteViz will set this value to PNG
       *  when a connection is initialized.
       *
       *  \param value : still encoder.
       */
      void setStillEncoder(Encoder value);
      /**
       *  Gets the still encoder. \n
       *  Default value is NONE when a FrameEncoders object is created.
       *  But note that, by default, RemoteViz will set this value to PNG
       *  when a connection is initialized.
       *
       *  \return the still encoder.
       */
      Encoder getStillEncoder() const;
      //@}

      //@{
      /**
       *  Gets the interactive encoder status. \n
       *  Default value is UNSUPPORTED.
       *
       *  \return the interactive encoder status.
       */
      Status getInteractiveEncoderStatus() const;
      //@}

      //@{
      /**
       *  Gets the still encoder status. \n
       *  Default value is UNSUPPORTED.
       *
       *  \return the still encoder status.
       */
      Status getStillEncoderStatus() const;
      //@}

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<FrameEncodersImpl> getImpl() const;
      /*! \endcond */

    private:
      
      /** Pointer to implementation */
      std::shared_ptr<FrameEncodersImpl> pImpl;
      
    };
  }
}
