/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <memory>
#include <RemoteViz/Rendering/IFrameEncodingPolicy.h>

namespace RemoteViz
{

  namespace Rendering
  {

    class Connection;
    class KeepFrameQualityPolicyImpl;

    /**
     *  @RVEXT
     *
     *  @ingroup RemoteViz
     *
     *  This class provides a calculation policy which allows to manage the FPS (frames per second) and the quality of
     *  interactive frames sent from Remoteviz service depending on the network bandwidth allocated to the Connection.
     *
     *  If the Client owns multiple connections, the network bandwidth allocated to each Connection will be equitably shared.
     *  Otherwise, the network bandwidth allocated to the Connection will be equal to the network bandwidth allocated to the Client. 
     *  The network bandwidth allocated to the Client can be defined from ClientSettings#setBandwidth.
     *  This policy works as described below:
     *  - When the network bandwidth allocated to the Connection decreases, the frame quality value will keep the frame quality 
     *    target value and the FPS will decrease.
     *
     *  - If the FPS decreases until the FPS threshold, then the frame quality will start to decrease.
     *
     *  - When the network bandwidth allocated to the Connection increases, the frame quality will increase until it reaches the 
     *    frame quality target and then the FPS will increase until it reaches the FPS target.
     *
     *  See ConnectionSettings#setFrameEncodingPolicy to define the frame encoding policy.
     *
     *  @SEE_ALSO
     *  KeepFramesPerSecondPolicy
     */
    class RENDERSERVICE_API KeepFrameQualityPolicy : public IFrameEncodingPolicy
    {
    public:
      /**
       *  Constructor.
       *  The frame quality target will be set to 0.7 \n
       *  The FPS target will be set to 30. \n
       *  The FPS threshold will be set to 15. \n
       *
       */
      KeepFrameQualityPolicy();

      /**
       *  Convenience Constructor.
       *
       *  \param qualityTarget : frame quality target, the value range is [0, 1].
       *  \param fpsTarget : FPS target, the value range is [1, 60].
       *  \param fpsThreshold : frames per second threshold, the value range is [1, 60]. (fpsThreshold <= fpsTarget)
       *
       */
      KeepFrameQualityPolicy( float qualityTarget, unsigned int fpsTarget, unsigned int fpsThreshold );

      /**
       *  Destructor.
       *
       */
      virtual ~KeepFrameQualityPolicy();

      /**
       *  Gets the FPS calculated after calling the method update().
       *  The value range is [1, 60].
       *
       *  \return the FPS.
       */
      virtual unsigned int getFramesPerSecond() const override;

      /**
       *  Gets the interactive frame quality calculated after calling the method update().
       *  The value range is [0, 1].
       *
       *  \return the interactive frame quality.
       */
      virtual float getFrameQuality() const override;

      /**
       *  Update value of the FPS and frame quality.
       *  The connection will be used to get values such encode frame buffer size or bandwidth.
       *
       *  \param connection : connection used to get values.
       */
      virtual void update( std::shared_ptr<Connection> connection ) override;

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<KeepFrameQualityPolicyImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<KeepFrameQualityPolicyImpl> pImpl;
    };
  }
}
