/*=======================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>

#include <memory>

namespace RemoteViz
{
  namespace Rendering
  {
    class Client;
    class RenderArea;
    class Connection;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     *  @DESCRIPTION
     *  This class can be overridden by an application to monitor the service.
     *  These listeners allow to report states and performances metrics of the service.
     *  The metrics/listeners can be enabled using Monitoring#enableMetrics.
     *  The overridden class will be passed as argument to the method Monitoring#addListener.
     *
     *  The metrics are triggered in the service thread, so costly operations in the listeners could impact global performances of the service.
     *
     *  This class is an abstract class which must be overridden.
     *
     *  See $OIVHOME/examples/source/RemoteViz/Monitoring
     *
     *  [OIVNET-WRAPPER-CLASS DERIVABLE{Default},SHARED_POINTER_USE]
     *  [OIVJAVA-WRAPPER-CLASS DERIVABLE{Strong_Ref},SHARED_POINTER_USE]
     *
     *  @SEE_ALSO
     *  Monitoring
     */
    class RENDERSERVICE_API MetricsListener
    {
    public:
      /**
      * Destructor
      */
      virtual ~MetricsListener();

      /**
      *  Triggered when a new measure of network latency is available.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NETWORK_LATENCY.
      *
      *  \param time : network latency in milliseconds
      *
      *  \param client : The network latency is related to this client.
      */
      virtual void onMeasuredNetworkLatency( unsigned int time, std::shared_ptr<Client> client );

      /**
      *  Triggered when a new measure of frame decoding time is available.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#DECODING_TIME.
      *
      *  \param time : frame decoding time in milliseconds
      *
      *  \param connection : The frame decoding time is related to this connection.
      */
      virtual void onMeasuredDecodingTime( unsigned int time, std::shared_ptr<Connection> connection );

      /**
      *  Triggered when a new measure of frame rendering time is available.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#RENDERING_TIME.
      *
      *  \param time : frame rendering time in milliseconds
      *
      *  \param renderArea : The frame rendering time is related to this render area.
      */
      virtual void onMeasuredRenderingTime( unsigned int time, std::shared_ptr<RenderArea> renderArea );

      /**
      *  Triggered when a new measure of frame encoding time is available.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#ENCODING_TIME.
      *
      *  \param time : frame encoding time in milliseconds
      *
      *  \param connection : The frame encoding time is related to this connection.
      */
      virtual void onMeasuredEncodingTime( unsigned int time, std::shared_ptr<Connection> connection );

      /**
      *  Triggered when the number of clients changes.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CLIENTS.
      *
      *  \param number : number of clients
      */
      virtual void onChangedNumClients( unsigned int number );

      /**
      *  Triggered when the number of connections changes.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CONNECTIONS.
      *
      *  \param number : number of connections
      */
      virtual void onChangedNumConnections( unsigned int number );

      /**
      *  Triggered when the number of renderArea changes.
      *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_RENDERAREAS.
      *
      *  \param number : number of render areas
      */
      virtual void onChangedNumRenderAreas( unsigned int number );

    };
  }
}
