/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <memory>

namespace RemoteViz
{
  namespace Rendering
  {

    class MonitoringImpl;
    class MetricsListener;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     * This class enables to manage metrics listeners that monitor states and performance of the service.
     *
     * A Monitoring object is automatically created by RemoteViz.
     * Use the Service::getMonitoring() method to get this object and (for example) add a listener.
     *
     * See $OIVHOME/examples/source/RemoteViz/Monitoring
     *
     * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{false},SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API Monitoring
    {
      /*! \cond PRIVATE */
      friend class ServiceSettingsImpl;
      friend class ConnectionImpl;
      friend class ServiceImpl;
      friend class RenderAreaImpl;
      /*! \endcond */

    public:
      /**
      * Types of metrics
      */
      enum MetricType
      {
        /** No metrics **/
        NONE = 0x00,
        /** Network latency: Reported every 1 second. **/
        NETWORK_LATENCY = 0x1, // 2^0, bit 0
        /** Decoding time: Reported each time a frame is sent to the client. **/
        DECODING_TIME = 0x2, // 2^1, bit 1
        /** Rendering time: Reported each time a frame is rendered. **/
        RENDERING_TIME = 0x4, // 2^2, bit 2
        /** Encoding time: Reported each time a frame is sent to the client. **/
        ENCODING_TIME = 0x8, // 2^3, bit 3
        /** Number of clients: Reported when a client is added or removed. **/
        NUM_CLIENTS = 0x10, // 2^4, bit 4
        /** Number of connections: Reported when a connection is added or removed. **/
        NUM_CONNECTIONS = 0x20, // 2^5, bit 5
        /** Number of render areas: Reported when a render area is added or removed. **/
        NUM_RENDERAREAS = 0x40, // 2^6, bit 6
        /** All metrics **/
        ALL = 0xFFFF
      };

      /**
       *  Adds a listener for metrics
       *
       *  \param listener : object that listens to the service metrics.
       */
      void addListener(std::shared_ptr<MetricsListener> listener);

      /**
       *  Removes a listener for metrics
       *
       *  \param listener : object that listens to the service metrics.
       */
      void removeListener(std::shared_ptr<MetricsListener> listener);

      /**
       *  Return the number of listeners
       */
      unsigned int getNumListeners() const;

      /**
       *  Enable metrics
       *  By default, all metrics are disabled
       *
       *  \param mask : binary mask of MetricType
       * [OIV-WRAPPER-ARG WRAP_AS{#MetricType}&ENUM_SET]
       */
      void enableMetrics( unsigned int mask );

      /**
       *  Check if a metric is enabled
       *
       *  \param metric : metric to check
       *
       *  Return true if the metric is enabled, false otherwise
       */
      bool isMetricEnabled( MetricType metric ) const;

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<MonitoringImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Constructor */
      Monitoring();

      /** Pointer to implementation */
      std::shared_ptr<MonitoringImpl> pImpl;
    };
  }
}
