/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <string>
#include <memory>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>

class SbRasterImage;

namespace RemoteViz
{
  namespace Rendering
  {

    class Connection;

    class RenderArea;

    class FrameEncoders;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  This class can be overridden by an application to receive notifications from a RenderArea.
     *
     * There are several categories of notifications:
     * - Connection: To know when a connection has been created, initialized and disposed.
     * - Render: To know the start/end of rendering and when an image is sent to the client.
     * - Input events: To handle MouseDown, MouseUp, etc. events triggered by the user.
     * - Messages: To handle messages sent from the client.
     *
     * For notifications related to the lifetime of the RenderArea itself, see
     * the ServiceListener class.
     *
     * Typically the application will attach a listener to the RenderArea in the
     * onInstantiatedRenderArea() method of its ServiceListener.  But applications can
     * also create a RenderArea explicitly and attach a listener.
     *
     * A typical sequence of calls to this listener is:
     * - #onOpenedConnection @BR
     *   Connection object has been created (isOpen will return true)
     * - #onInitializedConnection @BR
     *   Connection is fully initialized (e.g. calibration has finished)
     *   @BR @BR While application is rendering... @BR @BR
     *   - #onPreRender @BR
     *     Scene will be rendered
     *   - #onPostRender @BR
     *     Scene has been rendered
     *   - #onSendingFrame @BR
     *     Rendered image will be sent to client @BR @BR
     * - #onClosedConnection @BR
     *   Connection object has been disposed (isOpen will return false)
     *
     * [OIV-WRAPPER-CLASS DERIVABLE{Default},SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API RenderAreaListener
    {
    public:
      /**
       *  Destructor
       */
      virtual ~RenderAreaListener();
      /**
       *  Triggered when a client connects to the RenderArea.
       *
       *  Use this method to set the encoding format for images sent to the client,
       *  using the frameEncoders parameter.  The default is JPEG for interactive
       *  frames (i.e. while the user is interacting) and PNG for still frames.
       *  Video encoding (H264 or VP9) may provide better performance by reducing
       *  bandwidth requirements. Use the ConnectionSettings#isSupportedEncoders()
       *  method to query if the proposed encoders are available.
       *  If a requested frame encoder cannot be initialized,
       *  RenderAreaListener#onRefusedEncoder will be triggered.
       *  Otherwise, RenderAreaListener#onInitializedConnection will be triggered.
       *
       *  \param renderArea : the RenderArea requested by the connection
       *
       *  \param connection : the Connection
       *
       *  \param frameEncoders : frame encoders for interactive and still frames\n
       *                         Supported pairs of frame encoders (still/interactive):
       *                         JPEG/JPEG, PNG/JPEG, H264_NVENC/H264_NVENC,
       *                         H264_OPENH264/H264_OPENH264, VP9_VPX/VP9_VPX\n
       *                         Default behavior: PNG/JPEG.
       */
      virtual void onOpenedConnection(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> connection, std::shared_ptr<FrameEncoders> frameEncoders);
      /**
       *  Triggered when the connection and the frame encoders are initialized successfully.
       *  Use the FrameEncoders object to query which frame encoders will actually be used.
       *
       *  \param renderArea : the RenderArea requested by the connection
       *
       *  \param sender : the Connection
       *
       *  \param frameEncoders : frame encoders that have been initialized by the connection
       */
      virtual void onInitializedConnection(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, std::shared_ptr<const FrameEncoders> frameEncoders);
      /**
       *  Triggered when a connection is closed.
       *
       *  Default behavior : Dispose the renderArea if all its connections are closed
       *  and the application does not have a reference to the renderArea. If the
       *  application has a reference, it can use RenderArea#isDisposed() to know the state.
       *
       *  \param renderArea : the RenderArea
       *
       *  \param connectionId : ID identifying the connection
       *
       *  \param aborted : true if the network connection was closed unexpectedly by the client or has been lost.
       */
      virtual void onClosedConnection(std::shared_ptr<RenderArea> renderArea, const std::string& connectionId, bool aborted);
      /**
       *
       *  Triggered when a text message is received from a client.
       *  Default behavior : do nothing
       *
       *  \param renderArea : the RenderArea requested by the connection
       *
       *  \param sender : the Connection that receives the text message
       *
       *  \param message : the received message sent by the client
       */
      virtual void onReceivedMessage(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, const std::string& message);
      /**
       *
       *  Triggered when a binary message is received from a client.
       *  Default behavior : do nothing
       *
       *  \param renderArea : the RenderArea
       *
       *  \param sender : the Connection that receives the binary message
       *
       *  \param buffer : the received binary buffer sent by the client
       */
      virtual void onReceivedMessage(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, const std::vector<unsigned char>& buffer);
      /**
       *
       *  Triggered before a rendering is done.
       *  Default behavior : Do the rendering by returning true.
       *
       *  In the ServiceSettings#INDEPENDENT_SERVICE mode, this callback is never triggered.
       *
       *  \param renderArea : the RenderArea that will perform the rendering.
       *
       *  \param clearWindow : if true, this clears the graphics window before rendering
       *
       *  \param clearZbuffer : if true, the z buffer will be cleared before rendering
       *
       *  \return false to abort the rendering. \n If there are many listeners, the logical operator OR will 
       *  be applied on all the returned value of listeners. \n If there are no listeners to call, the rendering 
       *  will be not aborted.
       *
       * [OIVJAVA-WRAPPER-ARG IN,IN&OUT,IN&OUT]
       */
      virtual bool onPreRender(std::shared_ptr<RenderArea> renderArea, bool &clearWindow, bool &clearZbuffer);
      /**
       *  Triggered after a rendering is done.
       *  Default behavior : Do nothing.
       *
       *  In the ServiceSettings#INDEPENDENT_SERVICE mode, this callback is never triggered.
       *
       *  \param renderArea : the RenderArea that performed the rendering.
       *
       */
      virtual void onPostRender(std::shared_ptr<RenderArea> renderArea);
      /**
       *  Triggered when the renderArea is resized.  See RenderArea#resize().
       *  Default behavior : Do nothing.
       *
       *  \param renderArea : the resized RenderArea
       *
       *  \param width : new width of the renderArea
       *
       *  \param height : new height of the renderArea
       */
      virtual void onResize(std::shared_ptr<RenderArea> renderArea, unsigned int width, unsigned int height);
      /**
       *  Triggered when a client requests a new renderArea size.
       *  To resize the renderArea, call the method RenderArea#resize.\n
       *  Default behavior : Resize the renderArea to the requested size.
       *
       *  @see onResize, RenderArea#resize
       *
       *  \param renderArea : the RenderArea of the Connection
       *
       *  \param sender : the connection that made the size request
       *
       *  \param width : requested width
       *
       *  \param height : requested height
       */
      virtual void onRequestedSize(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, unsigned int width, unsigned int height);
      /**
       *  Triggered when a frame encoder cannot be initialized.
       *  If this case occurs, set another frame encoder. The status of encoders
       *  explains the reason for refusal.
       *
       *  Default behavior: Try to load other encoders automatically (fallback): 
       *  H264_NVENC/H264_NVENC => VP9_VPX/VP9_VPX => H264_OPENH264/H264_OPENH264 => PNG/JPEG => JPEG/JPEG
       *  Fallback directly to JPEG/JPEG if the pair of frame encoders are incompatible.
       *  If none of frame encoders can be initialized, the connection will be closed.
       *
       *  \param renderArea : the RenderArea of the Connection
       *
       *  \param sender : the connection that requests frame encoders
       *
       *  \param encoders : frame encoders refused by the connection
       */
      virtual void onRefusedEncoder(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, std::shared_ptr<FrameEncoders> encoders);
      //@{
      /**
       *  Triggered when a MouseUp event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param button : The button that was pressed when the mouse event was fired.
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseUp(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, SoMouseButtonEvent::Button button);
      /**
       *  Triggered when a MouseDown event is received from the client.\n
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param button : The button that was pressed when the mouse event was fired.
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseDown(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, SoMouseButtonEvent::Button button);
       /**
       *  Triggered when a MouseDoubleClick event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param button : The button that was pressed when the mouse event was fired.
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseDoubleClick(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, SoMouseButtonEvent::Button button);
      /**
       *  Triggered when a MouseMove event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseMove(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y);
      /**
       *  Triggered when a MouseEnter event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseEnter(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y);
      /**
       *  Triggered when a MouseLeave event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseLeave(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y);
      /**
       *  Triggered when a mouse wheel event (rotatation) is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param delta : abstract value which indicates how far the wheel turned
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onMouseWheel(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, int delta);
      /**
       *  Triggered when a KeyUp event is received from the client.
       *  Default behavior : accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param key : released key
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onKeyUp(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, const SoKeyboardEvent::Key& key);
      /**
       *  Triggered when a KeyDown event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \param key : pressed key
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onKeyDown(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, int x, int y, const SoKeyboardEvent::Key& key);
      /**
       *  Triggered when a TouchStart event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param id : touch identifier
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onTouchStart(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, unsigned int id, int x, int y);
      /**
       *  Triggered when a TouchMove event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param id : touch identifier
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onTouchMove(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, unsigned int id, int x, int y);
      /**
       *  Triggered when a TouchEnd event is received from the client.
       *  Default behavior : Accept the event by returning true.
       *
       *  \param renderArea : the RenderArea that will process the event
       *
       *  \param sender : the connection that sent the event
       *
       *  \param id : touch identifier
       *
       *  \param x : horizontal coordinate
       *
       *  \param y : vertical coordinate
       *
       *  \return true to process the event, this will apply an SoHandleEventAction to the scene graph. \n
       *  This return value does not have any effect when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled. \n
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners. \n 
       *  If there are no listeners to call, the event will be processed.
       */
      virtual bool onTouchEnd(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, unsigned int id, int x, int y);
      //@}

      /**
       *  Triggered when a new frame is requested by a connection.
       *  This method only works when the ServiceSettings#INDEPENDENT_SERVICE mode is enabled.
       *
       *  To send a new frame to the client, set the pixel buffer of the raster image using a buffer object.
       *  If the buffer object of the raster image is NULL, no frame will be sent to the client.
       *  IMPORTANT: This callback is executed in a separate thread for each connection requesting a frame.
       *
       *  @see ServiceSettings#setRunMode
       *
       *  \param renderArea : the RenderArea of the Connection
       *
       *  \param sender : the connection that requests a new frame
       *
       *  \param rasterImage : contains the RGB raster image to render.
       *
       *  \param isInteractive : This value can be modified to indicate if an user interaction is in progress. Default value is true.\n
       *                         If true (interactive frame), the frame will be compressed with the interactive frame encoder and a quality adapted to the bandwidth setting.\n
       *                         If false (still frame), the frame will be compressed with the still frame encoder and the best available quality.\n
       */
      virtual void onRequestedFrame(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, SbRasterImage* rasterImage, bool& isInteractive);

      /**
       *  Triggered before sending a frame to the client.
       *  The sending frame can be retrieved using Connection#getLastEncodedFrame.
       *  IMPORTANT: This callback is executed in a separate thread for each connection sending a frame.
       *
       *  \param renderArea : the RenderArea of the Connection
       *
       *  \param sender : the connection that will send the frame to the client
       *\if_cpp
       *  \param message : This value can be modified to attach a text message to the frame. Default value is an empty string.
       *                   This message can be retrieved from the client listeners: onReceivedFrame and onDecodedFrame.\n
       *\endif
       *\if_dotnet
       *  \param message : This value can be specified to attach a text message to the frame. Default value is an empty string.
       *                   This message can be retrieved from the client listeners: onReceivedFrame and onDecodedFrame.\n
       *\endif
       *\if_java
       *  \return a text message attached to the frame. Default value is an empty string.
       *          This message can be retrieved from the client listeners: onReceivedFrame and onDecodedFrame.\n
       *\endif
       */
      virtual void onSendingFrame(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Connection> sender, std::string& message);

    };


  }
}
