/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>
#include <SoDeprecationRules.h>

#include <RemoteViz/Rendering/HTTPResponse.h>

#include <string>
#include <memory>

namespace RemoteViz
{
  namespace Rendering
  {

    class RenderArea;

    class Client;

    class ConnectionParameters;

    class NetworkPerformance;

    class RenderAreaHardware;

    class HTTPRequest;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  This class can be overridden by an application to receive notifications from the rendering Service.
     *
     * A typical sequence of calls to this listener is:
     * - #onConnectedClient @BR
     *   Client object has been created (isConnected will return true)
     * - #onPendingCreateRenderArea @BR
     *   RenderArea object is about to be created
     * - #onInstantiatedRenderArea @BR
     *   RenderArea object has been created (isDisposed will return false)
     * - #onInitializedClient @BR
     *   Client is running and bandwidth calibration has finished
     *   @BR @BR Application is running... @BR @BR
     * - #onDisposingRenderArea @BR
     *   RenderArea object will be disposed
     * - #onDisposedRenderArea @BR
     *   RenderArea object has been disposed (isDisposed will return true)
     * - #onDisconnectedClient @BR
     *   Client object has been disposed (isConnected will return false)
     *
     * Since RemoteViz 10.6, this class also allows the Service to respond to standard HTTP requests
     * using standard HTTP response codes (and messages). This can be useful, for example, to respond
     * to "health check" requests. This method is not suitable to serve high volume requests or HTTP
     * resources, in these cases, prefer a dedicated HTTP server like Apache or Nginx.
     * See #onHTTPRequest().
     *
     * Typically the application will attach a listener to the Service instance before
     * calling the Service's open() method.
     *
     * For notifications about connections, rendering, input events and client messages see
     * the RenderAreaListener class.
     *
     *  [OIVNET-WRAPPER-CLASS DERIVABLE{Default},SHARED_POINTER_USE]
     *  [OIVJAVA-WRAPPER-CLASS DERIVABLE{Strong_Ref},SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API ServiceListener
    {
    public:
      /**
       *  Destructor
       */
      virtual ~ServiceListener();
      /**
       *  Triggered when a connection is pending and the requested renderArea does not exist.\n\n
       *  Default behavior : Accept the connection by returning true.
       *
       *  Use this callback to implement connection management, e.g. authentication.
       *  See the Authentication page and the \ref ConnectionManagement example.
       *  Accepting the connection means that RemoteViz will create a RenderArea and
       *  give it the id requested by the Client (renderAreaId).
       *
       *  \param renderAreaId : ID identifying the renderArea requested by the client
       *
       *  \param width : client requested width if the connection parameters contain this information, 
       *  otherwise it's the renderArea default width. This value can be modified, it determines the 
       *  created RenderArea width.
       *
       *  \param height : client requested height if the connection parameters contain this information, 
       *  otherwise it's the renderArea default height. This value can be modified, it determines the 
       *  created RenderArea height.
       *
       *  \param renderAreaHardware : hardware settings used to compute the render.
       *
       *  \param client : client that is being connected to the service. If the client does not exist yet, 
       *  this parameter will be null.
       *
       *  \param parameters : field-value pairs included in the url during the client connection. 
       *  See connectTo function in RemoteVizRenderArea.
       *
       *  \return true if the connection is accepted. The renderArea will be created.\n 
       *  Otherwise returns false. In this case, the client will be notified by an REFUSED disconnect message.\n 
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners.\n 
       *  If there are no listeners to call, the connection will be accepted.\n 
       * [OIVJAVA-WRAPPER CUSTOM_CODE]
       * [OIVNET-WRAPPER-ARG IN,IN&OUT,IN&OUT,IN&OUT,IN,IN]
       */
      virtual bool onPendingCreateRenderArea(const std::string& renderAreaId, unsigned int& width, unsigned int& height, std::shared_ptr<RenderAreaHardware> renderAreaHardware, std::shared_ptr<Client> client, std::shared_ptr<const ConnectionParameters> parameters);
      /**
       * 
       *  Triggered when a connection is pending and the requested renderArea exists.\n\n
       *  Default behavior : accept the connection by returning true. 
       *
       *  \param renderArea : the requested renderArea
       *
       *  \param client : client that is being connected to the service. If the client does not exist yet, 
       *  this parameter will be null.
       *
       *  \param parameters : field-value pairs included in the url during the client connection. 
       *  See connectTo function in RemoteVizRenderArea.
       *
       *  \return true if the connection is accepted, otherwise returns false. 
       *  In this case, the client will be notified by an REFUSED disconnect message.\n 
       *  If there are many listeners, the logical operator OR will be applied on all the returned value of listeners.\n 
       *  If there are no listeners to call, the connection will be accepted.\n 
       */
      virtual bool onPendingShareRenderArea(std::shared_ptr<RenderArea> renderArea, std::shared_ptr<Client> client, std::shared_ptr<const ConnectionParameters> parameters);
      /**
       *
       *  Triggered when a renderArea has been instantiated.\n\n
       *  Default behavior : do nothing
       *
       *  \param renderArea : the instantiated renderArea
       *
       * [OIVJAVA-WRAPPER NATIVE_HELPER_BEGIN{refRenderArea(renderAreaPtr)}]
       */
      virtual void onInstantiatedRenderArea(std::shared_ptr<RenderArea> renderArea);
      /**
       *  Triggered before a renderArea is disposed. See RenderArea#closeConnectionsAndDispose.
       *  After disposing the renderArea, the object will be considered as "disposed" (see RenderArea#isDisposed)
       *  and will never be reused by RemoteViz internally.
       *  This listener is called before onDisposedRenderArea.\n\n
       *  Default behavior : do nothing
       *
       *  \param renderArea : the renderArea that will be disposed
       *
       * [OIVJAVA-WRAPPER NATIVE_HELPER_END{unrefRenderArea(renderAreaPtr)}]
       */
      virtual void onDisposingRenderArea(std::shared_ptr<RenderArea> renderArea);
      /**
       *  Triggered when a renderArea has been disposed. See RenderArea#closeConnectionsAndDispose.
       *  At this point, the resources of the renderArea have been released.
       *  This listener is called after onDisposingRenderArea.\n\n
       *  Default behavior : do nothing
       *
       *  \param renderAreaId : ID identifying the disposed renderArea
       */
      virtual void onDisposedRenderArea(const std::string& renderAreaId);
      /**
       *
       *  Triggered when a client is connected to the service.\n\n
       *  Default behavior : do nothing
       *
       *  If network performance calibration is enabled, the network bandwidth
       *  is measured after this call and before calling #onInitializedClient.
       *  Therefore the NetworkPerformance queries getBandwidth and getLatency
       *  will return zero at this point. Use the queries in onInitializedClient
       *  to get the measured values.
       *
       *  Calibration is enabled by default. Calibration can be disabled using
       *  the networkPerformance parameter (see NetworkPerformance#enableCalibration).
       *  The duration of the calibration measurement (default 5 seconds) can also
       *  be modified (see NetworkPerformance#setMaxCalibrationDuration).
       *
       *  \param clientId : ID identifying the client
       *
       *  \param networkPerformance : network performance for the client.
       */
      virtual void onConnectedClient(const std::string& clientId, std::shared_ptr<NetworkPerformance> networkPerformance);
      /**
       *
       *  Triggered when a client is initialized.
       *
       *  Default behavior : If network performance calibration is enabled
       *  NetworkPerformance#enableCalibration), the client bandwidth value (ClientSettings#setBandwidth)
       *  is set to the measured network bandwidth value (NetworkPerformance#getBandwidth).
       *
       *  If calibration is enabled, the network performance measures can be retrieved
       *  using the networkPerformance parameter (see NetworkPerformance#getBandwidth
       *  and NetworkPerformance#getLatency).
       *
       *  The client bandwidth value (ClientSettings#setBandwidth) influences the quality
       *  and the number of interactive frames sent to the client (see
       *  ConnectionSettings::setFrameEncodingPolicy). The higher the value, the better
       *  the quality and the number of interactive frames.
       *
       *  \param client : client that is being connected to the service.
       *
       *  \param networkPerformance : network performance for the client.
       */
      virtual void onInitializedClient(std::shared_ptr<Client> client, std::shared_ptr<const NetworkPerformance> networkPerformance);
      /**
       *
       *  Triggered when a client is disconnected from the service.\n\n
       *  Default behavior : do nothing
       *
       *  \param clientId : ID identifying the client
       */
      virtual void onDisconnectedClient(const std::string& clientId);
#if SoDEPRECATED_BEGIN(10900)
      /**
       *  This listener is deprecated from Open Inventor 10.9 and is not triggered by RemoteViz anymore.
       *
       *  Triggered when a new connection is requested and a RemoteViz license is not found.
       *  In this callback, if you release one RemoteViz license (for instance, closing another connection),
       *  the new connection will then be opened.
       *  Otherwise, a LICENSE disconnect message will be sent to the client (which can choose to
       *  notify the user).
       *
       *  IMPORTANT: This callback is executed in a separate thread.
       *
       *  RemoteViz is protected by a license key mechanism limiting its use to specified computers 
       *  or network environments based on commercial agreements. RemoteViz uses floating licenses
       *  managed by a license server or an encrypted "master password" string (see SoLockManager).
       *
       *  Default behavior : Post an error message.
       *
       */
      SoDEPRECATED_METHOD(10900, "This listener is not triggered anymore because connections do not require RemoteViz licenses from Open Inventor 10.9")
      virtual void onMissingLicense(const std::string& renderAreaId, std::shared_ptr<const ConnectionParameters> parameters);
#endif /** @DEPRECATED_END */
      /**
       * Triggered when the secure connection is enabled. 
       * This callback has to return the private key passphrase.
       *
       * If the private key does not contain a passphrase, do not implement this callback.
       * The default implementation will return an empty string.
       *
       * Security warning !! DO NOT hard-code the passphrase into this callback !!\n
       * Read it from a SECURE location on your system.\n\n
       *
       * @see ServiceSettings#enableSecureConnection
       *
       * Default behavior : returns an empty string.
       */
      virtual std::string onRequestedPrivateKeyPassphrase();

      /**
       * Triggered when an HTTP connection is made that does not attempt to upgrade the connection to the WebSocket protocol.
       *
       * This allows a service to respond to these requests with regular HTTP responses.
       * It can be useful, for example, to manage health checks:
       *
       *   Health checks are usually used with an external monitoring service or container orchestrator to check the status of the service.
       *   They periodically invoke the endpoint to check the health of the service instance.
       *
       *   In a Kubernetes environment, a pair of health checks are used that distinguish between two states:
       *   - The service is functioning but not yet ready to receive requests. This state is the service's readiness.
       *   - The service is functioning and responding to requests. This state is the service's liveness.
       *
       * This method is not suitable to serve high volume requests or HTTP resources, in these cases, prefer a dedicated HTTP server like Apache or Nginx. 
       * Only HTTP 1.0 features are supported (status code, method, content type and headers). Connections are terminated immediately after the response.
       *
       * Default behavior : returns an HTTPResponse with "404 NOT FOUND" status.
       *
       * IMPORTANT: This listener is executed in a separate thread. 
       * It gives your program the chance to perform longer calculations without blocking other network operations.
       *
       * \param httpRequest HTTP request message
       *
       * \return HTTP response message
       * [OIVJAVA-WRAPPER CUSTOM_CODE]
       */
      virtual HTTPResponse onHTTPRequest(std::shared_ptr<const HTTPRequest> httpRequest);
    };
  }
}
