/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Jan 2013)
**=======================================================================*/
#ifndef _SO_DEPRECATION_RULES_H_
#define _SO_DEPRECATION_RULES_H_

#include <SoVersion.h>

#ifndef __has_feature
#define __has_feature(x) 0
#endif

#ifndef __has_extension
#define __has_extension(x) __has_feature
#endif


/**
 * Used to hide APIs deprecated after the given version.
 * This will cause an error to be signaled at compile time if a deprecated API is used.
 *
 * By default it is defined to 0.
 *
 * @NOTES
 *    - all visible deprecated API will generate a warning if used (unless it is deactivated at compiler flag level).
 *    - all demos/contrib provided by the toolkit do not use any deprecated API.
 *    - IMPORTANT : A build which use SoDEPRECATED_ERROR_VERSION != 0 must not be run!
 *    It may fail silently due to inconsistent declarations in the libraries and user code.
 *
 */
#if defined(SoDEPRECATED_HIDDEN)
#define SoDEPRECATED_ERROR_VERSION (SO_INVENTOR_VERSION)

#elif defined(SO_VERSION_DEPRECATED_ERROR) && SO_VERSION_DEPRECATED_ERROR != 0
#define SoDEPRECATED_ERROR_VERSION SO_VERSION_DEPRECATED_ERROR*1000

#else
#define SoDEPRECATED_ERROR_VERSION 0
#endif

#ifndef HIDDEN_FROM_DOC

/**
 * Preprocessor block condition used in Open Inventor header to change visibility
 * of methods at compilation time. see #SoDEPRECATED_ERROR_VERSION for details.
 */
#define SoDEPRECATED_BEGIN(DeprecationVersion) (DeprecationVersion>SoDEPRECATED_ERROR_VERSION) \
  /** @DEPRECATED_BEGIN */

/**
 * Declare a protoype as deprecated starting with Open Inventor version DeprecationVersion and print the compiler message DeprecationMessage
 *
 * @NOTES
 *   It relies on compiler capability to declare deprecated methods.
 */
#if defined(SoDEPRECATED_NO_WARN)
  #define SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage)
#else
#if defined(_MSC_VER) && (_MSC_VER >= 1300)
#define SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage) __declspec(deprecated("Deprecated since OIV" #DeprecationVersion ". " DeprecationMessage " Will be removed on next major version."))
#elif defined(__GNUC__) && ((__GNUC__ >= 4) || ((__GNUC__ >= 3) && (__GNUC_MINOR__ >= 4)))
#if __has_extension(attribute_deprecated_with_message)
#define SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage) __attribute__ ((deprecated("Deprecated since OIV" #DeprecationVersion ". " DeprecationMessage)))
#else
#define SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage) __attribute__ ((deprecated()))
#endif
#else
#error "You need to define SoDEPRECATED_PROTO macro"
#endif
#endif

/**
 * Declare a complete class as deprecated starting with Open Inventor version DeprecationVersion and print the compiler message DeprecationMessage.
 * use this macro on class/struct with no inlined method (else a warning will be issued even if not used)
 *
 * @EXAMPLE
 * \code
 *   class SoDEPRECATED_CLASS(9300,"No more used") INVENTOR_API SoMyDeprecatedClass
 *   {
 *   }
 * \endcode
 */
#define SoDEPRECATED_CLASS(DeprecationVersion,DeprecationMessage) \
  SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage)

#if defined(SoDEPRECATED_NO_WARN)
  #define SoDEPRECATED_HEADER(DeprecationVersion, DeprecationMessage)
#else
#if _MSC_VER
  #define SoDEPRECATED_HEADER(DeprecationVersion, DeprecationMessage) \
    __pragma(message("warning C4996: Deprecated since OIV" #DeprecationVersion ". " DeprecationMessage " Will be removed in a future major version."))
#elif defined(__GNUC__) && ((__GNUC__ >= 4) || ((__GNUC__ >= 3) && (__GNUC_MINOR__ >= 4)))
  #define DO_PRAGMA(x) _Pragma(#x)
  #define SoDEPRECATED_HEADER(DeprecationVersion, DeprecationMessage) \
  DO_PRAGMA(message ("warning: Deprecated since OIV" #DeprecationVersion ". " DeprecationMessage " Will be removed in a future major version."))
#else
  #error "You need to define SoDEPRECATED_HEADER macro"
#endif
#endif

/**
 * Declare a complete class as deprecated starting with Open Inventor version DeprecationVersion but not print the compiler message DeprecationMessage.
 * use this macro on class/struct with no inlined method (else a warning will be issued even if not used)
 *
 * @EXAMPLE
 * \code
 *   class SoDEPRECATED_CLASS_NOWARN(9300,"No more used") INVENTOR_API SoMyDeprecatedClass
 *   {
 *   }
 * \endcode
 */
#define SoDEPRECATED_CLASS_NOWARN(DeprecationVersion,DeprecationMessage)

/**
 * Declare a method or a member as deprecated starting with Open Inventor version DeprecationVersion and print the compiler message DeprecationMessage
 *
 * @EXAMPLE
 * \code
 *   SoDEPRECATED_METHOD(9300,"Use simpleMethod instead.")
 *   void simpleMethod93();
 * \endcode
 */
#define SoDEPRECATED_METHOD(DeprecationVersion,DeprecationMessage) \
  SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage)

/**
 * Declare a method or a member as deprecated starting with Open Inventor version DeprecationVersion but not print the compiler message DeprecationMessage
 *
 * @EXAMPLE
 * \code
 *   SoDEPRECATED_METHOD_NOWARN(9300,"Use simpleMethod instead.")
 *   void simpleMethod93();
 * \endcode
 */
#define SoDEPRECATED_METHOD_NOWARN(DeprecationVersion,DeprecationMessage)

/**
 * Declare a member as deprecated starting with Open Inventor version DeprecationVersion.
 *
 * @EXAMPLE
 * \code
 *   SoDEPRECATED_MEMBER(9300,"Use accessor getMyMember() instead.")
 *   int m_myMember;
 * \endcode
 */
#define SoDEPRECATED_MEMBER(DeprecationVersion,DeprecationMessage) \
  SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage)

/**
 * Declare a member as deprecated starting with Open Inventor version DeprecationVersion but not print the compiler message DeprecationMessage.
 *
 * @EXAMPLE
 * \code
 *   SoDEPRECATED_MEMBER_NOWARN(9300,"Use accessor getMyMember() instead.")
 *   int m_myMember;
 * \endcode
 */
#define SoDEPRECATED_MEMBER_NOWARN(DeprecationVersion,DeprecationMessage) \

/**
 * Declare a typedef as deprecated starting with Open Inventor version DeprecationVersion.
 *
 * @EXAMPLE
 * \code
 *   typedef SoDEPRECATED_TYPEDEF(9300,"Use accessor getMyMember() instead.") int myTypeDef;
 * \endcode
 */
#define SoDEPRECATED_TYPEDEF(DeprecationVersion,DeprecationMessage) \
  SoDEPRECATED_PROTO(DeprecationVersion,DeprecationMessage)

/**
 * Declare enum value DeprecatedEnumValue as deprecated starting with Open Inventor version DeprecationVersion.
 *
 * @EXAMPLE
 * \code
 *   enum SoDEPRECATED_ENUM(9300,"No more supported use otherClass::otherEnum")
 *   MyEnum1 {
 *     VALUE0_1,
 *     VALUE1_1,
 *     VALUE2_1
 *    };
 * \endcode
 *
 * @NOTES
 *   No cpp compiler warning will be issued on usage of this enum.
 */
#define SoDEPRECATED_ENUM(DeprecationVersion,DeprecationMessage)

/**
 * Declare enum value DeprecatedEnumValue as deprecated starting with Open Inventor version DeprecationVersion.
 *
 * @EXAMPLE
 * \code
 *   enum MyEnum0 {
 *     VALUE0_0,
 *     SoDEPRECATED_ENUM_VALUE(9300,"No more supported.")
 *     VALUE1_0,
 *     VALUE2_0
 *   };
 * \endcode
 *
 * @NOTES
 *   No compiler warning will be issued.
 *
 */
#define SoDEPRECATED_ENUM_VALUE(DeprecationVersion,DeprecationMessage)

/**
 * Declare a field as deprecated starting with Open Inventor version DeprecationVersion but not print the compiler message DeprecationMessage.
 * Fields have this specificity that removing them break the iv file format compatibility.
 * Means that they can move to the private section, but cannot be totally removed for serialization purpose.
 *
 * @EXAMPLE
 * \code
 *   SoDEPRECATED_FIELD(9300,"Use a method or another field instead.")
 *   SoSFBool enableThisMode;
 * \endcode
 */
#define SoDEPRECATED_FIELD(DeprecationVersion,DeprecationMessage)\
  SoDEPRECATED_MEMBER_NOWARN(DeprecationVersion,DeprecationMessage)

#endif // HIDDEN_FROM_DOC

#endif // _SO_DEPRECATION_RULES_H_
