/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Dec 2008)
**=======================================================================*/



#ifndef _SO_ORTHOSLICE_DRAGGER_H_
#define _SO_ORTHOSLICE_DRAGGER_H_

#include <Inventor/draggers/SoTranslate1Dragger.h>
#include <Inventor/fields/SoSFPath.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/fields/SoSFColor.h>

class SoCallback;
class SoOrthoSlice;
class SoSensor;
class SoNodeSensor;
class SoFieldSensor;
class SoGetBoundingBoxAction;
namespace inventor{namespace helper{class SoOrthoSliceBorderHelper;}}

/**
* @VREXT Object that moves an SoOrthoSlice by dragging with the mouse.
* 
* @ingroup VolumeVizDraggers
* 
* @DESCRIPTION
*   SoOrthoSliceDragger is derived from SoTranslate1Dragger and has similar behavior,
*   but is specifically designed to allow direct manipulation of an SoOrthoSlice.
*   The difference is that this class uses the actual geometry of the ortho slice to
*   detect mouse clicks and automatically updates the slice's SoOrthoSlice::sliceNumber field as
*   the dragger is moved.  Conversely the dragger updates itself if the slice's axis
*   or sliceNumber fields are changed by the application.
*
*   The dragger will not do anything until the field #orthoSlicePath has been set.
*   The path can be a partial path, i.e. it does not need to start at the root of the scene graph.
*   However, it must include the SoOrthoSlice node.
*
*   The dragger always sets its "translator" part to be the actual ortho slice node.
*   When the dragger is created it sets its "translatorActive" part to be an empty
*   Separator.  You can reset this to something else.  You can attach callbacks to
*   this dragger in the usual way.
*
*   If the camera faces the slice (i.e. the orientation of the slice and the
*   camera are exactly the same), the dragger will automatically change its orientation
*   to another axis so that it can be manipulated (otherwise it can't). When the dragger
*   itself is displayed, this behaviour may be surprising, and it can be
*   disabled by setting the SoPreferences value
*   IVVR_ENABLE_ORTHOSLICE_DRAGGER_AXIS_SWITCH to FALSE.
*
* @EXAMPLE
* \if_cpp
* \code
*   SoSeparator* volSep = new SoSeparator();
*   root->addChild( volSep );
*
*   // Create and initialize data node
*   SoVolumeData* volData = new SoVolumeData();
*   volSep->addChild( volData );
*
*   // Create and initialize orthoslice node
*   SoOrthoSlice* orthoSlice = new SoOrthoSlice();
*   volSep->addChild( orthoSlice );
*
*   // Create path to slice node
*   // Note: Can be a partial path but must include the slice node.
*   SoPath* path = new SoPath( volSep );
*   path->append( orthoSlice );
*
*   // Create and initialize dragger
*   SoOrthoSliceDragger* dragger = new SoOrthoSliceDragger();
*   dragger->orthoSlicePath  = path;
*   volSep->addChild( dragger );
* \endcode
* \endif
* \if_dotnet
* \code
*   SoSeparator volSep = new SoSeparator();
*   root.AddChild(volSep);
*
*   // Create and initialize data node
*   SoVolumeData volData = new SoVolumeData();
*   volSep.AddChild( volData );
*
*   // Create and initialize orthoslice node
*   SoOrthoSlice orthoSlice = new SoOrthoSlice();
*   volSep.AddChild( orthoSlice );
*
*   // Create path to slice node
*   // Note: Can be a partial path but must include the slice node.
*   SoPath path = new SoPath(volSep);
*   path.Append( orthoSlice );
*
*   // Create and initialize dragger
*   SoOrthoSliceDragger dragger   = new SoOrthoSliceDragger();
*   dragger.orthoSlicePath.Value  = path;
*   volSep.AddChild(dragger);
* \endcode
* \endif
* \if_java
* \code
*   SoSeparator volSep = new SoSeparator();
*   root.addChild(volSep);
*
*   // Create and initialize data node
*   SoVolumeData volData = new SoVolumeData();
*   volSep.addChild( volData );
*
*   // Create and initialize orthoslice node
*   SoOrthoSlice orthoSlice = new SoOrthoSlice();
*   volSep.addChild( orthoSlice );
*
*   // Create path to slice node
*   // Note: Can be a partial path but must include the slice node.
*   SoPath path = new SoPath(volSep);
*   path.regular.append( orthoSlice );
*
*   // Create and initialize dragger
*   SoOrthoSliceDragger dragger = new SoOrthoSliceDragger();
*   dragger.orthoSlicePath.setValue( path );
*   volSep.addChild(dragger);
* \endcode
* \endif
*
* @FILE_FORMAT_DEFAULT
*    OrthoSliceDragger {
*    @TABLE_FILE_FORMAT
*       @TR orthoSlicePath       @TD NULL
*       @TR volumeDimension      @TD 0 0 0
*       @TR volumeExtent         @TD 0 0 0
*       @TR enableHighlight      @TD TRUE
*       @TR highlightColor       @TD 1.0 1.0 0
*    @TABLE_END
*    }
* 
* @SEE_ALSO
*    SoDragger,
*    SoTranslate1Dragger,
*    SoOrthoSlice
* 
*/
class VOLUMEVIZ_API SoOrthoSliceDragger : public SoTranslate1Dragger
{
  SO_KIT_HEADER(SoOrthoSliceDragger);

public:
  /**
   * Constructor.
   */
  SoOrthoSliceDragger();

  /**
   * Path to the SoOrthoSlice node to drag. @BR
   * Default is empty (no path).
   */
  SoSFPath orthoSlicePath;

#if SoDEPRECATED_BEGIN(101100)

  /**
   * Dimension of the volume.
   */
  SoDEPRECATED_FIELD(101100, "This field is ignored. The dimension is fetched automatically from the SoDataSet specified by the field dataSetId of the SoOrthoSlice handled by this dragger.")
  SoSFVec3i32 volumeDimension;

  /**
   * Extent of the volume.
   */
  SoDEPRECATED_FIELD(101100, "This field is ignored. The extent is fetched automatically from the SoDataSet specified by the field dataSetId of the SoOrthoSlice handled by this dragger.")
  SoSFBox3f volumeExtent;

#endif /** @DEPRECATED_END */

  /**
   * Specifies the highlight color. See also #enableHighlight.
   * The default color is yellow (1,1,0).
   */
  SoSFColor highlightColor;

  /**
   * Enables highlighting.
   * If this field is true, during a mousemouve, a border is drawn and highlighted according to the #highlightColor
   * when the cursor moves over the slice.
   * This border disappears once the cursor is no longer over the slice.
   * Default is true.
   */
  SoSFBool enableHighlight;

SoINTERNAL public:
  /**
   * Database class init.
   */
  static void initClass();
  /**
   * Database class exit.
   */
  static void exitClass();

  virtual void internalRemoveChild(int);
  virtual void internalRemoveChild(SoNode*);
  virtual void internalRemoveAllChildren();
  virtual void internalAddChild(SoNode*);
  virtual void internalInsertChild(SoNode*, int);
  virtual void internalReplaceChild(int, SoNode*);
  virtual void internalReplaceChild(SoNode*, SoNode*);

SoEXTENDER public:
  /**
   * Called when the dragger gains or loses status as "grabber" of events.
   */
  virtual void grabEventsSetup();

  virtual void callback(SoCallbackAction* action);
  virtual void getBoundingBox(SoGetBoundingBoxAction* action);
  virtual void getMatrix(SoGetMatrixAction* action);
  virtual void rayPick(SoRayPickAction* action);

protected:
  /**
   * Destructor.
   */
  virtual ~SoOrthoSliceDragger();

  /**
   * Manage notification.
   */
  virtual void notify(SoNotList *list);

  void handleEvent(SoHandleEventAction *action) override;

  /** @copydoc SoTranslate1Dragger::readInstance() */
  virtual SbBool readInstance(SoInput* in, unsigned short flags);

private:
  /**
   * Synchronize slice number from dragger position.
   */
  static void valueChangedCB( void *data, SoDragger *dragger);

  /**
   * Associate dragger with an ortho slice and volume
   * Note: We need a path to the ortho slice node because we're going
   * to use the slice as proxy geometry for the dragger
   */
  void initialize( const SoPath *pathToSlice);

  /**
   * Synchronize dragger position from slice number.
   */
  void synchronize();

  /**
   * set dragger direction.
   */
  void setDraggerDirection();

  void GLRender(SoGLRenderAction* action) override;

  bool pushDataSetModelMatrix(SoAction* action);

private:
  // current handled orthoSlice.
  SoOrthoSlice* m_orthoSlice;
  SoCallback* m_updateBorderCBNode;
  SoCallback* m_orthoSliceMatrixCBNode;
  bool m_useDataSetMatrix;
  SbMatrix m_savedDataSetMatrix;
  int m_sliceAxis;
  int m_sliceNumber;

  // Enables/disables axis switch when camera faces the dragger
  // ( see m_isOrthoSliceInSamePlaneThanViewer )
  static bool s_enableAxisSwitch;

  // Detect if orthoSlice is in the same plane than the viewer
  // If true then draggerDirection is changed to be able to move the slices.
  bool m_isOrthoSliceInSamePlaneThanViewer;

  // Track changes to SoOrthoSlice node
  SoNodeSensor* m_sliceChangedSensor;
  static void sliceChangedSensorCB( void *, SoSensor * );
  static void sliceDeletedSensorCB( void *, SoSensor * );
  
  // Used to avoid recursive notification.
  static bool m_isNotifying;

  inventor::helper::SoOrthoSliceBorderHelper *m_orthoSliceBorderHelper;
  void drawHighLight(const bool enable);
  bool m_drawHightLight;
  SoFieldSensor* m_onValueChangedFieldSensorEnableBorder;
  SoFieldSensor* m_onValueChangedFieldSensorEnableImage;

  void setupAutoValues(SoState* state);
  SbVec3i32 m_autoVolumeDimensions;
  SbBox3f m_autoVolumeExtent;
};

#endif //_SO_ORTHOSLICE_DRAGGER_H_


