/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author : Julien SALLANNE (Dec 2012)
**=======================================================================*/

#ifndef  _SO_HEIGHTFIELD_RENDER_H
#define  _SO_HEIGHTFIELD_RENDER_H

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/SbBox.h>
#include <VolumeViz/nodes/SoSlice.h>
#include <Inventor/fields/SoSFColor.h>

class SoHeightFieldGeometry;
class SoHeightFieldProperty;
class SoHeightFieldRenderImpl;
class SoVolumeStateHeightField;

/**
 *
 * @VREXT Heightfield rendering node.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 * SoHeightFieldRender displays a uniform grid in the XY plane whose vertices are
 * height (Z) values stored in 2D LDM format (any LDM data set with the Z dimension
 * equal to 1). Adding the combination of LDM data management with advanced GPU features
 * provides a way to handle extremely large surfaces. Just as with volume data, LDM
 * uses tiles of data and multiple levels of resolution to enable interactive frame
 * rates even for data sets that cannot fit in system memory.
 *
 * This node is used similarly to other VolumeViz shapes, but instead of an SoVolumeData node,
 * you use an SoHeightFieldGeometry for the data set (height values) and optionally one or more
 * SoHeightFieldProperty nodes for property data sets. When one or more SoHeightFieldProperty
 * nodes are used, because there are multiple data nodes, they must be children of an
 * SoMultiDataSeparator node and the rules for multiple data sets apply (see SoVolumeData the
 * parent class of SoHeightFieldGeometry). Otherwise, if no SoHeightFieldProperty nodes
 * are used and only an SoHeightFieldGeometry node is present, the following rules apply:
 * - If an SoTransferFunction node is on the state, then the SoHeightFieldGeometry node
 *   is also used as a property node and the height field is colored according to the height
 *   values and the transfer function, just as if an SoHeightFieldProperty node was defined
 *   with the same data.
 * - If no SoTransferFunction node is on the state, then the height field is colored using
 *   the current material on state.
 *
 * Note that SoHeightFieldRender is derived from SoSlice and those inherited fields
 * affect the rendering. In particular note that the @I alphaUse@i field applies to
 * height field rendering and the default value (since Open Inventor 10.0) is
 * ALPHA_OPAQUE. It means, for example, that the transparency in the color map will not
 * affect rendering of the height map unless alphaUse is set to ALPHA_AS_IS.
 *
 * Data set values are converted to height values in 3D space in two ways depending
 * on the data type:
 * - Integer values are normalized between [0,1]  ([-1,1] for signed types)
 *   based on the range of values for the specific data type.  For example, for
 *   UNSIGNED BYTE values the range 0..255 is mapped to 0..1.
 *
 * - Floating point values are not normalized (are used "as is").
 *
 * Any height value in the SoHeightFieldGeometry data set that is equal to the
 * "undefined" value will be rendered as a hole in the mesh. The undefined value can be
 * specified during the LDM conversion using the "-u" option to the LDM converter:
 * \verbatim
      convert -u 127 -b 1 inputFile.lst
   \endverbatim
 * Or by setting the @I undefinedValue@i field of the SoHeightFieldGeometry node.
 * The default value is NaN (Not a Number).
 *
 * An SoHeightFieldPropertyMask node can also be used to specify undefined cells in the
 * mesh. An undefined cell effectively removes the four corresponding height values
 * from the mesh. (SoVolumeMask does not apply to height field rendering.)
 *
 * @TABLE_1B
 * @TR @IMAGE horizon.jpg
 * @TR A lighted heightfield
 * @TABLE_END
 *
 * VolumeViz provides default shaders that conveniently color the surface using a single
 * property, as shown in the images. However, it is also possible to combine multiple
 * properties using a custom shader program, in the same way that you would combine multiple
 * volumes.
 *
 * The field #cellOutline enables drawing the edges of the mesh cells.
 * BoundaryCells are cells close to an undefined value. If the #boundaryCells field is set to ALWAYS,
 * these cells must always be considered at all resolution levels to avoid artifacts.
 * In SMART mode, the default, we don't take this into account for distant views.
 *
 * Normally this node uses the OpenGL tessellation shader extension to speed up
 * rendering and automatically adjust the number of generated triangles depending on
 * the camera position and orientation.
 * In this mode, to change the number of generated triangles, use an SoComplexity node.
 * A value of 1 means a full tessellation with a maximum of 4 triangles per pixel
 * and a value of 0.5 means a maximum of 1 triangle per pixel.
 *
 * If tessellation shaders are not available, a value of 1 means full resolution data is used to 
 * generate triangles (a grid of 10x10 will then generate 10x10 x2 triangles). A value of 0 means
 * the lower resolution will be used.
 * 
 * To check if tessellation shaders are supported, use the method isAvailable( "GL_ARB_tessellation_shader" )
 * of SoGLExtension class
 *
 * When no SoHeightFieldProperty nodes are used and only an SoHeightFieldGeometry node is defined,
 * the standard VolumeViz rules apply to the creation of a scene graph using SoHeightFieldRender.
 * A minimal scene graph displaying a height field in this case is:
 *
 * @EXAMPLE
 * \if_cpp
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry* HFGeom = new SoHeightFieldGeometry;
 *     HFGeom->fileName = "$OIVHOME/examples/data/VolumeViz/horizon.ldm";
 *
 *   // Base color
 *   SoMaterial* material = new SoMaterial;
 *     material->diffuseColor.setValue( 1, 1, 1 );
 *
 *   SoTransferFunction* TF = new SoTransferFunction;
 *     TF->predefColorMap = SoTransferFunction::STANDARD;
 *
 *   SoHeightFieldRender* HFRend = new SoHeightFieldRender;
 *
 *   // Limit triangles (optional)
 *   SoComplexity* complexity = new SoComplexity();
 *     complexity->value = 0.25;
 *
 *   // Build scene graph
 *   SoSeparator* volSep = new SoSeparator;
 *     volSep->addChild( HFGeom );
 *     volSep->addChild( material );
 *     volSep->addChild( complexity );
 *     volSep->addChild( TF );
 *     volSep->addChild( HFRend );
 *   root->addChild( volSep );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry HFGeom = new SoHeightFieldGeometry();
 *     HFGeom.fileName.Value = "$OIVHOME/examples/data/VolumeViz/horizon.ldm";
 *
 *   SoMaterial Material = new SoMaterial();
 *     Material.diffuseColor.SetValue(1, 1, 1);
 *
 *   SoTransferFunction TF = new SoTransferFunction();
 *     TF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.STANDARD;
 *
 *   SoHeightFieldRender HFRend = new SoHeightFieldRender();
 *
 *   // Limit triangles
 *   SoComplexity Complexity = new SoComplexity();
 *     Complexity.value.Value = 0.25;
 *
 *   // Build scene graph
 *   SoSeparator volSep = new SoSeparator();
 *     volSep.AddChild(HFGeom);
 *     volSep.AddChild(Material);
 *     volSep.AddChild(Complexity);
 *     volSep.AddChild(TF);
 *     volSep.AddChild(HFRend);
 *   root.AddChild( volSep );
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry HFGeom = new SoHeightFieldGeometry();
 *     HFGeom.fileName.setValue( "$OIVHOME/examples/data/VolumeViz/horizon.ldm" );
 *
 *   SoMaterial material = new SoMaterial ();
 *     material.diffuseColor.setValue( 1, 1, 1 );
 *
 *   SoTransferFunction TF = new SoTransferFunction();
 *     TF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.STANDARD );
 *
 *   SoHeightFieldRender HFRend = new SoHeightFieldRender();
 *
 *   // Limit triangles (optional)
 *   SoComplexity complexity = new SoComplexity();
 *     complexity.value.setValue( 0.25 );
 *
 *   // Build scene graph
 *   SoSeparator volSep = new SoSeparator();
 *     volSep.addChild( HFGeom );
 *     volSep.addChild( material );
 *     volSep.addChild( complexity );
 *     volSep.addChild( TF );
 *     volSep.addChild( HFRend );
 *   root.addChild( volSep );
 * \endcode
 * \endif
 * \htmlonly </UL> \endhtmlonly
 *
 * When one or more SoHeightFieldProperty nodes are used, multidata rules apply to
 * the creation of a scene graph using SoHeightFieldRender and an SoMultiDataSeparator
 * node @I must@i be used instead of SoSeparator.
 * A minimal scene graph displaying a heightfield in this case is:
 *
 * @EXAMPLE
 * \if_cpp
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry* HFGeom = new SoHeightFieldGeometry;
 *     HFGeom->fileName = "$OIVHOME/examples/data/VolumeViz/horizon.ldm";
 *     HFGeom->dataSetId = 1;
 *
 *   SoHeightFieldProperty* HFProp = new SoHeightFieldProperty;
 *     HFProp->fileName = "$OIVHOME/examples/data/VolumeViz/property1.ldm";
 *     HFProp->dataSetId = 2;
 *
 *   // Base color
 *   SoMaterial* material = new SoMaterial;
 *     material->diffuseColor.setValue( 1, 1, 1 );
 *
 *   SoTransferFunction* TF = new SoTransferFunction;
 *     TF->predefColorMap = SoTransferFunction::STANDARD;
 *
 *   SoHeightFieldRender* HFRend = new SoHeightFieldRender;
 *
 *   // Limit triangles (optional)
 *   SoComplexity* complexity = new SoComplexity();
 *     complexity->value = 0.25;
 *
 *   // Build scene graph
 *   SoMultiDataSeparator* volSep = new SoMultiDataSeparator;
 *     volSep->addChild( HFGeom );
 *     volSep->addChild( HFProp );
 *     volSep->addChild( material );
 *     volSep->addChild( complexity );
 *     volSep->addChild( TF );
 *     volSep->addChild( HFRend );
 *   root->addChild( volSep );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry HFGeom = new SoHeightFieldGeometry();
 *     HFGeom.fileName.Value = "$OIVHOME/examples/data/VolumeViz/horizon.ldm";
 *     HFGeom.dataSetId.Value = 1;
 *
 *   SoHeightFieldProperty HFProp = new SoHeightFieldProperty();
 *     HFProp.fileName.Value = "$OIVHOME/examples/data/VolumeViz/horizon.ldm";
 *     HFProp.dataSetId.Value = 2;
 *
 *   SoMaterial Material = new SoMaterial();
 *     Material.diffuseColor.SetValue(1, 1, 1);
 *
 *   SoTransferFunction TF = new SoTransferFunction();
 *     TF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.STANDARD;
 *
 *   SoHeightFieldRender HFRend = new SoHeightFieldRender();
 *
 *   // Limit triangles
 *   SoComplexity Complexity = new SoComplexity();
 *     Complexity.value.Value = 0.25;
 *
 *   // Build scene graph
 *   SoMultiDataSeparator volSep = new SoMultiDataSeparator();
 *     volSep.AddChild(HFGeom);
 *     volSep.AddChild(HFProp);
 *     volSep.AddChild(Material);
 *     volSep.AddChild(Complexity);
 *     volSep.AddChild(TF);
 *     volSep.AddChild(HFRend);
 *   root.AddChild( volSep );
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Create nodes
 *   SoHeightFieldGeometry HFGeom = new SoHeightFieldGeometry();
 *     HFGeom.fileName.setValue( "$OIVHOME/examples/data/VolumeViz/horizon.ldm" );
 *     HFGeom.dataSetId.setValue( 1 );
 *
 *   SoHeightFieldProperty HFProp = new SoHeightFieldProperty();
 *     HFProp.fileName.setValue( "$OIVHOME/examples/data/VolumeViz/horizon.ldm" );
 *     HFProp.dataSetId.setValue( 2 );
 *
 *   SoMaterial material = new SoMaterial ();
 *     material.diffuseColor.setValue( 1, 1, 1 );
 *
 *   SoTransferFunction TF = new SoTransferFunction();
 *     TF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.STANDARD );
 *
 *   SoHeightFieldRender HFRend = new SoHeightFieldRender();
 *
 *   // Limit triangles (optional)
 *   SoComplexity complexity = new SoComplexity();
 *     complexity.value.setValue( 0.25 );
 *
 *   // Build scene graph
 *   SoMultiDataSeparator volSep = new SoMultiDataSeparator();
 *     volSep.addChild( HFGeom );
 *     volSep.addChild( HFProp );
 *     volSep.addChild( material );
 *     volSep.addChild( complexity );
 *     volSep.addChild( TF );
 *     volSep.addChild( HFRend );
 *   root.addChild( volSep );             
 * \endcode
 * \endif
 * \htmlonly </UL> \endhtmlonly
 *
 * @B Shaders@b
 *
 * When used with an SoVolumeShader, a new shader function is available to compute
 * lighting:
 *  - @I@B vec4 VVizComputeFrontColor(vec3 normal, vec4 color))@b@i:
 *   Add lighting to the color @I col@i.
 *
 * The following shader code will light a heightfield:
 * \par
 * \code
 * //!oiv_include <VolumeViz/vvizCoordinates_frag.h>
 * //!oiv_include <VolumeViz/vvizfnc_frag.h>
 * //!oiv_include <VolumeViz/vvizCombine_frag.h>
 * //!oiv_include <VolumeViz/vvizComputeFragmentColor_frag.h>
 * //!oiv_include <VolumeViz/vvizOutputColor_frag.h>
 *
 * vec4 VVizComputeFrontColor(vec3 n, vec4 col);
 *
 * void main()
 * {
 *   vec3 tCoord0 = VVizComputeCoordinates(OivFragmentTexCoord(0).xyz);
 *   vec3 normal = normalize(VVizComputeNormal(texCoord));
 *
 *   float sf = VVizCombineData(tCoord0);
 *   vec4 col = VVizComputeFragmentColor(vox, texCoord);
 *
 *   col = VVizComputeFrontColor(normal, col);
 *   VVizOutputColor(col);
 * }\endcode
 *
 * @B Multiple Data@b:
 *
 * Using an SoMultiDataSeparator, it is possible to combine datasets that have
 * different dimensions or tile sizes. For instance, an SoHeightFieldGeometry
 * dataset can be used with an SoHeightFieldProperty even if they don't have the
 * same dimensions or tile sizes.
 *
 * @B Composition with VolumeData@b:
 *
 * It is possible to color a height field according to the values of a volume
 * data. To do such rendering, the HeightField datasets and the SoVolumeData
 * nodes must be in the same SoMultiDataSeparator. A custom shader must also be
 * defined to fetch values from the right datasets using their dataSetId. This
 * is demonstrated in the HorizonInVolume example.
 *
 * In addition, texture coordinates conversion functions are provided in the
 * @I@B VolumeViz/vvizStructure.h@b@i shader include in order to help fetch
 * the correct VolumeData values in custom shaders. @BR
 * For instance,
 * \code
 * vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 * \endcode
 * can be used to convert texture coordinates related to one dataset to texture
 * coordinates related to another dataset. @BR
 * The conversion is based solely on the transformations applied to each
 * dataset, which are defined by their model matrix and their extent. @BR
 * Please note that the model matrix of a dataset is defined by the
 * SoTransformation nodes that are placed @B before @b the SoDataSet node in the
 * order of the traversal.
 *
 * @B Picking@b:
 *
 *  Similar to other geometry, SoPickedPoint can return an SoDetail object specific to the SoHeightFieldRender
 *  node.  The specific class is SoHeightFieldDetail.
 *
 *  Only GPU picking is supported. This means that the SoRayPickAction used for picking must have its scene manager
 *  initialized using the method SoAction::setSceneManager(). SoHandleEventAction does this automatically, so it is
 *  not necessary for the application to take any action when using (for example) an SoEventCallback node and
 *  calling the getPickedPoint() method. However if the application creates its own SoRayPickAction then it must
 *  set the scene manager. If no scene manager is specified, a warning message is issued.
 *
 * @B Limitations: @b
 * @UL
 *  @LI At least one SoHeightFieldProperty which defines a property data set associated with the grid
 *      must be in the state.
 *
 *  @LI If an SoROI is in the state, SoHeightFieldRender renders only one box of the ROI
 *      (ie: correct rendering only if SoROI's flag field is set to SUB_VOLUME)
 *
 *  @LI Only SoHeightFieldGeometry data sets with a depth of 1 (Z dimension = 1) can be rendered.
 *
 *  @LI The SoHeightFieldProperty and SoHeightFieldPropertyMask data sets must have exactly the same dimensions
 *      as the SoHeightFieldGeometry.
 *
 *  @LI Triangle orientation after GPU tessellation cannot be controlled. Undefined value rendering is indeterministic.
 *
 *  @LI For low resolution, undefined values can be taken into account in the property but not in the geometry.
 *      In such cases, coloring artifacts can appear.
 *
 *  @LI Lighting is limited to directional lights and base color.
 *      Other kinds of lights can be implemented manually using a custom shader.
 *
 *  @LI By default, two-sided lighting is not enabled ("back" side of surface will not be lighted).
 *      Use an SoShapeHints node to enable two-sided lighting.
 *
 *  @LI The #enableBumpMapping field is not supported on this node
 *
 *  @LI SoHeightFieldPropertyMask and #cellOutline can be used only if tesselation shaders are supported.
 *      Use the SoGLExtension method isAvailable( "GL_ARB_tessellation_shader" ) to check this support.
 * @ul
 *
 * @FILE_FORMAT_DEFAULT
 *    HeightFieldRender {
 *    @TABLE_FILE_FORMAT
 *       @TR alphaUse        @TD ALPHA_OPAQUE
 *       @TR alternateRep    @TD NULL
 *       @TR bumpScale       @TD 1.0
 *       @TR cellOutline     @TD FALSE
 *       @TR enableBumpMapping @TD FALSE
 *       @TR useRGBA         @TD FALSE
 *       @TR largeSliceSupport @TD FALSE
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SoHeightFieldProperty,
 *    SoHeightFieldPropertyMask,
 *    SoHeightFieldRender,
 *    SoHeightFieldGeometry,
 *    SoMultiDataSeparator
*/
class VOLUMEVIZ_API SoHeightFieldRender : public SoSlice
{
  SO_NODE_HEADER( SoHeightFieldRender );

public:

  /**
   * If true, draw outline of each heightField cell (default is false).
   */
  SoSFBool cellOutline;

  /**
  * When #cellOutline is TRUE, this value specifies the cell outline width in pixels.
  * Default is 2 pixels.
  *
  * @FIELD_SINCE_OIV 9.7
  */
  SoSFFloat cellOutlineWidth;

  /**
  * When #cellOutline is TRUE, this value specifies the cell outline color.
  * Default is black : (0, 0, 0).
  *
  * @FIELD_SINCE_OIV 9.7
  */
  SoSFColor cellOutlineColor;

  /** Boundary cells mode. */
  enum BoundaryCells
  {
    /** Always compute boundary cells. */
    ALWAYS,

    /** Computes boundary cells according to the distance to the view. */
    SMART,

    /** Never computes boundary cells. */
    NONE
  };

  /**
   * Boundary cells mode.
   * Setting this field to SMART will not compute boundary cells for distant views.
   * @useenum{BoundaryCells}. Default is SMART.
   *
   */
  SoSFEnum boundaryCells;

  /**
   *  Constructor
   */
  SoHeightFieldRender();

  /**
   * Returns true if graphic card can render an SoHeightFieldRender.
   * GPU must support geometry shaders, floating point textures and
   * vertex buffer objects (VBO) and tessellation shaders
   */
  static SbBool isSupported(SoState* state=NULL);

SoEXTENDER public:

  /** Computes the bounding box */
  virtual void computeBBox(SoAction *, SbBox3f &box, SbVec3f &center);

  /** Depth of drawn horizon is saved in a texture and then, the depth of the picked point is recovered
      With screen coordinate, */
  virtual void rayPick(SoRayPickAction* action);

  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /**
   * Handles field change
   */
  virtual void notify(SoNotList *list);

  /**
   * Return the volumeState associated to this node.
   */
  SoVolumeStateHeightField* getVolumeStateHeightField() const;

protected:
  /** Generate primitive for the raypick action */
  virtual void generatePrimitives(SoAction *action);

  /** Render */
  virtual void doRendering(SoGLRenderAction *action);

  /** Must be redefined because by default returns false when there are tesselationShader on states and here, 
   * we can manage tesselation shaders. */
  virtual SbBool shouldGLRender(SoGLRenderAction* action, SbBool isPointOrLine = FALSE);

  /** Destructor */
  virtual ~SoHeightFieldRender();

SoINTERNAL protected :

  /** Register ldm geometry to octree */
  virtual void ldmAction( SoLdmValuationAction* action );

protected:
  /**
   * Fill detail with properties value
   */
  virtual SoDetail* createTriangleDetail( SoRayPickAction* action,
                                          const SoPrimitiveVertex* v0,
                                          const SoPrimitiveVertex* v1,
                                          const SoPrimitiveVertex* v2,
                                          SoPickedPoint* pp );

private:

#if SoDEPRECATED_BEGIN(9300)

  enum SoDEPRECATED_ENUM(9300,"No longer used. Normals are now computed on the fly on gpu.")
  NormalPrecision
  {
    /** Normals are saved in a memory efficient format */
    NORMAL,

    /** Normals are saved in a high precision format but use more GPU memory */
    HIGH
  };

  SoDEPRECATED_FIELD(9300,"No longer used. Normals are now computed on the fly on gpu.")
  SoSFEnum normalPrecision;
#endif /** @DEPRECATED_END */

  /**
   * Initialize implementation class using current context to
   * know if tessellation shaders are available if needed and return
   * m_heightFieldRenderImpl.
   */
  SoHeightFieldRenderImpl* getImpl( SoState* state );
  SoHeightFieldRenderImpl* m_heightFieldRenderImpl;

  /** VolumeState associated to this node is equal to SoVolumeShape::m_volumeState but is directly cast
   * to propert type for convenience. */
  SoVolumeStateHeightField* m_volumeStateHeightField;

  friend class SoVolumeStateHeightField;
};

#ifdef _WIN32
#pragma warning( pop )
#endif
#endif // _SO_HEIGHTFIELD_RENDER_H

