/*=================================================================================
***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
***                                                                            ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
***                                                                            ***
***                        RESTRICTED RIGHTS LEGEND                            ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
***                                                                            ***
***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                    ***
***                              BORDEAUX, FRANCE                              ***
***                            ALL RIGHTS RESERVED                             ***
=================================================================================*/
#ifndef _SO_OFFSCREEN_VOLUME_RENDER_H_
#define _SO_OFFSCREEN_VOLUME_RENDER_H_

#include <Inventor/SbPImpl.h>
#include <Inventor/nodes/SoNode.h>

#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/fields/SoSFPath.h>
#include <Inventor/fields/SoSFMatrix.h>
#include <Inventor/fields/SoSFEnum.h>

#include <VolumeViz/nodes/SoVolumeData.h>

class SoBufferObject;
class SoCamera;
class SoFrameBufferObject;
class SoSceneManager;

SO_PIMPL_PUBLIC_DECLARATION( SoOffscreenVolumeRender )

/**
* @VREXT Extract data from an SoVolumeRender
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
* The SoOffscreenVolumeRender node extracts data by rendering one or more volumes (via SoVolumeRender)
* into a buffer that can then be used for different computation tasks.
*
* Volumes of interest are specified by a subscenegraph in #volumerenderSceneGraph.
*
* The region to be extracted is the bounding box of the scene graph specified in
* the #bboxSceneGraph field. 
*
* To implement an extraction, the application must derive a new class from SoOffscreenVolumeRender
* and implement :
* - the boxComputed method.  This method is called with parameters that give
* access to information about the processed box.
* - the getNextSubBox method. This method must return subBox size smaller than getMaxBoxSize.
* We recommend to request subbox even smaller in order to get more parallelism between extraction 
* processing (done on GPU) and application processing and reduce peak GPU memory resources used.
*
* The resolution of the extraction is controlled by #boxSize field.
* As extraction is done using GPU techniques, there is a size limit on the boxSize
* that can be extract in one call. This limit can be retreived through getMaxBoxSize call.
* 
*
* @B Limitations: @b 
* - It is not possible to extract a volume displayed as an isosurface
* (via SoVolumeDataDrawStyle or SoVolumeIsosurface).
* - SoResetTransform node usage should be avoided in volumeRenderSceneGraph or bboxSceneGraph
* scene graph, in order to get the best extraction precision even with very flat extraction box.
* In general it is possible to use SoTransformSeparator node instead.
*
* @FILE_FORMAT_DEFAULT
*    SoOffscreenVolumeRender {
*    @TABLE_FILE_FORMAT
*      @TR volumerenderSceneGraph @TD NULL
*      @TR bboxSceneGraph   @TD NULL
*      @TR boxSize          @TD (64, 64, 64)
*      @TR trigger          @TD FALSE
*      @TR opacityThreshold @TD 0
*      @TR components       @TD ALPHA
*    @TABLE_END
*    }
*
* @SEE_ALSO
* SoDataSet, SoVolumeData, SoVolumeRender
*/
class VOLUMEVIZ_API SoOffscreenVolumeRender : public SoNode
{
  SO_NODE_ABSTRACT_HEADER(SoOffscreenVolumeRender);

  SO_PIMPL_PUBLIC_HEADER( SoOffscreenVolumeRender )

public:

  /** Default constructor */
  SoOffscreenVolumeRender();

  /** Head of the scene graph that contains one or multiple VolumeRender node to render in 3D off-screen cube */
  SoSFNode volumerenderSceneGraph;

  /** Head of the scene graph that defines the 3D off-screen scene cube to consider in world coordinates */
  SoSFNode bboxSceneGraph;

  /** Resolution of the output data in the off-screen box */
  SoSFVec3i32 boxSize;

  /** Setting this field to TRUE will start the off-screen rendering process.
  * This field is set to FALSE when computation is finished. Default is FALSE.
  */
  SoSFBool trigger;

  /** During extraction, voxels with an opacity less than the specified value will be ignored
  * Default is 0. Valid range is [0-1].
  */
  SoSFFloat opacityThreshold;

  /** */
  enum Components
  {
    /** */
    ALPHA,
    /** */
    RGBA
  };

  /**
  * Components to get back from the offscreen render.
  * @useenum{Components}. Default is ALPHA.
  */
  SoSFEnum components;

  /** 
   * Returns the maximum boxSize that can be extract in one boxComputed
   * for the current state.
   */
  SbVec3i32 getMaxBoxSize( SoState* state = NULL );


  /** 
   * Convert ijk box coordinates returned by boxComputed method to the bboxSceneGraph space.
   */
  SbBox3d ijkToXyzBox( const SbBox3i32& ijkBox );

  /**
  * This transformation is applied on the volume render texture coordinate during the extraction.
  * It allows to change the extraction frame.
  * This matrix is applied on normalized texture coordinates all components are between 0-1 and must be in [0-1] after the transformation.
  * So to inverse an axis, we must apply the following transformation: -1*u+1 (where u is a texture coordinate) 
  * and not just a simple negative scale.
  *
  * To swap I and J axis, the matrix would be:
  * \code
  * SbMatrix( 0, 1, 0, 0,
  *           1, 0, 0, 0,
  *           0, 0, 1, 0,
  *           0, 0, 0, 1);
  * \endcode
  *
  * To reverse I axis, the matrix would be:
  * \code
  * SbMatrix(-1, 0, 0, 0,
  *           0, 1, 0, 0,
  *           0, 0, 1, 0,
  *           1, 0, 0, 1);
  * \endcode
  *
  * To reverse J axis, the matrix would be:
  * \code
  * SbMatrix( 1, 0, 0, 0,
  *           0,-1, 0, 0,
  *           0, 0, 1, 0,
  *           0, 1, 0, 1);
  * \endcode
  *
  * To reverse K axis, the matrix would be:
  * \code
  * SbMatrix( 1, 0, 0, 0,
  *           0, 1, 0, 0,
  *           0, 0, -1, 0,
  *           0, 0, 1, 1);
  * \endcode
  *
  */
  SoSFMatrix dataSpaceMatrix;

  /** 
   * Does extraction synchronously, whereas using trigger field will
   * execute extraction on next redraw.
   *
   * @param sceneManager the scene manager of the viewer scene to be extracted.
   * @return TRUE on success.
   */
  bool synchronousExtraction( SoSceneManager* sceneManager );

SoEXTENDER_Documented protected:
  /**
   * Called by extraction process to get iteratively the subbox inside boxSize to extract.
   * @param box to extract in ijk space. It must be contained in boxSize
   * @return FALSE when no more subbox should be extract
   */
  virtual SbBool getNextSubBox( SbBox3i32& box ) = 0;

  /** Method called each time a subBox has been computed.
   * If FALSE is returned then the whole process is aborted.
   * @param action is the current applied action
   * @param data contains extracted values for the current box
   * @param box is the current box in ijk space
   */
  virtual void boxComputed( SoGLRenderAction* action, SoBufferObject* data, const SbBox3i32& box ) = 0;

SoEXTENDER public:

  /** reimplement GLRender behavior to launch computation when trigger touch */
  virtual void GLRender( SoGLRenderAction* action );

SoINTERNAL public:

  /**
   * Render only a path (contrary to volumerenderSceneGraph which render
   * a whole scenegraph). If volumerenderSceneGraph is NULL, volumerenderPath
   * is rendered. 
   * In SoINTERNAL public because this is a patch for eBug #4339 but we can put
   * it in public if needed.
   */
  SoSFPath volumerenderPath;

  // Internal:
  static void initClass();
  static void exitClass();

  SbBox3f getBoundingBox( SoAction* action );

  void setSliceNumber( SoState* state, int slice );

  /**
   * This function return if slice's RTT need to be cleared.
   * @warning this function modify clear flag of the slice.
   */
  bool needClearTarget( SoState* state, int slice );

  void beginOffscreenRendering( SoState* state );
  void endOffscreenRendering( SoState* state );
  virtual void notify( SoNotList* list );

  // check that the not is OK to do SoVolumeRender offscreen operation
  bool isReady() const;

protected:
  /** Destructor */
  virtual ~SoOffscreenVolumeRender();

  /** Subbox that is currently extracted.
   * Empty if we are not currently extracting. */
  SbBox3i32 m_currentSubBox;

  /** Slice that is currently extracted.
   * -1 if we are not currently extracting. */
  int m_currentSliceNumberInSubBox;

  /** Number of subBox that is currently extracted
   * => number of boxes that have been extracted before the current one.
   * -1 if not currently extracting. */
  int m_currentSubBoxNumber;

private :

  void commonConstructor();

};

#endif //_SO_OFFSCREEN_VOLUME_RENDER_H_


