/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Aug 2006)
**=======================================================================*/


#ifndef SO_UNIFORM_GRID_CLIPPING
#define SO_UNIFORM_GRID_CLIPPING

#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/SbBox.h>
#include <Inventor/STL/vector>
#include <LDM/SoLDMTileID.h>

class SoShaderObject;
class SoShaderProgram;
class SoFragmentShader;
class SoVertexShader;
class SoState;
class SoVolumeData;
class SoROI;
class SoLDMNodeFrontMgrOctree;
class SoCache;

SO_PIMPL_PUBLIC_DECLARATION(SoUniformGridClipping);

/**
* @VREXT Clips a volume using a 2D surface.
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
* The SoUniformGridClipping node specifies a clipping surface defined by a clipping
* axis (axis field),a bounding box (extent field) in world coordinates and a uniform
* scalar clipping grid (a 2D array of height values). This tool provides an efficient
* way for seismic applications to clip volume rendering against horizon surfaces (see
* also SoUniformGridProjectionClipping). A flat clipping surface with non-zero thickness
* can also be used in medical applications to implement "thick slice" rendering. The
* clipping grid does not need to have the same dimensions as the volume to be clipped
* (it will be sampled and interpolated). The clipping grid may be rotated and translated
* by standard Open Inventor transform nodes.
*
* The height values are specified using a 2D texture. Each texel is a normalized value
* that, together with the extent, specifies a height. For integer valued textures, the
* range of the integer type (e.g. 0..255 for byte values) is normalized to the floating
* point range 0..1. For float valued textures the values should already be in the range
* 0..1. Each normalized value specifies a height based on the range of values in the
* extent box along the specified clipping axis. The default clipping axis is the Y axis.
*
* NOTE: For integer valued textures this is similar to the way values are interpreted
* for the SoHeightFieldGeometry node. However for float valued textures it is quite
* different because the SoHeightFieldGeometry node interprets float values as actual
* height values and ignores the extent values.
*
* It is also possible to specify an "undefined" value using the undefinedValue field.
* Texels with this value will clip all voxels above and below the surface. The default
* undefined value is Nan (Not a Number).
*
* The X, Y and Z coordinates of the extent are in the same coordinate system as the
* volume extent. They specify the bounding box of the clipping surface and also the
* range of height values that can be specified with the normalized values in the height
* field texture. They are typically, but not necessariliy, set to the same values as the
* volume extent. The extent of the clipping surface may be larger than the volume extent.
* In this case the volume is only clipped where its extent overlaps with the surface.
* The extent of the clipping surface may also be smaller than the volume extent. In this
* case the volume is clipped to the "horizontal" extent of the clipping surface (the
* dimensions perpendicular to the clipping axis). However along the clipping axis the
* volume is only limited by the surface itself, not by its extent.
*
* SoUniformGridClipping is derived from SoTexture2 , so the clipping grid texture can be
* specified by setting the filename field or by directly setting the image field. Directly
* setting the image field is convenient if the height field data is floating point. All
* the usual texturing parameters apply. Wrapping and filtering modes are particularly
* important. Setting wrap mode to something other than CLAMP_TO_EDGE may cause unwanted
* interpolation on edges. Setting filter mode to NEAREST will give a blocky result which
* may or may not be desired. 
*
* The format of the texture used for the grid should be the LUMINANCE* type for best
* performance. For example, a grayscale .png file will automatically be in this format.
* We suggest to use #LUMINANCE_FLOAT32 explicitely for floating point data in order to
* preserve floating point precision. If no red component is available in the format used
* (for example ALPHA texture...), the grid will be totally flat.
*
* Each clipping surface texture must be stored in a different OpenGL texture unit. One
* that is not currently used by VolumeViz for multi-data composition (i.e., a texture
* unit number greater than the highest dataSetId plus 1). Setting the texture unit for a
* clipping surface is done using the SoTextureUnit node, just like with any texture node.
*
* The SoUniformGridProjectionClipping node provides an alternate way to specify the
* clipping grid.  This node may be more convenient when clipping using surface geometry
* that is already in the scene graph, for example an SoHeightFieldRender node.
*
* The figure below shows the result (right) of applying the left texture to a volume.
* \par
* @TABLE_1B
*      @TR Clipping surface @TD Clipping applied to a volume
*         @TR  @IMAGE clipgrid.jpg
*                   @TD @IMAGE clipgrid_result.jpg
*      @TABLE_END
*
* The following code shows one way to achieve this result.
* It loads the clipping surface "horizon2D.png" and puts it in texture unit 2:
* \par
* \if_cpp
*   \code
*   SoTextureUnit *texUnit2 = new SoTextureUnit();
*     texUnit2->unit = 2;
*   SoUniformGridClipping *grid = new SoUniformGridClipping();
*     grid->filename = "horizon2D.png";
*
*   SoSeparator* volSep = new SoSeparator();
*     volSep->addChild( volumeData );
*     volSep->addChild( transferFunction );
*     volSep->addChild( texUnit2 );
*     volSep->addChild( grid );
*     volSep->addChild( volumeRender );
*   root->addChild( volSep );
*   \endcode
* \endif
* \if_dotnet
*   \code
*   SoTextureUnit texUnit2 = new SoTextureUnit();
*     texUnit2.unit.Value = 2;
*
*   SoUniformGridClipping grid = new SoUniformGridClipping();
*     grid.filename.Value = "horizon2D.png";
*
*   SoSeparator volSep = new SoSeparator();
*     volSep.AddChild( volumeData );
*     volSep.AddChild( transferFunction );
*     volSep.AddChild( texUnit2 );
*     volSep.AddChild( grid );
*     volSep.AddChild( volumeRender );
*   root.AddChild( volSep );
*   \endcode
* \endif
* \if_java
*   \code
*   SoTextureUnit texUnit = new SoTextureUnit();
*     texUnit.unit.setValue( 2 );
*
*   SoUniformGridClipping gridClip = new SoUniformGridClipping();
*     gridClip.filename.setValue( "horizon2D.png" );
*
*   SoSeparator volSep = new SoSeparator();
*     volSep.addChild( texUnit );
*     volSep.addChild( gridClip );
*     volSep.addChild( transferFunction );
*     volSep.addChild( volumeData );
*     volSep.addChild( volumeRender );   
*   root.addChild( volSep );
*   \endcode
* \endif
*
* The uniform grid clipping node provides both #clipAbove and #clipBelow boolean fields.
* By default only clipAbove is true. Using a pair of grid clipping nodes, one with
* clipAbove and one with clipBelow, allows clipping between two surfaces. This is useful
* in seismic applications to clip between two "horizon" surfaces. Note that if both of
* these fields are true and the thickness field is zero then the entire volume will be clipped.
*
* The thickness field expands the clipping surface in both directions perpendicular to the
* surface.The value of this field is specified in 0..1 normalized units spanning the specified
* extent of the surface. A completely flat clipping surface with thickness greater than zero
* can be used to implement "thick slice" (aka "thin slab") rendering. A transform node, for
* example SoTransform , can be used to position and orient the clipping surface. In this case
* the transform node and the clipping surface node should usually be placed under an
* SoTransformSeparator node. This allows the surface to clip the volume primitives, but
* prevents the transform node from affecting the volume primitives. Remember that, as usual,
* the transform node will rotate the clipping surface around the origin (0,0,0) of the world coordinate system.
* \par
* @TABLE_1B
*      @TR Clipping surface thicknes
*      @TR @IMAGE VVClipping15.png
*      @TR @IMAGE VVClipping16.png
* @TABLE_END
*
* @B Notes: @b 
* @UL 
*  @LI When using a custom SoVolumeShader with this node and redefining the @I main() @i
*   function, you must call @I VVizClipVoxel() @i in the @I main() @i function if you are writing a
*   fragment shader or @I VVizCalcDistToUniformGrid() @i if it's a vertex shader. If you don't
*   do this, you will get a GLSL compilation error or clipping won't work.
* @ul
*
* @B Limitations: @b
* @UL
*  @LI Only graphics cards supporting the GLSL language can use this node.
*
*  @LI The maximum number of SoUniformGridClipping nodes that can be
*   applied to the same volume is limited to getMaxNumberOfUniformGrids().
*   This limitation depends on the maximum number of sets of OpenGL texture coordinates
*   supported on the graphics board (GL_MAX_TEXTURE_COORDS), which is typically 8,
*   minus 3 sets used elsewhere in VolumeViz, leaving 5 for uniform grids.
* @ul
*
 * @FILE_FORMAT_DEFAULT
 *    UniformGridClipping {
 *    @TABLE_FILE_FORMAT
 *       @TR axis                     @TD Y
 *       @TR extent                   @TD -1,0,-1 1,1,1
 *       @TR filename                 @TD ""
 *       @TR image                    @TD 0 0 0
 *       @TR wrapS                    @TD CLAMP_TO_EDGE
 *       @TR wrapT                    @TD CLAMP_TO_EDGE
 *       @TR model                    @TD MODULATE
 *       @TR enableCompressedTexture  @TD FALSE
 *       @TR blendColor               @TD 0 0 0
 *       @TR enableBorder             @TD FALSE
 *       @TR borderColor              @TD 0 0 0 0
 *       @TR maxAnisotropy            @TD 1.0
 *       @TR minFilter                @TD AUTO
 *       @TR magFilter                @TD AUTO
 *       @TR loadingMode              @TD AUTO
 *       @TR useAutoMipmap            @TD FALSE
 *       @TR internalFormat           @TD AUTO_INTERNAL_FORMAT
 *       @TR thickness                @TD 0
 *       @TR clipBelow                @TD FALSE
 *       @TR clipAbove                @TD TRUE
 *    @TABLE_END
 *    }
* @SEE_ALSO
*  SoVolumeRender,
*  SoPreferences,
*  SoShaderProgram,
*  SoTexture2,
*  SoTextureUnit,
*  SoVolumeClippingGroup,
*  SoVolumeIsosurface,
*  SoVolumeRenderingQuality,
*  SoUniformGridProjectionClipping
*
*/
class VOLUMEVIZ_API SoUniformGridClipping : public SoTexture2
{

  SO_NODE_HEADER(SoUniformGridClipping);

  SO_PIMPL_PUBLIC_HEADER(SoUniformGridClipping);

public:
  /**
   * Constructor.
   */
  SoUniformGridClipping();

  /**
   * Bounding box of the surface in 3D geometric coordinates. @BR
   * Default is a box of size (2, 1, 2) with its back left corner at (-1, 0, -1). @BR
   * @FIELD_SINCE_OIV 8.0
   */
  SoSFBox3f extent;

  /** Slice Axis */
  enum Axis {
    /** X Axis */
    X,
    /** Y Axis (Default) */
    Y,
    /** Z Axis */
    Z
  };

  /**
   * Height axis: X, Y, or Z.
   * @useenum{Axis} . Default is Y.
   *
   * @FIELD_SINCE_OIV 8.0
   */
  SoSFEnum axis;

  /** 
   * Defines a clipping thickness in the axis direction.
   * Ignored if set to 0. Default is 0;
   *
   * Value is in the normalized #extent space (can be seen as a percentage of the extent).
   * @IMAGE UniformGripClipping_thickness.png
   *
   * Notes:
   * - If thickness is 0 and clipBelow and clipAbove are both TRUE then the whole volume is clipped.
   * - A value greater than 1.0 makes sense only if the extent of the 
   * SoUniformGridClipping is smaller than the extent of the volume data it applies to.
   *
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFFloat thickness;

  /**
   * Clip voxels below the surface.
   * Default is FALSE;
   *
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFBool clipBelow;

  /**
   * Clip voxels above the surface.
   * Default is TRUE;
   *
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFBool clipAbove;

  /** 
   * Texels in the clipping texture with this value will clip all voxels above and below the surface. @BR
   * Default is NaN (not a number).
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFFloat undefinedValue;

  /**
   * Returns TRUE if uniform grid clipping is supported by the graphics board.
   * The GPU must support GLSL. Check the maximum number of clipping surfaces
   * using getMaxNumberofUniformGrids.
   *
   * When using a debug build of Open Inventor, some "no context available"
   * warning messages may be generated. You can ignore them or see
   * SoGLExtension for an example of using SoGLContext to avoid them.
   */
  static SbBool isSupported(SoState* state=NULL) ;

  /**
   * Returns the maximum number of clipping surfaces supported by the hardware.
   */
  static int getMaxNumberOfUniformGrids(SoState* state=NULL);

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_FIELD(8000,"Use #SoUniformGridClipping::extent instead.")
  SoSFBox3f size;

#endif /** @DEPRECATED_END */

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void doAction(SoAction *action);
  virtual void doActionOnRTT(SoAction *action);
  virtual void rayPick(SoRayPickAction *action);

SoINTERNAL public:

  /**
   * Allows to revert elevation values elevation = elevationMax - (elevation - elevationMin).
   * This is usefull when generating elevation map from a normalized depth map 
   * where highest values is 0 and smallest is 1.
   * Default is FALSE;
   * INTERNAL FIELDS only used by SoUniformGridProjectionClipping
   *
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFBool revertMode;

  static void initClass() ;
  static void exitClass() ;

  /** Handle field change*/
  virtual void notify(SoNotList *list);

  /** Manages field dependencies, when one of them change */
  virtual void fieldHasChanged( SoField *field ); // override of SoTexture2

protected:

  /** Destructor */
  virtual ~SoUniformGridClipping();

private:

  void commonConstructor();

#ifndef HIDDEN_FROM_DOC
  // friendship needed to access pimpl
  friend class SoVolumeState;
  friend class SoUniformGridClippingElement;
#endif
};

#endif


