/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Aymeric Chataigner (Sept 2012)
**=======================================================================*/
#ifndef __VOLUME_BUFFERED_SHAPE
#define __VOLUME_BUFFERED_SHAPE

#include <Inventor/nodes/SoBufferedShape.h>

#include <VolumeViz/nodes/SoVolumeGeometry.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/SbBox.h>

class SoVolumeGeometryVbo;

/** 
 * @VREXT Buffered shape node for volume data.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 * This node defines a shape which is the intersection of the volume data
 * defined by an SoVolumeData node and the 3D shape defined by an SoBufferedShape node.
 * The shape defined by this intersection is colored using the data values
 * from the current SoVolumeData node and color map from the current SoTransferFunction node.
 *
 *   The shape defined by this intersection is textured using the data
 *   from the current SoVolumeData node and SoTransferFunction node.
 *   The #interpolation field controls how the texture is interpolated.
 *   Texture coordinates are automatically computed for each vertex based on its
 *   position relative to the 3D extent of the volume.  If texture coordinates are
 *   specified, they are ignored.
 *
 *   This node can be used to create custom "slices" through the volume, for example
 *   a cylinder slice.  It can be considered a generalization of the SoOrthoSlice,
 *   SoObliqueSlice, SoFenceSlice, etc features. But note:
 *   - Compared to an equivalent standard slice (e.g. SoOrthoSlice), the rendering may
 *     be slightly different due to positioning and interpolation.
 *
 *   - The standard slices (e.g. SoOrthoSlice) have "two-sided lighting" enabled by
 *     default as a convenience.  This is not true for volume geometry nodes (which
 *     have the same default as their corresponding standard geometry node), so the
 *     "back" side of the geometry will not be lighted unless you enable two-sided
 *     lighting using an SoShapeHints node.
 *
 * When set to FALSE, the #clipGeometry field allows rendering of the
 * portion of the geometry that does not intersect the volume data. This
 * portion is not textured and is rendered as a regular SoBufferedShape.
 *
 * This node uses 3D texturing and is not available if the hardware does not
 * support this feature.
 *
 * SoBufferedShape provides fields for:
 *
 * - Vertices
 * - Indices (optional)
 * - Colors (optional)
 * - Normals (optional)
 * - Texture coordinates (optional)
 *
 * SoVolumeBufferedShape is useful to manage the rendering of large geometry, provide 
 * application control over where the data is stored (CPU or GPU) and to 
 * integrate rendering with the Open Inventor computing framework (through the 
 * SoBufferObject classes).
 *
 * SoVolumeBufferedShape can render many types of geometric primitives including points, lines,
 * quads and triangles. (A single type must be specified per instance of SoVolumeBufferedShape.)
 * You specify the type of primitive in the SoSFEnum field #shapeType.
 *
 * SoVolumeBufferedShape can render multiple primitives of the same type.  You can
 * specify the number of vertices (or indices if provided) to use for each primitive 
 * in the SoMFInt32 field #numVertices (similar to SoFaceSet).
 *
 * You can also use the primitive restart feature to define multiple indexed strip shapes,
 * for example TRIANGLE_STRIP or LINE_STRIP. The end of each primitive is marked by a 
 * special index value in the index buffer and this value can be specified
 * in the #primitiveRestartValue field.  The behavior is similar to the "-1" value that can
 * be used in Open Inventor indexed shape nodes like SoIndexedFaceSet, but is implemented 
 * on the GPU.  @BR
 *
 * NOTE:
 *   - Primitive restart must be explicitly enabled using the #primitiveRestartEnabled field.
 *
 *   - Check availability of primitive restart using SoShape::isPrimitiveRestartAvailable().
 *
 * The geometry and its attributes must be stored in buffer objects (see SoBufferObject).
 * The buffer objects can be SoGLBufferObjects stored directly on the graphics board or 
 * SoCpuBufferObjects stored in system memory.  This allows the application to control 
 * what data is stored where. 
 *
* If lighting is enabled (there is no SoLightModel node or the model field of the SoLightModel
* is set to PHONG) and the #normalBuffer field is not set, then normal vectors are
* automatically generated, similar to other geometry nodes. Normal generation is 
* affected by the creaseAngle field of the SoShapeHints node if the vertices are NOT indexed.
* If the vertices are indexed the creaseAngle is forced to PI in order to have 
* smooth surface rendering. It is not possible to display sharp edges by
* using indexed vertices if the normals are not set by the application.
* It is possible to disable normal generation (if for example the normals are generated by a
* geometry shader) by setting the #useNormalsGenerator field to FALSE. Note that normal 
* generation is disabled in some cases.  See the limitations section.  If no normal vectors 
* are specified or generated, and lighting is enabled, the primitive may not be
* rendered correctly.
 *
 * SoVolumeBufferedShape provides fields to describe the content of each buffer, e.g. the data 
 * type and number of components in each buffer, as well as how to access the buffers, 
 * e.g. the offset into the buffer and "stride" separating data values in the buffer. 
 * The default values for offset and stride assume that the vertices, normals, etc are
 * each in a separate buffer.  However setting appropriate offset and stride allows, for
 * example, vertices and normals to be interleaved in a single buffer.  In this case the 
 * same buffer would be set into both the vertexBuffer and normalBuffer fields.
 * 
 * To disable computing the bounding box, which can take a long time with very large geometry,
 * use the SoBBox node to specify a pre-computed bounding box.
 *
 * @B Limitations @b
 *
 * - Projections: @BR
 *   The projections used in the VolumeViz mechanism are not handled by SoVolumeBufferedShape.
 *
 * - Lighting: @BR
 *   If there is no normalBuffer the normals are generated by Inventor. 
 *   The crease angle (field of SoShapeHints) is not considered by the normal
 *   generator if the vertices are indexed (crease angle is PI). So if the application wants 
 *   to render sharp edges on its shape, it must use the vertexBuffer without indices in the indexBuffer.
 *
 * - Transparency: @BR
 *   If there is no color buffer, making the entire shape transparent using an SoMaterial
 *   node works as usual.  However if there is a color buffer with RGBA values, note that
 *   Open Inventor does not currently check the color buffer for transparency (alpha values
 *   < 1). So in this case the SoVolumeBufferedShape will not be considered transparent geometry
 *   (even if there are alpha values < 1) and may not be rendered correctly. You can force Open 
 *   Inventor to handle the shape as transparent geometry by putting an SoMaterial node with
 *   non-zero transparency before it in the scene graph.
 *
 * - Normal generation: @BR
 *   Automatic generation of normal vectors is ONLY enabled when:
 *   - The #vertexComponentsCount field is set to 3 (default),
 *   - The #vertexComponentsType field is set to SbDataType::FLOAT (default),
 *   - The #primitiveRestartEnabled field is set to FALSE (default), and
 *   - The primitives are faces (i.e. neither points nor lines).
 *
 * - SoGetPrimitiveCountAction: @BR
 *   When using the primitive restart feature, the triangle/line count returned by the
 *   SoGetPrimitiveCountAction will not be accurate.
 *
 * - Concave polygons: @BR
 *   Unlike (for example) SoFaceSet, SoVolumeBufferedShape does not automatically tesselate
 *   concave or complex polygons.  Such primitives may not be rendered correctly.
 *
 * - SoWriteAction: @BR
 *   SoVolumeBufferedShape can be saved to and restored from a .iv file just like any other
 *   Open Inventor node. However, during the read operation any OpenGL buffer objects
 *   (SoGLBufferObject) in the file will be created as CPU buffers (SoCpuBufferObject) if 
 *   there is no OpenGL context bound during the read operation.
 *
 * - Material binding (etc): @BR
 *   SoVolumeBufferedShape effectively only supports per-vertex and per-vertex-indexed
 *   binding of materials, normals and texture coordinates.
 *
 * - Simplification: @BR
 *   SoVolumeBufferedShape is ignored by the "simplify" actions (SoShapeSimplifyAction,
 *   SoGlobalSimplifyAction, SoReorganizeAction).
 *
 * @B Notes: @b
 * -     The volume size and orientation (like geometry) can be modified by
 *       transformation nodes in the scene graph and this in turn modifies the
 *       appearance of volume rendering nodes like SoVolumeBufferedShape.
 *       However the same transformation must be applied to the volume data node
 *       and all volume rendering nodes associated with that volume.  So effectively
 *       any transformation nodes that affect the volume must be placed @B before @b
 *       the volume data node.
 *
 * - Composition with Multiple Data: @BR
 *   It is possible to compose datasets that have different dimensions,
 *   tile sizes and transformations. @BR
 *   In order to help fetch the correct data values in custom shaders,
 *   texture coordinates conversion functions are provided in the
 *   @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *   For instance,
 *   \code
 *   vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *   \endcode
 *   can be used to convert texture coordinates related to one dataset to
 *   texture coordinates related to another dataset. @BR
 *   The conversion is based solely on the transformations applied to each
 *   dataset, which are defined by their model matrix and their extent. @BR
 *   Please note that the model matrix of a dataset is defined by to the
 *   SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *   the order of the traversal.
 *
 * @FILE_FORMAT_DEFAULT
 *    VolumeBufferedShape {
 *    @TABLE_FILE_FORMAT
 *       @TR clipGeometry     @TD TRUE
 *       @TR offset           @TD 0
 *       @TR interpolation    @TD LINEAR
 *       @TR useNormalsGenerator      @TD TRUE
 *       @TR shapeType                @TD TRIANGLES
 *       @TR numVertices              @TD 0
 *       @TR vertexBuffer             @TD NULL
 *       @TR vertexComponentsCount    @TD 3
 *       @TR vertexComponentsType     @TD SbDataType::FLOAT
 *       @TR vertexStride             @TD 0
 *       @TR vertexOffset             @TD 0
 *       @TR normalBuffer             @TD NULL
 *       @TR normalComponentsType     @TD SbDataType::FLOAT
 *       @TR normalStride             @TD 3 * sizeof(float)
 *       @TR normalOffset             @TD 0
 *       @TR indexBuffer              @TD NULL
 *       @TR indexType                @TD SbDataType::UNSIGNED_INT32
 *       @TR indexOffset              @TD 0
 *       @TR colorBuffer              @TD NULL
 *       @TR colorComponentsType      @TD SbDataType::FLOAT
 *       @TR colorStride              @TD 0
 *       @TR colorOffset              @TD 0
 *       @TR colorComponentsCount     @TD 3
 *       @TR texCoordsBuffer          @TD NULL
 *       @TR texCoordsComponentsType  @TD SbDataType::FLOAT
 *       @TR texCoordsStride          @TD 0
 *       @TR texCoordsOffset          @TD 0
 *       @TR texCoordsComponentsCount @TD 2
 *       @TR primitiveRestartEnabled  @TD FALSE
 *       @TR primitiveRestartValue    @TD -1
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    @B SoGLRenderAction @b @BR
 *        Draws primitives based on the current coordinates, normals, materials, drawing
 *        style, and so on.
 *
 *    @B SoRayPickAction @b @BR
 *        Picks primitives based on the current coordinates and transformation. Details about
 *        the intersection are returned in an SoFaceDetail.
 *
 *    @B SoGetBoundingBoxAction @b @BR
 *        Computes the bounding box that encloses all vertices of the face set with the
 *        current transformation applied to them. Sets the center to the real center of
 *        the bounding box: center{ (xmax + xmin) / 2, (ymax + ymin) / 2, (zmax + zmin) / 2 }
 *
 *    @B SoCallbackAction @b @BR
 *        If any triangle callbacks are registered with the action, they will be invoked
 *        for each successive triangle generated from each face in the set.
 *
 *
 * @SEE_ALSO
 *    SoBufferedShape,
 *    SoVolumeData,
 *    SoTransferFunction,
 *    SoROI,
 *    SoVolumeIndexedFaceSet,
 *    SoVolumeIndexedTriangleStripSet,
 *    SoVolumeTriangleStripSet,
 *    SoFaceDetail
 *
 */
class VOLUMEVIZ_API SoVolumeBufferedShape : public SoBufferedShape, public SoVolumeGeometry
{
  SO_NODE_HEADER( SoVolumeBufferedShape );

public :

  /**
   * Specifies to clip the geometry at the volume boundaries. @BR
   * The SoROI node also affects the clipping process. Default is TRUE.
   */
  SoSFBool clipGeometry;

  /**
   * Sets an offset value used for computing the texture coordinates.  @BR
   * The texture coordinate for each vertex will be taken from a point @B offset @b
   * units away from the vertex.
   * The offset is measured in the direction of the vertex normal.
   * By default the offset is equal to 0 (no offset). When using this feature,
   * a vertex normal must exist for every vertex. If vertex normals are not
   * supplied, Open Inventor will compute vertex normals in the usual way except
   * that the crease angle will be fixed at PI in order to assure that a vertex
   * normal is computed at every vertex.
   */
  SoSFFloat offset;

  /**
   * @copydoc SoVolumeShape::interpolation
   */
  SoSFEnum interpolation;

  /** Constructor. */
  SoVolumeBufferedShape();

SoEXTENDER public:
  /** @copydoc SoNode::initClass() */
  static void initClass();

  /** @copydoc SoNode::exitClass() */
  static void exitClass();

  /** @copydoc SoBufferedShape::GLRender() */
  virtual void GLRender( SoGLRenderAction* action );

  /** @copydoc SoBufferedShape::computeBBox() */
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );

  /** @copydoc SoBufferedShape::computeBBox() */
  virtual void computeBBox( SoAction* action, SbXfBox3d& box, SbVec3d& center );

SoINTERNAL public:

  /** @copydoc SoBufferedShape::notify() */
  virtual void notify( SoNotList* list );

  /** Return interpolation value according to the SoInteractiveComplexityElement */
  virtual SoVolumeShape::Interpolation getInteractiveInterpolationValue(SoState* state) const;

protected:
  /** @copydoc SoVolumeRender::shapeRender */
  virtual void shapeRender( SoGLRenderAction* action );

  /** @copydoc SoVolumeRender::shapeGeneratePrimitives */
  virtual void shapeGeneratePrimitives( SoAction* action );

  /** @copydoc SoBufferedShape::GLRenderBoundingBox */
  virtual void GLRenderBoundingBox( SoGLRenderAction* action );

  /** @copydoc SoVolumeRender::getBBox */
  virtual void getBBox( SoAction* action, SbBox3f& box, SbVec3f& center );

  /** @copydoc SoVolumeRender::getBBox */
  virtual void getBBox( SoAction* action, SbXfBox3d& box, SbVec3d& center );

  /** @copydoc SoVolumeGeometry::getNumVertices */
  virtual int getNumVertices(SoState* /*state*/) { return 0; }

  /** @copydoc SoVolumeGeometry::getIndexCoord */
  virtual int getIndexCoord(int /*index*/) const { return 0; }

  /** Destructor. */
  virtual ~SoVolumeBufferedShape();

};

#endif //__VOLUME_BUFFERED_SHAPE


