/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Feb 2000)
**=======================================================================*/
#ifndef  _SO_VOLUME_DATA_
#define  _SO_VOLUME_DATA_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/STL/map>
#include <Inventor/STL/vector>

#include <Inventor/SbBox.h>
#include <Inventor/SbPList.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFMemObj.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFArray3D.h>

#include <LDM/SoLDMDataAccess.h>
#include <LDM/SoLDMMediator.h>
#include <LDM/SoLDMTileID.h>
#include <LDM/nodes/SoDataSet.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/readers/SoVolumeReader.h>

#include <VolumeViz/nodes/SoVolumeRendering.h>

class SoGLRenderAction;
class SoCallbackAction;
class SoWriteAction;
class SoVolumeReader;
class SoCache;
class SoState;
class SoGLVolCacheList;
class SoVolumeHistogram;
class SoDataCompositor;
class SoVolumeShader;
class SoVolumeState;

// LDM
class SoLDMTileManager;
class SoLDMTileVisitor;
class SoLDMNodeFrontManager;
class SoLDMTextureManager;
class SoLDMGeometry;
class SoLDMResourceManager;

// Algorithms
class SoDataExtract;
class LDMDefaultSliceAccessor;
class SoVolumeData;

SO_PIMPL_PUBLIC_DECLARATION(SoVolumeData)

/**
* @VREXT Volume data property node.
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
*   This class defines the data volume and its properties, and also provides utilities for
*   extracting a subset of the volume and for resampling the volume. The data can be
*   stored in memory, read from a file or accessed via a user-defined reader.
*   This node provides the data for the volume rendering shape nodes (SoVolumeRender,
*   SoOrthoSlice, SoObliqueSlice, etc.) and is the parent class for some specialized
*   data nodes (SoHeightFieldGeometry, SoVolumeMask, etc.).
*
* Note: Since the camera viewpoint is used to determine which part of the volume to load,
*   the behavior of SoVolumeData is not correctly defined when no camera node has been traversed
*   before the SoVolumeData. This can result in console warnings.
*
* The data volume can be specified by:
*
*   - Setting the @B #fileName field@b @BR
*     This implies that the volume is stored on disk, in one of the file
*     formats for which VolumeViz has a built-in reader. This is the most common
*     way to load data. VolumeViz will automatically select a volume reader class 
*     based on the file extension, for example ".am" for the AmiraMesh file format.
*     If the filename does not
*     have an extension or does not have the appropriate extension, the
*     application must specify a volume reader explicitly using setReader().
*     See the supported file formats below. @BR Note that, unlike other nodes, SoVolumeData
*     and its derived classes do not search the SoInput directory list to find files.
*
*   - Calling the @B setReader() method@b @BR
*     This is the most general method because an application can specify an instance of
*     one of the standard VolumeViz readers or an instance of a custom subclass of SoVolumeReader. @BR @BR
*     You might use this method with a standard VolumeViz reader class if the data file's
*     name has a non-standard extension.  In other words, if VolumeViz will not be able
*     to select the correct volume reader automatically, explicitly create an instance
*     of the correct reader and pass it to the setReader() method. @BR @BR
*     VolumeViz will get the volume properties (dimensions, size, data type, etc) and
*     access the volume data through the specified reader object.  Creating an instance
*     of a custom volume reader (see SoVolumeReader for more information) allows the
*     application to completely control how, and from where, the data is loaded.
*     For example the data could be accessed through an application specific API. @BR
*     Note: When using a custom reader, any reader method that changes the volume
*     properties (dimension, size, data type, etc) should notify the SoVolumeData
*     node by calling the reader's touch() method.  If this notification is not done,
*     SoVolumeData fields, for example #extent, won't be updated correctly. @BR
*     Applications can use the getReader() method to query the current volume reader,
*     but should always verify the class type before using the returned object.
*
*   - Setting the @B #data field@b @BR
*     This implies that the entire volume has already been loaded into a contiguous
*     block of CPU memory.  We call this an "in-memory" volume.
*     @BR Volume reader: Note that when a file format reader or custom volume reader is
*     currently being used, setting the data field will automatically replace the current
*     volume reader. Subsequent calls to getReader() will return an SoVRMemoryReader.
*     @BR Tile size: Note that VolumeViz always manages volume data internally as "tiles" and the
*     default tile size is small.  Generally you wll get better performance by
*     explicitly setting a larger tile size (see the ldmResourceParameters field and
*     SoLDMResourceParameters::tileDimension).
*
*  @B Volume Properties: @b
*   - Dimensions @BR
*     The dimensions of the volume (number of voxels on each axis) are normally
*     determined by the volume reader from information in ithe data file(s), the
*     number of images in the stack, etc.  (When you set the #data field directly
*     you specify the volume dimensions.)  You can query the volume dimensions
*     using the #data field.  For example:
*     \par
*     \if_cpp
*       \code
*       SbVec3i32 voldim = volumeData->data.getSize();
*       \endcode
*     \endif
*     \if_dotnet
*       \code
*       SbVec3i32 voldim = volumeData.data.GetSize();
*       \endcode
*     \endif
*     \if_java
*       \code
*       SbVec3i32 voldim = volumeData.data.getSize();
*       \endcode
*     \endif
*
*   - Extent @BR
*     The geometric extent of the volume in 3D is initially determined by the
*     volume reader but can also be set using the #extent field. The volume extent
*     is the bounding box of the volume in world coordinates. 
*     Often the volume extent in 3D is set equal to the dimensions of the volume or 
*     to values that are proportional to the volume dimensions. For example, -1 to 1 
*     on the longest axis of the volume and proportional values for the other axes 
*     puts the origin (0,0,0) at the center of the volume, simplifying rotations. 
*     However the volume extent can be any range, for example the range of line numbers
*     in a seismic survey. The volume extent indirectly specifies the voxel size/spacing.
*     You can query the current volume extent using the #extent field.  For example:
*     \par
*     \if_cpp
*     \code
*        SbBox3f volext = volumeData->extent.getValue();
*     \endcode
*     \endif
*     \if_dotnet
*     \code
*        SbBox3f volext = volumeData.extent.Value;
*     \endcode
*     \endif
*     \if_java
*     \code
*        SbBox3f volext = volumeData.extent.getValue();
*     \endcode
*     \endif
*     \par 
*     Notes:
*     - Modify extent: The application can change the volume extent (but not the dimensions) at
*       any time. This changes the bounding box of the volume and scales the
*       volume visualization nodes (SoOrthoSlice, etc) associated with the volume.
*       Changing the volume extent effectively changes the voxel size.  This is commonly used,
*       for example, to scale a seismic volume along the time axis. @BR @BR
*     - Transform nodes: The volume extent and orientation (like geometry) can be modified by
*       transformation nodes in the scene graph and this in turn modifies the
*       appearance of volume visualization nodes (SoVolumeRender, SoOrthoSlice, etc).
*       However the same transformation must be applied to the volume data node
*       as well as the volume rendering nodes associated with that volume.  Effectively
*       any transformation nodes that affect the volume must be placed @B before @b
*       the volume data node. @BR @BR
*     - DICOM: For some data formats, including DICOM, the volume 'position' is considered to
*       be the @I center@i of the first voxel. VolumeViz considers the volume extent to include
*       @I all@i of the first and last voxels.  Therefore the extent 'min' is the outside
*       corner of the first voxel. @BR
*       NOTE: Open Inventor versions < 9.8 do not automatically set the volume extent based on
*       the Image Position Patient attribute.  Use the MedicalHelper method dicomAdjustVolume.
*
*   - Voxel size/spacing @BR
*     If the volume data is uniformly sampled, the voxel size is the volume extent
*     divided by the volume dimensions.  The voxel size can be different for each 
*     axis (uniform along each axis). However VolumeViz is not limited to uniformly 
*     spaced voxels.  VolumeViz also supports "rectilinear" coordinates where the 
*     voxel positions are explicitly given for each axis and also supports "projection"
*     using the SoProjection node (with some limitations).  This allows rendering of 
*     volumes defined, for example, in cylindrical coordinates or geographic coordinates.
*
*   - Data type @BR
*     VolumeViz supports volumes containing scalar data as signed and unsigned integer
*     values (byte, short, int) or floating point values.  VolumeViz also supports
*     volumes containing RGBA data (explicit color value for each voxel).  In this case
*     the data type is always unsigned integer.  You can query if the volume contains
*     explicit RGBA values by getting the value of the #dataRGBA field.  The data type is
*     determined by the reader (or when setting the #data field).  You can query the
*     data type and/or number of bytes per voxel using methods inherited from SoDataSet.
*     For example:
*     \par
*     \if_cpp
*     \code
*       int bytesPerVoxel = volumeData->getDataSize();
*       SoDataSet::DataType type = volumeData->getDataType();
*     \endcode
*     \endif
*     \if_dotnet
*     \code
*       int bytesPerVoxel = volumeData.GetDataSize();
*       SoDataSet.DataTypes type = volumeData.GetDataType();
*     \endcode
*     \endif
*     \if_java
*     \code
*       int bytesPerVoxel = volumeData.getDataSize();
*       SoDataSet.DataTypes type = volumeData.getDataType();
*     \endcode
*     \endif
*
*   - Data range @BR
*     In volumes using data types larger than byte, the actual range of data values
*     is usually smaller than the range of the data type. The application can
*     use an SoDataRange node to specify the range of values that will be mapped
*     into the transfer function.  For some formats the application can get min and
*     max values from the meta-data. For DICOM data, see SoVRDicomData or the
*     MedicalHelper method dicomAdjustDataRange(). In any case, the application can
*     query the actual minimum and maximum values using the getMinMax methods.  For example:
*     \par
*     \if_cpp
*     \code
*       double minval, maxval;
*       SbBool ok = volumeData->getMinMax( minval, maxval );
*     \endcode
*     \endif
*     \if_dotnet
*     \code
*       double minval, maxval;
*       bool ok = volumeData.GetMinMax( out minval, out maxval );
*     \endcode
*     \endif
*     \if_java
*     \code
*       double[] minmax = volumeData.getDoubleMinMax();
*     \endcode
*     \endif
*     \par
*     Note 1: These methods might force VolumeViz to load the entire data set if the
*     volume reader does not implement the getMinMax query. For example in an LDM
*     format data set, the min and max values are stored in the LDM header, so the
*     query is very fast. For other data sets VolumeViz may be forced to load the
*     entire data set and scan all the values to compute the min and max values. @BR
*     Note 2: Changing the data range (SoDataRange) may be slow if the size of the
*     data values is larger than the SoDataSet::texturePrecision setting. For example
*     32 bit values with texturePrecision = 8 or 16. In this case the data values
*     must be re-scaled and resent to the GPU when the data range is changed.
*
*   - Other: @BR
*     Many other volume properties can be specified using fields of SoDataSet and
*     SoVolumeData. The #ldmResourceParameters field contains an
*     SoLDMResourceParameters object that controls, for example, the amount of
*     CPU memory and GPU memory that the volume can use.
*
*  @B Basic volume visualization tools: @b
*
*   - The volume data is specified by an SoVolumeData node.  (Multiple data sets
*     can also be specified.  See the next section.)
*
*   - The region of voxels to be rendered (the "region of interest") can be
*     specified using an SoROI node.  This node limits the extent of all
*     rendering nodes including slices.
*
*   - The range of data values to be mapped into the transfer function can be
*     specified using an SoDataRange node.
*
*   - The color and opacity associated with each data value can be specified
*     using an SoTransferFunction node.
*
*   - The base material properties of the voxels, e.g. emissive and specular
*     color, can be specified using an SoMaterial node.
*
*   - Advanced rendering effects, e.g. lighting, and image quality settings
*     can be specified using an SoVolumeRenderingQuality node.
*     (Note: You should always enable lighting using this node, not the lighting field of
*     SoVolumeRender.)  Features include:
*     - Lighting (gradient or deferred)
*     - Ambient occlusion
*     - Jittering (anti-aliasing)
*     - Gradient computation options
*     - Interpolation options
*     - Edge enhancement effects
*     - Cube rendering of voxels
*
*   - The SoVolumeDataDrawStyle node specifies the volume rendering style(s) for
*     a volume or for a voxel region defined by an SoVolumeMask node.  The available
*     rendering styles are hidden (not rendered), direct volume rendering, a set of
*     isosurfaces and/or the region boundary surface.
*
*   - Shadows can be enabled for volume rendering using an SoShadowGroup node.
*
*   - Many parameters related to memory management and performance optimizations
*     can be specified using SoLDMGlobalResourceParameters and SoLDMResourceParameters.
*
*   - VolumeViz provides many different ways of visualizing volume data, including:
*     - SoOrthoSlice: Renders a single axis aligned slice.
*     - SoObliqueSlice: Renders a single arbitrarily aligned slice.
*     - SoFenceSlice: Renders a "strip" of connected oblique slices.
*     - SoVolumeSkin: Renders the faces of the current Region of Interest (see SoROI).
*       Effectively a set of ortho slices.
*     - SoVolumeIndexedFaceSet (etc): Renders a "slice" made up of arbitrary
*       geometry, for example a curved or cylindrical slice.
*     - SoVolumeIsosurface: Renders the isosurface defined by the specified data
*       value. (This is done completely on the GPU. To get the geometry of an
*       isosurface back on the CPU see the MeshViz extension.)
*     - SoVolumeRender: Renders the volume using direct volume rendering.
*
*   @B Advanced volume visualization: @b
*
*   - Custom shaders
*     Custom transfer functions, custom rendering effects and custom blending are
*     just a few of the many possibilities that can be
*     specified using an SoVolumeShader node and GLSL shader functions.
*     VolumeViz provides a framework of prebuilt shader functions for commonly
*     used calculations and effects. This allows applications to extend or replace
*     stages in the rendering pipeline while still taking advantage of other
*     VolumeViz features.
*
*   - Clipping @BR
*     Volume visualizations can be clipped in multiple ways:
*       - Volume visualizations can be clipped by clipping planes (SoClipPlane) like
*         any other Open Inventor geometry. (However for clipping against axis
*         aligned planes, using an SoROI node is much more efficient.)
*       - Volume visualizations can be clipped against a box specified in voxel
*         coordinates using an SoROI node.  The box can be modified by a secondary
*         exclusion box, allowing cut-away views like the "chair cut" in seismic
*         visualization.
*       - Volume visualizations can be clipped against arbitrary polygonal geometry
*         using an SoVolumeClippingGroup node.  The clipping geometry can be any
*         closed shape, for example a cylinder or a shape extruded from a polygon.
*       - Volume visualizations can be clipping against a surface defined by a "height
*         field" (for example a seismic horizon) using an SoUniformGridClipping node.
*       - Volume visualizations can be clipped against an arbitrary voxel region using
*         an SoVolumeMask node.  SoVolumeMask also allows applying different transfer
*         functions to different regions and other powerful features.
*       - Multiple clipping techniques may be combined to "sculpt" volume regions.
*
*   - Interactivity @BR
*     VolumeViz allows the creation of highly interactive volume rendering applications:
*     - Picking works for volume rendering objects like for any other Open Inventor
*       geometry.  For example the application can determine which slice was clicked
*       when a mouse button event occurs. Specialized sub-classes of SoDetail provide
*       more specific information for each rendering class.  For example SoOrthoSliceDetail
*       provides the IJK coordinates and value of the specific voxel under the cursor.
*     - VolumeViz provides specialized draggers and manipulators to allow users to
*       directly interact with volume rendering scenes. For example, SoOrthoSliceDragger
*       allows dragging a slice through the volume and SoROIManip allows moving and
*       resizing a region of interest (SoROI).
*     - Modifying VolumeViz property nodes (like any Open Inventor node) automatically
*       causes the scene graph to be re-rendered. For example the color map can be
*       dynamically changed by modifying the fields of SoTransferFunction.
*     - The volume itself can be modified interactively using the "volume edit" methods
*       (#startEditing, #finishEditing, etc) in this class. Volume editing supports undo 
*       and redo and modifications can be saved back to the data source.  Convenience 
*       methods are provided to modify voxels in a subvolume, a tile or a region defined 
*       by a closed polygonal surface.
*     - Very large volumes and/or advanced rendering options may reduce the rendering
*       performance even on high-end graphics hardware. VolumeViz provides multiple
*       options for temporarily reducing the rendering complexity while interacting with
*       the scene. See the lowResMode field in SoVolumeRender and also the
*       SoInteractiveComplexity node.
*
*   - Transforming data @BR
*     Volume data can transformed "on the fly" at several stages in the pipeline:
*       - The SoLDMDataTransform class (see the dataTransform field of SoDataSet)
*         applies a computation to each LDM data tile requested from the volume reader
*         before the tile is stored in system memory. This can be used to apply
*         "static" filters to the data, for example to apply DICOM Rescale Slope
*         and Intercept values.
*       - The SoVolumeTransform node applies a computation to the LDM data tiles just
*         before they are sent to the GPU. This can be used to apply dynamic filters
*         to the data, for example computing seismic attribute values.
*       - Both mechanisms can be used to create multiple data sets from a single data set.
*
*   - Multiple data sets @BR
*       - Multiple SoVolumeData nodes can be inserted in the same scene graph.  
*       - If the volumes are independent and rendered separately, use an SoVolumeGroup node 
*         to manage and correctly render intersecting regions.  
*       - More commonly multiple volumes will be combined into a single data set or single 
*         rendering using @I data compositing@i or @I render compositing@i. In these cases 
*         use an SoMultiDataSeparator as the parent of the nodes that will be composited.
*       - Render compositing (see SoVolumeShader) means combining multiple volumes on the 
*         GPU at render time using a fragment shader. Render compositing can be used, for 
*         example, to implement "co-blending" of multiple volumes or to implement 
*         multi-channel color combining.
*       - Data compositing allows you to combine multiple volumes (see SoDataCompositor) 
*         on the CPU at data loading time.  Data compositing can be used, for example, to 
*         visualize the difference between two data sets.
*       - (More details below)
*
*   - Data Access @BR
*     LDM is a powerful data manager for large volume data and VolumeViz provides a 
*     Data Access API that allows applications to take advantage of this for applying 
*     algorithms to their volume data.
*       - SoLDMDataAccess provides the methods to extract data from a volume. The data is 
*         accessible whether the SoVolumeData is part of a scene graph or not.
*       - Data access methods can be invoked synchronously or asynchronously to allow 
*         simultaneous loading and computation. The application can request data at any 
*         resolution level, independent of the resolution level currently being used for 
*         rendering
*       - VolumeViz supports a variety of data requests including:
*           - Subvolume: The set of voxels inside a specified subvolume.
*           - Plane: The set of voxels intersecting an arbitrary plane.
*           - Line: The set of voxels intersecting an arbitrary line.
*           - Trace: A column of axis aligned voxels (e.g. a seismic trace).
*           - Polyline: The set of voxels intersecting an arbitrary polyline.
*           - Tile: Direct access to the tile containing a specified voxel.
*       - Extracted, modified or synthesized data can be written to an LDM format file
*         by subvolume or tile using the SoLDMWriter class.
*
*   - Computation @BR
*     Volume computations can take advantage of the Open Inventor computing framework to 
*     manage data, devices and algorithms on the CPU and GPU.
*       - The SoBufferObject class abstracts data management.  Each buffer object is an 
*         instance of a device specific data object, for example SoCpuBufferObject. 
*         This allows the application to control where memory is 
*         actually allocated.  It also allows algorithms to be implemented on CPU or GPU 
*         (or both) using SoBufferObject without knowing where the memory is allocated 
*         (Open Inventor will transfer when necessary).
*       - The SoDevice class abstracts computing and rendering devices.  Device specific 
*         classes, for example SoCpuDevice, allow querying the number of 
*         available devices, available memory and other properties.
*       - The computing framework classes are integrated with the data transform, 
*         compositing and data access classes mentioned previously.
*
*  @BR @B Multiple data sets: @b
*
*   Multiple SoVolumeData nodes can be inserted in the same scene graph.
*   If the volumes are independent and rendered separately, use an SoVolumeGroup
*   node to manage and correctly render intersecting regions. More commonly multiple
*   volumes will be combined together in a single rendering using render
*   compositing (SoVolumeShader) or data compositing (SoDataCompositor).
*   In these cases you must use an SoMultiDataSeparator
*   as the parent of the nodes that will be composited.
*
*   Render compositing (SoVolumeShader or SoVolumeRenderingQuality) is a way of combining
*   multiple volumes on the
*   GPU at render time using a GLSL fragment shader. The volumes can each have their 
*   own transfer function or they can all use the same one.
*   Render compositing can be used, for example, to implement "co-blending" of multiple
*   volumes or to implement multi-channel color combining.
*   The number of volumes to compose is limited by the number of OpenGL texture units supported by the
*   graphics board (normally at least 16). This number is returned by the #getMaxNumDataSets function.
*
*   Data compositing allows you to combine multiple volume data sets
*   (see SoDataCompositor) or
*   to transform a single data set in memory (see #setLDMDataTransformFunction()) instead
*   of storing the combined data sets on disk.
*   For example, it can be used to visualize the result of the difference between two data sets.
*   There is no limit on the number of volumes that can be composed on the CPU.
*
*   Note that the word @I composition@i is also used in SoVolumeRender. There it refers to the
*   way that samples along the raycasting ray are combined to form the final image.
*
*   The #dataSetId field is used to differentiate SoVolumeData nodes when doing
*   render or data compositing.
*
*   Some rules must be observed when doing render or data compositing:
*
*   - Each SoVolumeData node must have a unique #dataSetId. @BR
*
*   - All the SoVolumeData nodes to be composited must have the same volume dimensions
*     (number of voxels in X, Y, and Z) and tile size. @BR
*
*   - All the SoVolumeData nodes to be composited, as well as the compositing node
*     (e.g. SoVolumeShader or SoVolumeRenderingQuality) and the rendering node
*     (e.g. SoVolumeRender), must be under an SoMultiDataSeparator node.
*
*   - The SoVolumeData nodes to be composited must be all scalar data sets or all RGBA data
*     sets.  To composite scalar and RGBA data sets under the same SoMultiDataSeparator,
*     set the @I usePalettedTexture@i field to false in the scalar dataset's
*     SoVolumeData node to force the scalar data to be converted into RGBA data.
*
*   - An SoVolumeData node used in a data compositing scheme must not be inserted multiple
*     times in the scene graph. Use another volume data node pointing to the same file.
*
*   - Each SoVolumeData node has its own resource settings (see field #ldmResourceParameters).
*     The resources required for the composition are the sum of the resources for
*     all of the SoVolumeData nodes involved. @BR
*
*   When using a fragment shader to do render compositing, texture coordinates can be retrieved from
*   texture unit 0 (texture coordinates are sent using glTexCoord function).
*   To minimize the number of texture units needed, all the transfer functions (see
*   SoTransferFunction) for the volumes to be composited are stored in a single 2D texture.
*   By default this texture is loaded in texture unit 0.  However this default behavior can be
*   changed through SoPreferences using the environment variable IVVR_TF_TEX_UNIT.
*   Each volume's data is loaded in the texture unit specified by its #dataSetId.  Therefore
*   do not set #dataSetId to the texture unit used to store the transfer functions.
*
*   Please see SoMultiDataSeparator and SoVolumeShader for more information, and example
*   code, for compositing multiple volumes.
*
* @BR @B RGBA Data @b
*
*   Voxels in an RGBA volume are UNSIGNED_INT32, containing 8 bits each of Red, Green, Blue
*   and Alpha. All rendering nodes (slices, volume rendering, etc) work with RGBA volumes.
*   Region of Interest, clipping and other features also work with RGBA volumes.  However
*   because the volume already specifies the colors to be used for rendering, the data
*   range, transfer function and some rendering features are ignored.  Lighting works with
*   RGBA volumes using gradient vectors computed from the luminance value of the voxels.
*
*   Notes:
*   - In-memory volumes @BR
*     For compatibility with older versions of Open Inventor, if the data is
*     in-memory the unsigned int values @I must@i contain ABGR values, not RGBA.
*     Use the SbColor getPackedValueEndiannessOrder() method to convert values, @I not@i
*     the getPackedValue() method.
*
*   - Texture compression @BR
*     The #useCompressedTexture option (see SoDataSet) is true by default. This can significantly
*     reduce the amount of GPU memory required to hold an RGBA volume. But it uses a "lossy"
*     compression algorithm which may result in values on the GPU being slightly different than
*     the actual value in the volume. For example "gray scale" colors could have a slight color
*     in some cases. You should always set this option to false when using "RGBA" data to store
*     32-bit "id" values. Please also note that this compression is not supported on Intel GPUs,
*     so you should set this option to FALSE in this case as well.
*
* @BR @B Volume Editing @b
*
* Volume editing is based on transactions. A transaction is created by calling the following methods:
*     \par
*     \if_cpp
*     \code
*       int transationId;
*       volumeData->startEditing( transationId );
*       //... editing calls ...
*       volumeData->finishEditing( transationId );
*     \endcode
*     \endif
*     \if_dotnet
*     \code
*       int transactionId;
*       volumeData.StartEditing(out transactionId);
*       //... editing calls ...
*       volumeData.FinishEditing(transactionId);
*     \endcode
*     \endif
*     \if_java
*     \code
*       int transactionId = volumeData.startEditing();
*       //... editing calls ...
*       volumeData.finishEditing(transactionId);
*     \endcode
*     \endif
* Starting a transaction returns a unique id identifying the current transaction. 
* This id is used to finish the transaction by calling \if_dotnet FinishEditing(). \else finishEditing(). \endif
* After the transaction is finished, this id can also be used to undo or redo the transaction.
* The finish method will schedule a redraw so the modified data is displayed.
* Multiple transactions may be active at the same time. @BR
*
* After a transaction is finished, the effect can be undone by calling 
* \if_dotnet UndoEditing() \else undoEditing() \endif with the transaction id.
* Undo also schedules a redraw so the correct data is displayed.
* Similarly a transaction that was undone can be re-applied by calling
* \if_dotnet RedoEditing() \else redoEditing() \endif with the transaction id.
*
* Calling \if_dotnet SaveEditing() \else saveEditing() \endif will update the data source
* associated with this SoVolumeData node with all edits performed on the volume
* since the last save.  Updating is done using the SoVolumeWriter returned by the current
* SoVolumeReader.  The save method may only be called when no transactions are active,
* i.e. after finish has been called for all transactions. @BR
* Note: 
* - Until the edits have been saved, memory is required for both the original data
*   and the modified data. This implies both an additional memory requirement and
*   that buffer objects passed to (for example) editSubVolume() must not be modified
*   until after the saveEditing() call.
*
* - After the edits have been saved, all transaction ids are invalid, so undo and
*   redo can no longer be called with those transaction ids.
*
* - saveEditing currently only works for data loaded using SoVRLDMFileReader (an LDM
*   format tiled data set) or SoVRMemoryReader (data set completely in memory).
*
* - VolumeViz applies edition transaction on the fly each time a tile is requested.
*   Thus, loading, data access and picking performance may be impacted.
*   To retrieve initial performances, you may have to call \if_dotnet SaveEditing() \else saveEditing() \endif
*   to flush editing transactions.
*
* VolumeViz provides multiple methods to modify data including:
* - Fill a subvolume with a value or a buffer of values (\if_dotnet EditSubVolume() \else editSubVolume() \endif)
* - Fill a tile with a specified value or a buffer of values (\if_dotnet EditTile() \else editTile() \endif)
* - Fill a list of cubic subvolumes with a value (\if_dotnet EditBoxes() \else editBoxes() \endif)
* - Fill the voxels inside a closed polygonal solid with a value (\if_dotnet EditSolidShape() \else editSolidShape() \endif)
* - Fill the voxels intersecting a polygonal surface with a value (\if_dotnet EditSurfaceShape() \else editSurfaceShape() \endif)
*
* @BR @B Supported file formats: @b
*
*   @TABLE_1B
*   @TR @B File extension @b  @TD @B Reader class @b     @TD @B Description @b
*   @TR .am                   @TD SoVRAmFileReader       @TD Avizo Mesh file format
*   @TR .dc3, .dic, .dicom    @TD SoVRDicomFileReader    @TD DICOM file format
*   @TR .fld                  @TD SoVRAvsFileReader      @TD AVS field file format
*   @TR .lda or .ldm          @TD SoVRLdmFileReader      @TD LDM file format
*   @TR .sgy or .segy         @TD SoVRSegyFileReader     @TD SEG Y rev 1 file format
*   @TR .vol                  @TD SoVRVolFileReader      @TD Vol file format
*   @TR .vox                  @TD SoVRVoxFileReader      @TD Vox file format
*   @TR .lst                  @TD SoVRRasterStackReader  @TD Lst file format
*   @TABLE_END
*
*   File format notes:
*   - Avizo mesh @BR
*     Avizo mesh is a general purpose file format that can contain many different kinds of data. The
*     VolumeViz file reader can load Avizo mesh files containing a 3-dimensional "Lattice" data object
*     with uniform coordinates and any data type.  See SoVRAmFileReader for limitations. @BR @BR
*     Note: Unlike Amira and Avizo, VolumeViz cannot automatically open Amira/Avizo format data files
*     unless they have the file name extension ".am".  To open an Amira/Avizo data file with a different
*     extension, the application must explicitly create an instance of SoVRAmFileReader and use the
*     setReader() method.
*
*   - AVS field @BR
*     AVS field is a general purpose file format that can contain many different kinds of data. The
*     VolumeViz file reader can load AVS field files containing 3-dimensional, uniform data of type
*     "byte". See SoVRAvsFileReader.
*
*   - DICOM @BR
*     A widely used format for storing medical image data (CT, MRI, etc), defined by the National
*     Electrical Manufacturers Association (NEMA) (medical.nema.org). See SoVRDicomFileReader
*
*   - LDM @BR
*     LDM is a format defined by VSG for storing hierarchical multi-resolution volume data.
*     VolumeViz includes a utility program that can convert any other format supported by VolumeViz
*     into this format (see SoVolumeConverter). Preprocessing volume data into this format provides
*     the maximum benefits from the VolumeViz large data management (LDM) features. See SoVRLdmFileReader.
*
*   - SEGY @BR
*     A widely used format for storing seismic trace data, defined by the Society of Exploration
*     Geophysicists publication "Digital Tape Standards" (www.seg.org). The VolumeViz reader
*     supports all sizes of integer and float data, and can correctly determine the number of
*     samples per trace in many cases. However the reader also has many options to adapt to
*     differences in SEGY file headers.  See SoVRSegyFileReader.
*
*   - VOL @BR
*     A simple volume interchange format (see "Introduction to Volume Rendering", Lichtenbelt,
*     Crane, Naqvi, 1998). The VolumeViz reader can load files containing 8- or 16-bit voxels.
*     See SoVRVolFileReader.
*
*   - VOX @BR
*     A volume interchange format defined by TeraRecon Inc. (www.terarecon.com). The VolumeViz
*     reader can load "Vox1999a" files containing 8- or 16-bit voxels (first volume only).
*     See SOVRVoxFileReader.
*
*   - LST (stack of images) @BR
*     A simple format for loading a stack of images.
*     Specify the names of the image files in a .lst file.  VolumeViz can load image data in
*     most common image formats including BMP, DDS, GIF, JPEG, JPEG2000, PNG and TIFF. See
*     SoVRRasterStackReader for details and limitations.
*
*   Note: '3D TIFF' files (multiple images in one file) are not currently supported.
*
* @FILE_FORMAT_DEFAULT
*    VolumeData {
*    @TABLE_FILE_FORMAT
*       @TR allocateResourceOnRender   @TD FALSE
*       @TR data                       @TD NODATA 0 0 0 UBYTE 8
*       @TR dataRGBA                   @TD FALSE
*       @TR dataSetId                  @TD 1
*       @TR dataTransform              @TD NULL
*       @TR extent                     @TD -1 -1 -1 1 1 1
*       @TR fileName                   @TD ""
*       @TR texturePrecision           @TD 0
*       @TR undefinedValue             @TD NaN
*       @TR useCompressedTexture       @TD TRUE
*       @TR useExtendedData            @TD FALSE
*       @TR usePalettedTexture         @TD TRUE
*       @TR useSharedPalettedTexture   @TD TRUE
*    @TABLE_END
*    }
*
* @ACTION_BEHAVIOR
* SoCallbackAction,
* SoGLRenderAction,
* SoGetBoundingBoxAction,
* SoPickAction,
* SoWriteAction @BR
* Sets volume data parameters in the traversal state.
*
* @SEE_ALSO
*    SoVolumeRender,
*    SoOrthoSlice,
*    SoObliqueSlice,
*    SoVolumeReader,
*    SoVolumeSkin,
*    SoDataCompositor,
*    SoLDMGlobalResourceParameters,
*    SoLDMResourceParameters
*
*/
class VOLUMEVIZ_API SoVolumeData : public SoDataSet, public SoVolumeRendering
{
  SO_NODE_HEADER( SoVolumeData );

  SO_PIMPL_PUBLIC_HEADER( SoVolumeData );

public:
  using SoDataSet::getTexData;

  //compatibility issues:
  typedef SoLDMDataAccess LDMDataAccess;
  typedef ::SoLDMResourceParameters SoLDMResourceParameters;

  /**
  * Constructor.
  */
  SoVolumeData();

 /**
  * For a volume containing scalar data values, controls whether scalar values
  * (true) or RGBA values (false) are loaded on the GPU (the name is historical).
  *
  * The default is true, meaning to load scalar values on the GPU.  This implies
  * that the GPU will interpolate between data values and the GPU will apply the
  * color map.  Using scalar values has several advantages and is the recommended
  * setting in most cases.  Loading scalar values (generally) uses less GPU
  * memory per voxel and the color map can be changed interactively because no
  * volume data needs to be reloaded on the GPU.  Also note that interpolating
  * between color values (rather than data values) could result in unexpected
  * colors that are not even part of the color map.  The size (in bytes) of the
  * scalar values on the GPU is determined by the #texturePrecision field.
  *
  * If this field is false, the color map is applied on the CPU and the resulting
  * RGBA values are loaded on the GPU.  This implies that the GPU will interpolate
  * between color values.  There are a few cases where this setting is better.
  * For example, interpolating between data values could create values that not
  * actually in the data set.  Also, applying the color map on the CPU means that
  * a very large color map can be used because the size is not limited by the
  * maximum texture size on the GPU.
  *
  * If the volume contains explicit RGBA values, then this field is ignored and
  * RGBA values are sent to the GPU.
  */
  SoSFBool   usePalettedTexture;


  /**
  * Note: On graphics boards that support programmable shaders, this field is ignored
  * (virtually all graphics boards support programmable shaders).
  * @BR
  * Controls use of the OpenGL shared texture palette extension (if available). FALSE
  * means don't use it. Default is TRUE. On machines that support paletted textures,
  * using a shared palette conserves texture memory because a single palette (color map)
  * can be used for all the textures.
  */
  SoSFBool   useSharedPalettedTexture;

  /**
   *
   * If TRUE, VolumeViz stores an additional copy of each loaded tile.
   * There is an additional memory cost but the performance of SoOrthoSlice roaming will
   * be roughly the same along each axis.
   *
   * @FIELD_SINCE_OIV 6.0
   */
  SoSFBool   useExtendedData;

  /**
   * Specifies the volume data, including dimensions, data type and number of significant bits.
   * Use this field if your volume data is already loaded into memory in a contiguous block
   * of memory.  Otherwise use the #fileName field or the #setReader method.
   *
   * The @B numSigBits @b parameter of SoSFArray3D::setValue indicates the number of bits really
   * used for each value.
   * If it equals 0, that means to use all bits, e.g., 8 bits if @B type @b = UNSIGNED_BYTE,
   * 16 bits if @B type @b = UNSIGNED_SHORT and so on. This parameter is useful particularly when
   * textures are loaded in paletted mode. Currently most hardware supports only 8 bits,
   * so VolumeViz must discard the least significant bits. For example if @B type @b is
   * UNSIGNED_SHORT, by default in paletted mode VolumeViz will discard the lowest 8 bits. If
   * you specify @B numSigBits @b = 10, VolumeViz will discard only the lowest 2 bits.
   *
   * Calling SoSFArray3D::setValue with a CopyPolicy set to NO_COPY is equivalent to calling the deprecated
   * SoVolumeData::setValue method.
   *
   * Volume reader: Note that when a file format reader or custom volume reader is
   * currently being used, setting the data field will automatically replace the current
   * volume reader. Subsequent calls to getReader() will return an SoVRMemoryReader.
   *
   * @FIELD_SINCE_OIV 7.1
   */
  SoSFArray3D data;

  /**
   * Contains TRUE if the volume contains RGBA values rather than scalar values.
   * Default is FALSE.
   *
   * This field is set automatically by volume readers. If the application is setting
   * an in-memory volume into the #data field, the #data field must be set to
   * UNSIGNED_INT32 format and the #dataRGBA field must be set to TRUE.
   *
   *
   *   Note that, for compatibility with older versions of Open Inventor, if the data is
   *   in-memory the the unsigned int values @I must@i contain ABGR values, not RGBA.
   *   Use the SbColor getPackedValueEndiannessOrder() method to convert values, @I not@i
   *   the getPackedValue() method.
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFBool dataRGBA;

  /** @copydoc SoDataSet::getMinMax(int64_t &min, int64_t &max) */
  virtual SbBool getMinMax( int64_t &min, int64_t &max);

  /** @copydoc SoDataSet::getMinMax(double &min, double &max) */
  virtual SbBool getMinMax( double &min, double &max);

  /**
  * Returns the histogram of the volume data. 
  * Returns FALSE if the requested data is not available.
  *
  * Notes:
  * - Byte @BR
  *   For byte data, VolumeViz computes a histogram with 256 bins, 
  *   spanning the range of byte values (0..255 or -128..127).
  * - Integer @BR
  *   For integer data types larger than 1 byte, VolumeViz computes
  *   a histogram with 65536 bins, spanning the range of the data type.
  *   For example, if the volume contains signed_short data, the 
  *   histogram has 65536 bins spanning the range -32768..32767.
  * - Float @BR
  *   For float data, VolumeViz computes a histogram with 65536 bins,
  *   spanning the specified data range or -20000 to 20000 if no range was specified.
  *   The data range can be specified when converting to LDM format, using the
  *   converter's "-r" option.  Otherwise it is not currently possible to specify
  *   the data range for the histogram without using SoVolumeHistogram directly.
  * - Bins @BR
  *   For all data types larger than one byte, only the bins that
  *   are inside the actual range of the data are interesting.  For
  *   example a signed_short data set might range from -2048 to 5000.
  *   In this case the first non-empty bin is number 30720 (out of
  *   65536 total bins) and the last non-empty bin is number 37827.
  * - New histogram @BR
  *   To compute a histogram (for example with different parameters or for a subvolume)
  *   use the SoVolumeHistogram class.
  *
  * Warnings:
  * - This method returns a reference to VolumeViz internal storage. Do not modify this memory.
  * - Histogram computation @BR
  *   This method might force VolumeViz to load the @I entire @i data set if
  *   the volume reader does not respond to the getHistogram query. Normally for an
  *   LDM format data set, the histogram values are stored in the LDM header.
  *   For a non-LDM data set, if a filename and/or reader have been specified and
  *   the data set has not yet been loaded, VolumeViz will load the entire data set to
  *   compute these values. For a large data set this may take a long time.
  *
  * [OIV-WRAPPER-CUSTOM-CODE]
  */
  SbBool getHistogram( int &length, int64_t* &histogram);

  /**
  * Updates regions of the volume that have been modified.
  * This method notifies VolumeViz that voxels in the specified regions
  * have been modified by the application and textures may need to be
  * recreated.  Only bricks currently needed for rendering some volume
  * geometry, e.g., a slice or subvolume, are immediately updated.
  * [OIV-WRAPPER-ARG ARRAY,NO_WRAP{(region != NULL? region->get_Length(): 0)}]
  */
  void updateRegions(const SbBox3i32 *region, int num_regions);

  /**
  * Reloads textures corresponding to the given tile IDs. Only supported in LDM mode.
  * Useful for a bottom-up approach (a custom tile manager starts loading full resolution
  * and returns fake data for lower resolution. Once the real low resolution data is in memory,
  * update).
  * [OIV-WRAPPER-ARG ARRAY,NO_WRAP{(tiles != NULL? tiles->get_Length(): 0)}]
  */
  void updateTilesInTextureMemory(SoLDMTileID* tiles, int numTiles);

  /**
   * Force data to be considered as RGBA values. @BR
   * Note that this call should be used only when setting data from memory
   * through the #data field and the data format must be UNSIGNED_INT32.
   */
  void setRGBAData(const bool flag);

  /**
   * Coordinate type used by this data set.
   */
  enum CoordinateType {
  /**
   * Uniform grid spacing along each axis.
   */
	COORDINATES_UNIFORM = SoVolumeReader::COORDINATES_UNIFORM,
  /**
   * Grid spacing defined by x, y, z values.
   */
	COORDINATES_RECTILINEAR = SoVolumeReader::COORDINATES_RECTILINEAR
  };

  CoordinateType getCoordinateType() const;

  /**
   * Which axis to handle
   * @see getRectilinearCoordinates
   */

  enum Axis {
    X,
    Y,
    Z
  };

  /**
   * Returns a vector describing mapping from uniform space to
   * rectilinear space. You can specify that you want
   * the vector to return the coordinates from the X, Y, or Z axis.
   */
  const std::vector<float>& getRectilinearCoordinates(Axis axis) const;

  /** @copydoc SoDataSet::startEditing() */
  virtual SbBool startEditing(int &transactionId );

  /** @copydoc SoDataSet::editTile(const SoLDMTileID&, SoBufferObject*) */
  virtual int editTile( const SoLDMTileID& tileId, SoBufferObject* userData );

  /** @copydoc SoDataSet::editSubVolume(const SbBox3i32&, SoBufferObject*) */
  virtual int editSubVolume( const SbBox3i32& subVolume, SoBufferObject* userData );

  /** @copydoc SoDataSet::editTile(const SoLDMTileID&, const double&) */
  virtual int editTile( const SoLDMTileID& tileId, const double& value );

  /** @copydoc SoDataSet::editSubVolume(const SbBox3i32&, const double&) */
  virtual int editSubVolume( const SbBox3i32& subVolume, const double& value );

  /** @copydoc SoDataSet::editSurfaceShape */
  virtual int editSurfaceShape( const SoNode* surfaceShape, const float& thickness, const double& newValue );

  /** @copydoc SoDataSet::editSolidShape */
  virtual int editSolidShape( const SoNode* solidShape, const double& value );

  /** @copydoc SoDataSet::editBoxes */
  virtual int editBoxes(
    const std::vector<SbVec3i32>& boxCenters,
    const int& boxSize,
    const double& newValue
    );

  /** @copydoc SoDataSet::finishEditing() */
  virtual SbBool finishEditing(int transactionId);

  /** @copydoc SoDataSet::undoEditing() */
  virtual SbBool undoEditing( int transactionId );

  /** @copydoc SoDataSet::redoEditing */
  virtual SbBool redoEditing( int transactionId );

  /** @copydoc SoDataSet::saveEditing() */
  virtual SbBool saveEditing(bool recomputeLowerResolution = TRUE, const std::vector<char*> convertionParameters = std::vector<char*>(), SaveEditingCB* callback = NULL );

#if SoDEPRECATED_BEGIN(8600)

  /**
   * Storage Hints mode
   * TODO: this enum should be in the private section as its storageHints deprecated field coiunterpart BUT
   * it is used in other internal implementation place
   */
  enum SoDEPRECATED_ENUM(8600,"Field value will be ignored.")
  StorageHint {
    /** Auto (Default) */
    AUTO,
    /** Multi-texture 2D */
    TEX2D_MULTI,
    /** (synonym for TEXT2D_MULTI) */
    TEX2D = TEX2D_MULTI,
    /** Texture 3D */
    TEX3D,
    /** In memory */
    MEMORY,
#ifndef HIDDEN_FROM_DOC
    /** kept from 7.2, only for .NET binary compatibility issues */
    VOLUMEPRO,
    /** Single Texture 2D */
    TEX2D_SINGLE
#endif // HIDDEN_FROM_DOC
  };

#endif /** @DEPRECATED_END */


#if SoDEPRECATED_BEGIN(9000)

  /**
   * Sub sampling method mode
   */
  enum SoDEPRECATED_ENUM(9000,"No longer supported.")
  SubMethod {
    /** Nearest */
    NEAREST,
    /** Max */
    MAX,
    /** Average */
    AVERAGE
  };

  /**
   * Over sampling method mode
   */
  enum SoDEPRECATED_ENUM(9000,"No longer supported.")
  OverMethod {
    /** None */
    NONE,
    /** Constant */
    CONSTANT,
    /** Linear */
    LINEAR,
    /** Cubic */
    CUBIC
  };

  /**
  * Re-samples the volume down to or up to the given @B dimension @b using
  * the sub-sampling method #SubMethod() and the over-sampling method
  * #OverMethod(). If a reader is specified, the original volume data is not
  * loaded.
  *
  * Over-sampling is not yet implemented. NONE (default) indicates that no
  * over-sampling is performed. For example if the original dimension is 256x256x64
  * and the reSampling dimension is 128x128x128, the result is a volume of dimension
  * 128x128x64.
  */
  SoDEPRECATED_METHOD(9000,"No longer supported. Please use SoLDMDataAccess class.")
  SoVolumeData *reSampling( const SbVec3i32 &dimension, SoVolumeData::SubMethod subMethod, SoVolumeData::OverMethod = NONE);

  /**
  * Extracts the data volume defined by @B region@b. 
  * If a reader is specified, the original volume data is not loaded.
  */
  SoDEPRECATED_METHOD(9000,"No longer supported. Please use SoLDMDataAccess class.")
  SoVolumeData *subSetting( const SbBox3i32 &region );

  /**
  * Specifies the maximum texture memory size to use in mega texels. 64 is the default.
  */
  SoDEPRECATED_METHOD(9000,"No longer used. Please use SoLDMRessourceParameter maxTexMemory field.")
  void setTexMemorySize(int size);

  /**
   * Returns the maximum texture memory size to use in mega texels.
   */
  SoDEPRECATED_METHOD(9000,"No longer used. please use SoLDMRessourceParameter maxTexMemory field.")
  int getTexMemorySize();

#endif /** @DEPRECATED_END */

SoEXTENDER public:

  virtual void doAction( SoAction *action );
  virtual void GLRender( SoGLRenderAction *action );
  virtual void useFakeData(SoLDMTileID tileID, SoBufferObject* bufferObject);

  // ScaleViz specific : force rendering on all oiru in depth compositing mode
  // As VolumeViz shape nodes are considered as transparent (alpha)
  virtual int32_t getRenderUnitID() const;

SoINTERNAL public:

  /** Return voxel size of the volume data */
  SbVec3d getVoxelSize() const;

  virtual void updateTile( const SbBox3i32 &region, const SoLDMTileID& tileId );
  virtual void drawTilesInMainMemory(SoGLRenderAction* action, const SoLDM::DataSetIdPair& idPair);

  enum Cmd{
    CREATE_NODE,
    UPDATE_REGIONS,
    SET_LDM,
    SET_PAGING
  };

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  int64_t getValue (SbVec3i32 &pos, SbBool real = FALSE);
  double  getValueD(const SbVec3i32 &pos, SbBool real = FALSE);
  int64_t loadDataValue(void* value, const SbVec3i32 &pos, void*& data, SoBufferObject *dataBufferObject, SbBool real);

  static void initClass();
  static void exitClass();

  void             loadData(SoAction* action, SbBool& texLoadedOrReloaded);

  SoTransferFunction::GLColorMapType getColorMapType(SoState*);
  SbVec3i32&       getDimensionPowerOf2();
  SbVec3f&         getTexRatio();

  // CPU lighting
  void             setLighting(SoState* state, SbBool);
  void             setLightDirection(SoState* state,const SbVec3f&);
  void             setLightIntensity(SoState* state,float);

  virtual bool     isPalettedTexture(SoState* state);

  /** Compute texgen relative to the entire volume (instead of per-tile). */
  void getVolumeTexGen(SbVec4d texGen[3]) const;

  /** Compute texgen relative to the entire volume (instead of per-tile). */
  SbMatrix getVolumeTexGen() const;

  /** Return ratio between the volume dimension and the volume texture used */
  SbVec3f getVolumeTextureRatio() const;

  /** Return bbox in texture space */
  SbBox3f toTextureSpace(const SbBox3f& bbox) const;
  
  void drawOctree(SoGLRenderAction* pGLRA);

  //Return -1 when lighting is on, 0 in RGBA texture mode
  virtual int getTexPrec(SoState*state);

  virtual void getDataOneObliqueSlice( const SbVec3i32& tileSize, SoBufferObject* tileData, int Zaxis, const SbPlane& plane, SoBufferObject* slicedata, int interp );
  virtual SoBufferObject* getDataOneSlice( const SbVec3i32& tileSize, SoBufferObject* tileData, int Zaxis, int slice);

  SbBox3f getLDMOutliningBox();

  /**
   * Internal helper function which return LDM tile transformed depending on state
   * see SoVolumetransform for details.
   */
  virtual SoBufferObject *getTransformedTile(const SoLDMTileID& tile,
                                             const SoLDM::DataSetIdPair& p,
                                             SoState * state=NULL,
                                             const bool useExtendedData = false);


  /** Save editing using specified writer. For convenience, updateRegion is called at the end of the process.
   * Basically call doSaveEdtiing, clear editing stack and update display. */
  SbBool saveEditing( SoVolumeWriter* writer, SaveEditingCB* callback = NULL );

  /**
  * Returns a vector mapping back a rectilinear coordinates to uniform space.
  */
  const std::vector<float>& getInvertedRectilinearCoordinates(Axis axis) const;

  /**
  * Draw a colored box depending on its state and resolution.
  */
  virtual void drawXYZBox( SoGLRenderAction* action, BoxColor color, const SbBox3f& box );

  /**
  * Use the tileId to know if the tile is really full res
  */
  virtual void drawIJKBox( SoGLRenderAction* action, BoxColor color, const SbBox3f& box, const SoLDMTileID& tile, bool subTile = false );
  virtual void drawIJKBox( SoGLRenderAction* action, BoxColor color, const SbBox3f& box, bool fullRes, bool subTile = false );
  virtual void drawIJKBox( SoGLRenderAction* action, BoxColor color, const SbBox3f& box, bool subTile = false );

  void setupRectilinearCoordinates(const SbVec3i32&);
  bool isRectilinearCoordinatesUpdated(const SoVolumeShader* s);
  void resetRectilinearCoordinatesUpdated(const SoVolumeShader* s);

  SbVec3f computeRectilinearCoordinate(const SbVec3f& vertex);
  SbBox3f computeRectilinearCoordinate(const SbBox3f& box);

  SbVec3f computeInvertedRectilinearCoordinate(const SbVec3f& vertex) const;
  SbBox3f computeInvertedRectilinearCoordinate(const SbBox3f& box);

  void setDefaultValue( const double defValue );
  double getDefaultValue() const;

  /**
   * Compute xyz coordinate of given box
   */
  SbBox3f getXYZCoordinates(const SbBox3i32& box, const SbBox3i32& cropped);

  /**
   * Return world-space coordinates of the specified data-space box
   * If cropped[i] is true, the box has been cropped on axis i
   */
  SbBox3f getSubtileXYZCoordinates(const SbBox3i32& fullTileBox,
                                   const SbBox3i32& subTileBox,
                                   const SbBox3i32& cropped);

  virtual LDMSliceAccessor* getAppropriateSliceAccessor( const SoLDM::DataSetIdPair& p );

  virtual SoLDMReader* getAppropriateLDMReader( const SbString& pathname);

protected:

  /**
   * Reset the data field,
   * release memory and set the new reader
   */
  virtual void readerChanged();

  /**
   * Set the volume data element on the state
   */
  virtual void setElement(SoState* state);

  virtual SoVolumeReader *getAppropriateReader();

  // Destructor
  virtual ~SoVolumeData();
  void     releaseMemory();

  virtual bool enableMultidata(SoState* state);

  /**
   * Call data set unload and release paging.
   */
  virtual void unload();

  /**
   * Called when filename is changed
   */
  virtual void updateFilename();

  /**
   * Returns TRUE if the data represents RGBA color values.
   */
  virtual SbBool isRGBAData();

  /** @copydoc SoDataSet::buildTile */
  virtual void buildTile(SoLDMTileID tileID, SoBufferObject* buffer);

  /** @copydoc SoDataSet::computeLighting */
  virtual void  computeLighting( const SbVec3i32 & tileSize, SoBufferObject* rgbaBuffer );

SoINTERNAL protected :

  /** @copydoc SoDataSet::getAppropriateElement */
  virtual SoLDMGeomElement* getAppropriateElement( SoLDMGeometry* v );

private:

  /** Common code to constructors */
  void commonConstructor();

  SoDEPRECATED_FIELD(8600,"Field value will be ignored.")
  SoSFEnum	 storageHint;

  // This field is deprecated but cannot be remove without creating a new iv file version format
  SoDEPRECATED_FIELD(7000,"Use SoDataSet::#SoVolumeData::dataSetId instead.")
  SoSFInt32 volumeDataId;

  friend class ::SoVolumeState;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // _SO_VOLUME_DATA_


