/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef SO_VOLUME_GEOMETRY_H
#define SO_VOLUME_GEOMETRY_H

#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable:4251)
#endif

#include <Inventor/SbBox.h>
#include <Inventor/STL/vector>

#include <VolumeViz/nodes/SoVolumeShape.h>
#include <VolumeViz/nodes/SoSlice.h>

#include <LDM/actions/SoLdmValuationActionInterface.h>
#include <LDM/SoLDMTileID.h>
#include <LDM/actions/SoLdmValuationActionInterface.h>

class SoVertexShape;
class SoVolumeGeometryVbo;
class SoGLRenderAction;
class SoAction;
class SoState;
class SoBoundingBoxCache;
class SoPrimitiveVertex;
class SoLdmValuationAction;

/// @cond NOT_WRAPPING
/**
 * @VREXT LDM valuation interface for volume geometry nodes
 *
 * @DESCRIPTION
 *   Inherited by volume geometry nodes, e.g. SoVolumeFaceSet.
 */
class VOLUMEVIZ_API SoVolumeGeometry : public SoLdmValuationActionInterface
{
public:

  /** Constructor */
  SoVolumeGeometry();

  /** Destructor */
  virtual ~SoVolumeGeometry();

  /**
  * @copydoc SoVolumeShape::setRenderProgress
  */
  virtual void setRenderProgress(SoProgressIndicator* ps) { m_renderProgress = ps; }


SoINTERNAL public:

  /** 
   * Returns a distance between the given tile and the geometry.
   * if 0 then they intersect.
   * Nb: It is used internally to compute LDM tile priority based on distance to the geometry.
   */
  float intersectGeometry(const SoLDMTileID& tile) const;

  /** Called on ldmAction traversal for each triangle */
  virtual void triangleCB(SoAction* action,
    const SoPrimitiveVertex *v0,
    const SoPrimitiveVertex *v1,
    const SoPrimitiveVertex *v2 ) const ;

  /** Called on ldmAction traversal for each line */
  virtual void lineCB( SoAction* action,
    const SoPrimitiveVertex *v0,
    const SoPrimitiveVertex *v1 ) const ;

  /** Called on ldmAction traversal for each point */
  virtual void pointCB( SoAction* action, const SoPrimitiveVertex *v ) const ;

  /** For now, following method are used internally to allow
   *  fenceSlice implementation. May be exposed later.
   *
   * void setEnableBumMapping(bool flag);
   * bool isBumpMappingEnabled();
   * void setBumpScale(float scale);
   * float getBumScale();
   * void setAlphaUse (AlphaUse alphaUse);
   * AlphaUse getAlphaUse();
   * SoVolumeShape::Composition getComposition();
   * void setUseRGBA(bool flag);
   * bool isUsingRGBA();
   */

  /** Activate/Deactivate Bump Mapping on geometry */
  inline void setEnableBumMapping(bool flag);

  /** Retrieve bump state (activated/deactivated */
  inline bool isBumpMappingEnabled();

  /** Set the bump scale value */
  inline void setBumpScale(float scale);

  /** Retrieve alpha value */
  inline float getBumScale();

  /** Set AlphaUse value */
  inline void setAlphaUse (SoSlice::AlphaUse alphaUse);

  /** Retrieve AlphaUse value */
  inline SoSlice::AlphaUse getAlphaUse();

  /** Set UseRGBA value */
  inline void setUseRGBA(bool flag);

  /** Retrieve UsingRGBA value */
  inline bool isUsingRGBA();

#if SoDEPRECATED_BEGIN(9000)
  /** Return interpolation value according to the SoInteractiveComplexityElement.
   * WARNING: Let this method in pure virtual:
   * There an unresolvable design issue with volume geometry. Each implementation of volumeGeometry (like SoVolumeIndexedFaceSet) 
   * inherit from SoVolumeGeometry AND from an SoShape (SoIndexedFaceSet).
   * To avoid diamond inheritance issue, SoVolumeGeometry cannot inherit from SoNode, and thus cannot have any field. 
   * All SoVolumeGeometry implementation must redefine the interpolation field and getInteractiveInterpolationValue. */
  SoDEPRECATED_METHOD_NOWARN(9000,"SoVolumeGeometry::Interpolation not supported anymore.")
  virtual SoVolumeShape::Interpolation getInteractiveInterpolationValue(SoState* state) const = 0;
#endif /** @DEPRECATED_END */

  SoVolumeState* getVolumeState() const;

protected:

  /** Enable geometry clipping */
  inline void setClipGeometry(bool flag);

  /** Render the bounding box for the given shape */
  virtual void GLRenderBoundingBox(SoGLRenderAction* action) = 0;

  /** Render the inventor shape */
  virtual void shapeRender(SoGLRenderAction* action) = 0;

  /** Generate primitives of the inventor shape */
  virtual void shapeGeneratePrimitives(SoAction* action) = 0;

  /** Create needed render interfaces */
  void createRenderInterfaces(SoShape* bindedShape);

  /** Really compute the bounding box for the shape */
  virtual void getBBox(SoAction *action, SbBox3f &box, SbVec3f &center) = 0;

  /** Really compute the bounding box for the shape */
  virtual void getBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center) = 0;

  /** Do OpenGL rendering */
  void doRendering(SoGLRenderAction* action, float offset);

  /** Compute the bbox AABB */
  void computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center);

  /** Compute the bbox OOB */
  void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

  /** "Traverse" the vertexProperty node, putting its values into state.
      @return number of vertices */
  virtual int getNumVertices(SoState* state) = 0;

  /** @return index coord (nonIndexedShape return index) */
  virtual int getIndexCoord(int index) const = 0;

  /**
   * Invalidate both the standard bbox and bbox without projection caches
   */
  void invalidateBboxCache();

  /** Render interface for vbos */
  SoVolumeGeometryVbo* m_volVbo;

  /** If true, don't draw geometry out of roi*/
  bool m_clipGeometry;

  friend class SoLdmValuationAction;
  friend class SoVolumeGeometryVbo;
  friend class SoVolumeStateGeometry;
  friend class SoFenceSlice;

  /**
   * @copydoc SoSlice::AlphaUse
   * For SoVolumeGeometry default value is ALPHA_AS_IS
  */
  SoSlice::AlphaUse m_alphaUse;

 /**
  * Specifies if a bump mapping effect will be applied to the slice.
  * Default is FALSE.
  * Bump mapping is a rendering effect that simulates displacement of the slice 
  * surface using the gradient vectors (computed on the GPU) as normal vectors.
  * Normally lighting should be enabled to maximize the visual effect of bump mapping.
  *
  * This mode requires a graphics card supporting the OpenGL GLSL shading language.
  * If your graphics card does not have the necessary support, the slice will be
  * rendered without bump mapping.
  */
  bool m_enableBumpMapping;

  /**
  * Specifies the intensity of the bump mapping effect.
  * Default value is 1.0. Valid values range from 0 (no bump mapping)
  * to no upper limit. (2 is a fairly large value.)
  *
  */
  float m_bumpScale;

  
  /**
  * Specifies whether to create RGBA textures.
  * Default is FALSE.
  * If useRGBA is FALSE, indexed textures will be used if the graphics board supports
  * them.
  * If useRGBA is TRUE, RGBA textures will be used instead of indexed textures.
  * RGBA textures allow higher precision but are slower to load and use more
  * texture memory.
  * 
  * @IMPORTANT_NOTE If the dataSet is RGBA then this field is ignored and RGBA textures are used. @BR
  */
  bool m_useRGBA;

  SoProgressIndicator* m_renderProgress;

SoINTERNAL protected:
  /** Register ldm geometry to octree */
  virtual void ldmAction(SoLdmValuationAction* action);

private:

  /** render using the vbo interface */
  void renderVbo(SoGLRenderAction *action);

  /** 
   * Cache of bounding box 
   * - m_bboxCache[0] is the standard bbox
   * - m_bboxCache[1] is bbox without projection (used by ldmAction)
   */
  SoBoundingBoxCache* m_bboxCache[2];



};

/*******************************************************************************/
void
SoVolumeGeometry::setClipGeometry(bool flag)
{
  m_clipGeometry = flag;
}

bool 
SoVolumeGeometry::isBumpMappingEnabled()
{
  return m_enableBumpMapping;
}

float 
SoVolumeGeometry::getBumScale()
{
  return m_bumpScale;
}

SoSlice::AlphaUse 
SoVolumeGeometry::getAlphaUse()
{
  return m_alphaUse;
}

void 
SoVolumeGeometry::setEnableBumMapping(bool flag)
{
  m_enableBumpMapping = flag;
}


void 
SoVolumeGeometry::setBumpScale(float scale)
{
  m_bumpScale = scale;
}

void 
SoVolumeGeometry::setAlphaUse (SoSlice::AlphaUse alphaUse)
{
  m_alphaUse = alphaUse;
}

bool
SoVolumeGeometry::isUsingRGBA()
{
  return m_useRGBA;
}

void
SoVolumeGeometry::setUseRGBA(bool flag)
{
  m_useRGBA = flag;
}
/// @endcond

#ifdef _MSC_VER
#pragma warning(pop)
#endif

#endif


