/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T. DUFOUR (Nov 2002)
** Modified by : J. SALLANNE (Aug 2012)
**=======================================================================*/
#ifndef  _SO_VOLUME_GROUP_
#define  _SO_VOLUME_GROUP_

#include <VolumeViz/nodes/SoVolumeRender.h>
#include <Inventor/nodes/SoSeparator.h>

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable: 4251) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif

class SoGLObjectCache;
class SoLDMVirtualTexture;
class SoVolumeOffscreenImpl;

/** @brief Groups multiple volumes to be volume rendered in the same scene.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 *
 *   The SoVolumeGroup node allows multiple volumes to be volume rendered (SoVolumeRender)
 *   correctly in the same scene.  All SoVolumeRender nodes that are under an active
 *   SoVolumeGroup node (#multiVolumes=TRUE) are rendered together with correct blending,
 *   subject to some limitations.  Other VolumeViz geometry objects, for example
 *   SoOrthoSlice, are @I not @i affected by SoVolumeGroup.
 *
 *   The SoVolumeRender nodes can represent different volume data sets
 *   (see SoVolumeData) or different subvolumes (see SoROI) of the same volume
 *   data set.
 *
 *   This node forces the use of view-aligned slices for SoVolumeRender nodes
 *   that will be composed.
 *   It doesn't, however, actually modify the SoVolumeRender::samplingAlignment fields.
 *   Rather the values of these fields are ignored during traversal.
 *
 *   The #multiVolumes field (TRUE by default) controls whether the
 *   SoVolumeGroup's special handling of SoVolumeRender nodes is applied.
 *   If the (sub)volumes being rendered do not overlap in 3D space or only
 *   one of the volumes is transparent, then SoVolumeGroup special handling
 *   is automatically disabled.
 *
 *   SoVolumeGroup is primarily intended for cases where the volumes are
 *   sampled on different grids. For example if the volume dimension,
 *   size (3D extent) or orientation are different. For volumes that are
 *   actually multiple data sets sampled on the same grid, for example
 *   seismic attribute volumes, it may be more appropriate to blend the
 *   volumes using an SoDataCompositor or SoVolumeShader node.
 *
 *   Since OIV 9.3, SoVolumeGroup supports EdgeDetection(Gradient, Luminance, Depth) and LowResMode 
 *   (DECREASE_SCREEN_RESOLUTION). It can only apply these post process effects to either all or none of the volumes.
 *   If all fields corresponding to an effect are the same for all volumes, it activates the effect.
 *   If the fields are different, SoVolumeGroup uses the last value traversed.
 *
 *  @B Limitations: @b
 *   - Performance: @BR
 *     There is a significant performance decrease when #multiVolumes is enabled.
 *
 *   - Shaders: @BR
 *     Using custom shaders (e.g. SoVolumeShader or SoVolumeRenderingQuality) when 
 *     #multiVolumes is enabled gives incorrect lighting results.
 *
 *   - Raycasting: @BR
 *     The volume raycasting render algorithm (SoVolumeShader::raycasting) is not supported 
 *     when #multiVolumes is enabled. This implies that ray-casting effects like voxelizedRendering, 
 *     ambientOcclusion (SoVolumeRenderingQuality) and samplingAlignment = BOUNDARY_ALIGNED
 *     (SoVolumeRender) are not supported.
 *     When #multiVolumes is enabled, VolumeViz will automatically switch to "sliced" 
 *     (texture mapped polygon) rendering.
 *
 *   - Post process: @BR
 *     AmbientOcclusion and deferredLighting features are not supported when multiVolumes is enabled.
 *
 *   - Shadows: @BR
 *     32-bit framebuffer options (see e.g. SoWinGLWidget::setFloatingColorBuffer) 
 *     cannot be used if an SoVolumeGroup is inside an SoShadowGroup (request is ignored).
 *
 * @FILE_FORMAT_DEFAULT
 *    VolumeRender {
 *    @TABLE_FILE_FORMAT
 *       @TR multiVolumes      @TD TRUE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    @B SoGLRenderAction @b @BR
 *       Draws a volume-rendered image based on current SoVolumeData.
 *
 *    @B SoGetBoundingBoxAction @b @BR
 *       Computes the bounding box that encloses the volume.
 *
 * @SEE_ALSO
 *    SoVolumeRender,
 *    SoSeparator,
 *    SoDataCompositor,
 *    SoVolumeShader
 *
 *
 */
class VOLUMEVIZ_API SoVolumeGroup : public SoSeparator {

  SO_NODE_HEADER( SoVolumeGroup );

 public:
  /**
   * Activates the multi-volumes feature. Default is TRUE.
   * If set to FALSE, behaves as a regular SoGroup.
   */
  SoSFBool multiVolumes;

  /**
   * Constructor.
   */
  SoVolumeGroup();

  //------------------------------------------------------------------------------

 SoINTERNAL public:

 /**
  * Define actual state of the VolumeGroup.
  * The volumeGroup is done in several pass.
  *
  * 1- In the main pass, we are in REGISTERING_VR.
  * We register VolumeRenders that must be rendered in "VolumeGroup" mode. If a VR is opaque or binary
  * opaque, it can be rendered directly. Else it is registered and render later.
  * 
  * 2- A pass is generated for each registered VR. These passes render the PATH to each registered VR.
  * When we render this path, we are in COLLECTING_SLICE mode. We collect the slices of currently registered VR
  * (see m_currentVolume)
  *
  * 3- For the last COLLECTING_SLICE path, once we have collect all the registered VR slices, we sort them and render them
  * (see renderSlices). If, once ordered, a bunch of slices belongs to the same VolumeRender, we will try to render them at once.
  *
  */
  enum VGState
  {
    /**
     * 1. Check if path need to be registered
     */
    REGISTERING_VR,
    /**
     * 2. Call addSlice function for Traversal path registered during REGISTERING_VR state
     */
    COLLECTING_SLICE,
    /**
     * 3. Render
     */
    RENDERING,
  };

  /**
   * Returns TRUE if a node has an effect on the traversal path. The default
   * method returns FALSE. Node classes such as SoSwitch that allow
   * to traverse scene in different ways returns TRUE.
   */
  virtual bool affectsPath() const;

  static void initClass();
  static void exitClass();

  void addVolume(SoState* state, const SoVolumeRender* vr, SoLDMVirtualTexture* vt, SbVec3f planeNormal, int vpScale, float sliceDensity);
  void addSlice(const SoVolumeRender* vr, int numVertices, SbVec3f* vertices);
  VGState getVGState() const { return m_vgState; };

  /* return edgeDetectfragDataId created by registerRTT */
  int getEdgeDetectFragDataId() const;

  float getSliceDensity(const SoVolumeRender* vr) const;

  SoVolumeRenderingQuality* getVRQ(const SoVolumeRender* vr) const;

  typedef std::pair< unsigned int, SoRef<SoShape> > IdShapePair;

  const std::vector<IdShapePair>& getInternalShapeListForThisVr(const SoVolumeRender* vr) const;

SoEXTENDER public:
  /** @copydoc SoNode::GLRenderBelowPath */
  virtual void GLRenderBelowPath(SoGLRenderAction* action);
  /** @copydoc SoNode::GLRenderInPath */
  virtual void GLRenderInPath(SoGLRenderAction* action);

 protected:
  // Destructor
  virtual ~SoVolumeGroup();

 private:

  struct Slice {
    // slice distance from eye (affine component of slice plane), in view space.
    // Note: in view space Z axis goes FROM screen TO viewer. A slice is IN FRONT OF another if its distance is greater.
    float distance;
    std::vector<SbVec3f> vertices;
    const SoVolumeRender* vr;
    Slice(float distance, int numVertices, SbVec3f* vertices, const SoVolumeRender* vr);
  };


  struct VRInfo {
    const SoVolumeRender* vr;
    SoPath* path;
    SoVolumeRenderingQuality* vrq;
    SoLDMVirtualTexture* texID;
    int vpScale;
    float sliceDensity;
    SbVec3f planeNormal;
    SbMatrix viewingMatrix;

    bool operator==(const SoVolumeRender* _vr) const
    {
      return this->vr == _vr;
    }
  };

  typedef std::vector<VRInfo> VRInfoList;
  typedef std::map< const SoVolumeRender*, std::vector< IdShapePair > > VRContinuousShapeMap;

  std::map < const SoVolumeRender*, SoRef<SoVolumeRenderingQuality> > m_vrVrqMap;

  VRContinuousShapeMap m_vrContinuousShapeMap;

  SoVolumeOffscreenImpl* m_volumeOffscreenImpl;

  /** Use to apply post-process effects (edgeDetect, AO, etc...). */
  SoVRImageSpaceEffects* m_imageSpaceEffects;

  /**
  * Create SoShape containing slices geometry
  */
  SoRef<SoShape> buildShapeSlices(const SoVolumeRender* vr, unsigned int orderedShapeId, unsigned int id, const std::vector<SbVec3f>& vertices);

  VGState m_vgState;
  VRInfoList m_vrInfoList;

  /** Slices that have been collected during the COLLECTING_SLICE pass.
  * At the end of COLLECTING_SLICE pass, it contains all the slices of all the registered volumeRender.
  * Empty if no volumeRender have been registered. */
  std::vector<Slice*> m_slices;

  void sortSlices();
  void renderSlices(SoState *state);
  static bool compareSlice(const Slice*s1, const Slice*s2);

  /**
  * Called just before rendering
  * @param numberOfVolume is required for postRender function.
  * number of volumeRender that have been registered in the REGISTERING_VR pass
  * @return false if the node behaves as an SoSeparator. It will not be necessary do call postRender
  */
  bool preRender(SoGLRenderAction *action, int &numberOfVolume);
  /**
  * Called just after rendering
  */
  void postRender(SoGLRenderAction *action, int numberOfVolume);

  /**
  * Begin offscreen rendering:
  * Initialize RTT and fbo if needed.
  */
  void beginOffscreenRendering( SoGLRenderAction* action );

  /**
  * End offscreen rendering:
  * release RTT and fbo if needed.
  */
  void endOffscreenRendering( SoGLRenderAction* action );

  /** Return true if we need RTT EDGE_DETECT to apply the current EdgeDetect effect.
  * If true, internalFormat is the needed texture format. */
  bool needRttEdgeDetect(SoState* state) const;

  /** Return edgeDetect2DMethod
  * mask value: LUMINANCE, DEPTH, GRADIENT (see VolumeRenderingQuality::EdgeDetect2DMethod) */
  int getEdgeDetect2DMethod(SoState* state) const;

  /** Return lowResolutionScale */
  int getLowResolutionScale() const;

  /** set all SoVolumeRenderingQuality to the same values for edgeDetection */
  void updateVRQ(int bitmask, std::vector<int>& bitMaskVector);

  /** restore all SoVolumeRenderingQuality to original values saved during updateVRQ */
  void restoreVRQ(const std::vector<int>& bitMaskVector);
};

#if defined(_WIN32)
#pragma warning(pop)
#endif

#endif // _SO_VOLUME_GROUP_


