/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#pragma once

#include <Inventor/caches/SoCache.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/SbBox.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/map>

#include <VolumeViz/nodes/SoVolumeShape.h>

#include <VolumeViz/LDM/SoLDMTileID.h>

class SbProjection;
class SoFrameBufferObject;
class SoGLProgramGLSL;
class SoGLTexture;
class SoVRImageSpaceEffects;
class SoLDMTileID;
class SoLdmSubDivisionIsosurfacePolicy;
class SoLdmSubDivisionPolicy;
class SoLdmSubDivisionTransferFunctionPolicy;
class SoShaderProgram;
class SoVolumeGroup;
class SoVolumeIsosurface;
class SoVolumeRenderInterface;
class SoVolumeRenderRaycast;
class SoVolumeRenderLdm;
class SoVolumeRenderLdm;
class SoVolumeRenderPaging;
class SoVolumeRenderingQuality;
class SoVolumeStateVr;
class SoVolumeRenderDetail;
class SoNodeSensor;
class SoVolumeOffscreenImpl;

/**
@VREXT Renders data volumes using direct volume rendering

@ingroup VolumeVizNodes

@DESCRIPTION
  This node renders volume data using "direct volume rendering". The volume
  process involves sampling, interpolation, classification and composition.
  The rendering algorithm is a GPU-based technique called raycasting.
  Raycasting has similarities to the well known algorithm called raytracing, but is specialized
  for volume data (renders voxels not triangles) and does not currently implement
  reflection or refraction. One or more "rays" are cast through the volume from
  each relevant pixel on the screen. Samples are taken at intervals along each
  ray. The sample interval is controlled by the #numSlices and #numSlicesControl
  fields. At each sample point a value is interpolated from the closest voxels.
  The interpolation technique is controlled by the #interpolation field. Classification means
  that color and opacity are computed based on the current SoDataRange and 
  SoTransferFunction (and possibly values from other volumes - see SoVolumeShader).
  Optional rendering effects may modify the base color and/or opacity. These effects are controlled by
  an SoVolumeRenderingQuality node and include lighting, shadows, edge coloring, 
  boundary opacity and more. The sample is then composited with other samples along
  the ray. Composition is controlled by the #renderMode field. By default colors are 
  combined based on the opacity of the sample (alpha blending).  The ray is terminated
  if it reaches full opacity or reaches the current depth in the depth buffer.

  The voxel's color to be combined during the composition step are retrieved in 2 different
  ways according the type of volume data:
  - For a volume containing scalar data values: each voxel's color is determined by applying
    the current SoDataRange and SoTransferFunction to the voxel's value.
  - For an RGBA volume, each voxel's color comes directly from the volume data and the
    SoDataRange and SoTransferFunction are ignored.

  The voxel's color is also combined with the current base color and transparency
  (see definition of base color @ref SoLightModel-base-color-def "here")
  set for example with an SoMaterial or an SoPhysicalMaterial node. This means that, for
  example, the current transparency can be used as a *global alpha* scale factor
  to decrease the opacity of all voxels.

  If the light SoLightModel#model is either PER_VERTEX_PHONG or PER_PIXEL_PHONG, the final
  voxel's color is also affected by the fields emissiveColor, specularColor and shininess of
  the current SoMaterial. If the light SoLightModel#model is PHYSICALLY_BASED the final
  voxel's color is also affected by the fields specular roughness metallic of the
  current SoPhysicalMaterial.

  The #samplingAlignment field controls whether
  the samples are axis aligned (perpendicular to one axis of the volume), 
  view aligned (perpendicular to the view direction) or boundary aligned (each
  ray starts at the first intersected voxel with alpha value > 0).
  Generally boundary aligned slices should be used (better image quality).
  Using SoVolumeGroup, SoVolumeIsoSurface or SoProjection nodes will
  automatically switch to view-aligned samples.

  The property nodes SoVolumeIsoSurface and SoVolumeDataDrawStyle add additional
  rendering styles and can be used, for example, to force SoVolumeRender to draw
  a GPU computed isosurface instead of volume rendering.

  Example rendering:

  \htmlonly
  <TABLE border=1 cellspacing=0 cellpadding=5>
  <TR><TD valign=_top> lighting = FALSE </TD> <TD> lighting = TRUE </TD> </TR>
  <TR><TD valign=_top> <img src="lightoff.jpg" alt=""/> </TD> <TD> <img src="lighton.jpg" alt=""/> </TD> </TR>
  <TR><TD valign=_top> composition = ALPHA_BLENDING </TD> <TD> composition = MAX_INTENSITY </TD> </TR>
  <TR><TD valign=_top> <img src="compalpha.jpg" alt=""/> </TD> <TD> <img src="compmax.jpg" alt=""/> </TD> </TR>
  </TABLE>
  \endhtmlonly


# Multiple volumes:

  VolumeViz provides several mechanisms for combining or rendering multiple volumes,
  depending on the application's specific requirements. There are several cases:

  - CPU data combining  
    The data values from multiple volumes can be combined on the CPU during data
    loading, using the SoDataCompositor node.  For example computing the "difference"
    between two volumes on the fly.  SoLDMDataTransform and SoVolumeTransform can be
    used to modify data for a single volume, for example scaling data values or
    computing a derived data set.  SoLDMDataTransform is applied when data is loaded
    and SoVolumeTransform is applied before data is transferred to the GPU.

  - Multiple data sets  
    These are volumes that are really multiple data sets on the same "grid", in other
    words volumes having exactly the same dimensions and extent. For example, seismic attribute
    volumes. These volumes can be combined on the GPU using a simple fragment shader
    function.  For example, replacing the VVizComputeFragmentColor (GLSL) function
    allows "co-blending" the colors from multiple volumes.  See SoVolumeShader for more
    details.  Use an SoMultiDataSeparator node to group the volume data nodes that
    should be combined.

  - @anchor MULTIPLE_INDEPENDENT_VOLUMES Multiple independent volumes  
    These are volumes in the same scene that are completely independent (but
    might overlap in space) or have different dimensions or different extents. For
    example, medical datasets from different modalities.

    In order to ensure that the rendering of overlapping transparent volumes is properly interleaved, there are 2 options:
    - if each volume needs their own rendering parameters and shading options, then use an
      SoVolumeGroup node to group the volume data nodes and rendering nodes, with an SoVolumeRender node for each volume.
    - if the same rendering parameters and shading options can be applied to all volumes, then use
      an SoMultiDataSeparator node to group the volume data nodes and a single SoVolumeRender node.
      This option should be preferred as it does not suffer the limitations of SoVolumeGroup.
      However, note that SoVolumeRenderingQuality#voxelizedRendering is not supported by this feature.

# Custom shaders:

  The SoVolumeShader node allows a variety of custom shader functions to be defined for
  special computation or rendering effects on single volumes or multiple volumes.  All of
  these features require programmable shader support on the GPU.  Be sure to use an
  SoMultiDataSeparator (instead of SoSeparator) when combining multiple volumes.

# Composition of multiple independent volumes:

  It is possible to compose datasets that have different dimensions, tile
  sizes and transformations. In those cases, the volume rendering will be applied
  on the union of the extents of all the SoVolumeData nodes on state.

  In addition, texture coordinates conversion functions are provided in the
  *VolumeViz/vvizStructure.h* shader include in order to help fetch
  the correct data values in custom shaders.

  For instance,
  \code
  vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
  \endcode
  can be used to convert texture coordinates related to one dataset to
  texture coordinates related to another dataset.

  The conversion is based solely on the transformations applied to each
  dataset, which are defined by their model matrix and their extent.
  Please note that the model matrix of a dataset is defined by to the
  SoTransformation nodes that are placed **before** the SoDataSet node in
  the order of the traversal.

  See SoVolumeShader#FRAGMENT_COMPUTE_COLOR for an example of a shader implementing the
  VVizComputeFragmentColor() function in that case.

# Lighting

  The SoVolumeRenderingQuality property node allows you to to enable GPU
  computed lighting based on the first SoLight node in the scene graph.
  (Note that this is typically the viewer's "headlight".) VolumeViz supports
  two lighting algorithms. They are both computed on the GPU and are independent
  (but normally only one should be enabled).
  - Gradient lighting  
    Computes the effect of lighting for every sample along the ray, similar to the
    computation for polygonal geometry, but using a gradient vector computed from the
    voxel data values instead of a normal vector. Gradient lighting is enabled by the
    field SoVolumeRenderingQuality#lighting. Gradient lighting is the classic
    solution but can have problems in relatively homogeneous regions where the gradient
    magnitude is small and gradient direction somewhat random. Multiple fields affect this computation
    including interpolation, gradientThreshold, gradientQuality and surfaceScalarExponent.
  - Deferred lighting   
    Computes the effect of lighting for every visible voxel, as a post-processing step,
    using a gradient vector computed from the (visible) voxel depth values. Deferred
    lighting is enabled by the field SoVolumeRenderingQuality#deferredLighting.
    Deferred lighting generally provides better performance because fewer lighting
    computations are needed. Deferred lighting generally provides better image quality
    for volumes that inherently contain surfaces (sharp gradients) like medical and
    industrial scans.  Deferred lighting is most effective when the opacity values in
    the transfer function are approximately binary (mostly 0 and 1 values). See also
    the #opacityThreshold field.

  \htmlonly
  <TABLE border=1 cellspacing=0 cellpadding=5>
  <TR><TD valign=_top> Gradient lighting </TD> <TD> Deferred lighting </TD> </TR>
  <TR><TD valign=_top> <img src="Lighting_gradient.png" alt=""/> </TD> <TD> <img src="Lighting_deferred.png" alt=""/> </TD> </TR>
  </TABLE>
  \endhtmlonly

# Shadows:

  Open Inventor shadow rendering works for volume rendering similar to any other
  geometry. When shadow rendering is enabled (see SoShadowGroup), non-transparent
  voxels can cast and receive shadows (see SoShadowStyle).  Shadow rendering is
  independent of whether lighting is enabled for the volume.

  SoVolumeRender also supports "ambient occlusion" rendering (see the
  field SoVolumeRenderingQuality#ambientOcclusion). This rendering mode is visually a kind of
  self-shadowing and represents an approximation of the effect of ambient global lighting
  in the scene. Ambient occlusion can be combined with gradient or deferred lighting
  and with shadow casting.

  \htmlonly
  <TABLE border=1 cellspacing=0 cellpadding=5>
  <TR><TD valign=_top> Shadow casting </TD> <TD> Ambient occlusion </TD> </TR>
  <TR><TD valign=_top> <img src="Lighting_shadows.png" alt=""/> </TD> <TD> <img src="Lighting_ambientOcclusion.png" alt=""/> </TD> </TR>
  </TABLE>
  \endhtmlonly

# Clipping:

  VolumeViz provides multiple tools for clipping volume rendering.
  Any or all of these tools may be combined for more complex clipping situations.
  Note that these tools clip all volume shapes including slices.

  - The SoROI (Region of Interest) node limits volume rendering to a subvolume.
    The SoROI node's EXCLUSION_BOX mode can also be used to exclude
    a sub-region, forming what is sometimes called a "chair cut".  Note that the
    Region of Interest also limits *data loading *, so it is valuable when
    the total size of the volume exceeds the available system memory.

  - The SoVolumeClippingGroup node clips volume rendering to any closed shape
    defined by a group of standard Open Inventor geometry nodes.  Both "inside"
    and "outside" clipping are supported. The SoScreenDrawer, SbExtrusionGenerator
    and SoCSGShape nodes are useful for creating clipping geometry for interactive
    volume "sculpting".

  - The SoUniformGridClipping and SoUniformGridProjectionClipping nodes clip
    volume rendering against one or more surfaces defined by a height field.
    This is particularly useful in seismic applications for clipping against
    (or between) horizon surfaces.

  - The SoVolumeMask node can be used to clip volume rendering against a boolean
    mask volume on a per-voxel basis. But the mask
    mechanism is much more powerful than that. Each region can have its own transfer
    function (color map) using SoTransferFunction nodes. Each region can also have its
    own draw style (volume rendering, isosurface or boundary) using SoVolumeDataDrawStyle
    nodes. Each region, including the original unmasked volume, is only visible if there
    exists a transfer function (SoTransferFunction) with the same id value.

# Picking:

 SoRayPickAction handles picking of VolumeViz shapes similar to other geometry in the
 scene, but with additional features.  Picking on an SoVolumeRender node can return
 the first non-transparent voxel "hit" or the entire set of intersected voxels along
 the pick ray, depending on the pickAll flag of the SoRayPickAction.  Similar to other
 geometry, SoPickedPoint can return a "detail" class specific to SoVolumeRender.
 SoVolumeRenderDetail returns the IJK (voxel coordinate) position of the pick and the
 data value at that point.

 The SoVolumeRender node (by default) uses the GPU to
 compute the picked voxel during an SoRayPickAction. For this to work, the SoRayPickAction
 must have its scene manager initialised using the method SoAction::setSceneManager().
 SoHandleEventAction does this automatically, so it is not necessary for the application
 to take any action when using (for example) an SoEventCallback node and calling the
 getPickedPoint() method.  However if the application creates its own SoRayPickAction
 then it should set the scene manager.  If no scene manager is specified, a warning
 message is issued and software picking is done.
 If necessary, using the GPU for volume picking may be disabled by setting the
 environment variable IVVR_GPU_PICKING to 0 (see SoPreferences).

# Projection:

  The SoVolumeRender node supports projected volume rendering, for example
  rendering a volume defined on a grid of latitude / longitude coordinates.
  Projection is enabled by adding an SoProjection node before the SoVolumeRender
  node (see SoProjection for more information about supported coordinate systems,
  ellipsoids and map projections). The projection quality versus speed ratio can
  be controlled using the new #projectedTileSubdivision field that defines how
  often each tile's geometry will be subdivided when projected.  This is important
  because only the corner points of the tiles are projected, not the individual
  voxels.  So subdividing the tiles provides a better approximation of the
  actual shape of the grid. Volume projection works with both regular (uniform
  voxel spacing) and rectilinear (non-uniform voxel spacing) grids. SoProjection
  automatically selects view-aligned sampling.

  Warning:
  - Volume projection is incompatible with some options enabled by
           the VolumeRenderingQuality node.  
           Do not enable the preIntegrated, jittering or edgeDetect2D fields.
  - Volume projection requires all culling to be disabled.
           The following options in class SoLDMGlobalResourceParameters
           should be disabled: setScreenResolutionCulling (default is false),
           setViewpointRefinement (default is TRUE) and setViewCulling (default is TRUE).

# Performance:

Volume rendering performance is affected by many factors including the size of the
volume and the rendering options selected. Image quality is also affected by many
rendering options and, in general, higher quality implies lower performance. Some 
of the factors affecting volume rendering performance are:
  - Number of voxels:   
    This mainly depends on the size of the volume, but can be 
    reduced using an SoROI (region of interest) node.

  - Number of pixels:   
    A larger number of pixels means a larger number of rays must be cast through
    the volume and therefore the shader execution time on the GPU will be longer.
    This effect is most noticeable when high quality rendering options are enabled.
    The number of pixels rendered can be temporarily reduced by setting the 
    #lowResMode field to DECREASE_SCREEN_RESOLUTION. This reduces the number of
    times the shader programs running on the GPU must be executed.

  - Number of samples (slices):   
    This is controlled by the #numSlices and #numSlicesControl fields.
    Note that better image quality can obtained with the same number of samples by
    enabling options like preintegrated rendering (see SoVolumeRenderingQuality) and/or
    the BOUNDARY_ALIGNED setting for #samplingAlignment. The number of samples can be 
    automatically decreased when interacting using an SoInteractiveComplexity node.

    We recommend to set the #numSlicesControl field to AUTOMATIC
    and the #numSlices field to -1. The number of samples will be computed based on the 
    dimensions of the volume (number of voxels on each axis), the #SoComplexity::value
    setting and the viewing direction. If the viewing direction changes, the number of 
    samples will be automatically adjusted.

  - Opacity:   
    Increasing the number of opaque, or nearly opaque, voxels in the volume (using
    SoTransferFunction) will generally improve performance because the sampling rays
    can terminate sooner. See also IVVR_ALPHA_THRESHOLD_INTERACTIVE in SoPreferences.

    If you are using a completely opaque transfer function, for example with a "volume
    probe", SoVolumeSkin will generate the same image much faster.

  - Rendering options:   
    Many of the advanced rendering options and rendering effects
    enabled by SoVolumeRenderingQuality have an additional performance cost. These
    include lighting, edge coloring, boundary opacity, cubic interpolation and
    gradient quality. These settings can be automatically changed while interacting
    using an SoInteractiveComplexity node.

  - Tile size:   
    For backward compatibility, the default tile size is still only 64. This is quite
    small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
    number of tiles that must be managed by VolumeViz. This overhead can be significant,
    especially for operations that require reloading the data textures on the GPU, for
    example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
    it can be efficient to set the tile size large enough to contain the entire volume.
    For very large volumes, larger tile sizes are efficient for SoVolumeRender but
    somewhat inefficient for slice rendering because complete tiles must be loaded
    even though the slice only uses part of the data (see also SoSlice::largeSliceSupport).
    Applications should experiment.

    For volumes stored in LDM file format, the tile size must be specified when the
    volume is converted to LDM (see SoConverter and the "-t" option).  For other data
    data formats the tile size can be specified using the field SoVolumeData#ldmResourceParameters,
    but only after setting the field SoDataSet#filename
    or calling the \if_dotnet SetReader() \else setReader() \endif method.

  - Tile cache policy:   
    It specifies how the tiles are stored in CPU memory. The selected policy
    can significantly impact the data loading performance versus the CPU memory footprint.
    See SoLDMResourceParameters::tileCachePolicy for detail.

  - If rendering performance is too slow, it may be necessary to render with high quality
    settings when the user is not interacting with the scene, but temporarily switch to
    high performance (lower quality) settings when the user is interacting.  Open Inventor
    automatically sets "interactive mode" when the user is moving the camera or moving a
    dragger. The application can explicitly set interactive mode, for example while the
    user is moving a slider in the user interface. Some important tools are:
      - LowResMode   
        For example, set the #lowResMode field to #DECREASE_SCREEN_RESOLUTION and set
        the #lowScreenResolutionScale field to 2 or 4.  In interactive mode VolumeViz will
        render the volume at lower resolution (reducing the number of sample rays and
        shader executions).
      - SoInteractiveComplexity   
        This node allows you to specify different values to use for certain fields in
        interactive mode.  For example set a smaller value for #SoComplexity::value (reduces
        the number of samples) or turn off an expensive rendering option. The values
        specified in SoInteractiveComplexity override the actual fields in the scene graph.

# Limitations:
  - Multi-thread rendering:   
    Unlike most Open Inventor nodes, VolumeViz nodes do not support simultaneous
    rendering in multiple threads (even when Open Inventor is initialized using 
    one of the initThread() methods).

  - Geometric transforms:   
    The volume size (extent in 3D space) and orientation can be modified by
    transformation nodes in the scene graph just like any geometry.  For a volume
    this in turn modifies the appearance of volume rendering nodes like SoVolumeRender.
    However please **note:** The same transformation must be applied to the volume data node
    and all volume rendering nodes associated with that volume.  So effectively
    any transformation nodes that affect the volume must be placed **before**
    the volume data node.

  - Multiple dataset:   
    The field #dataSetIds has limitations described @ref LIMITATIONS "here"

  - Render modes:   
    When using one of the projection render mode (i.e. MIN_INTENSITY_PROJECTION, MAX_INTENSITY_PROJECTION,
    SUM_INTENSITY_PROJECTION or AVERAGE_INTENSITY_PROJECTION):
      - Only raycasting mode is supported
      - No depth information is retained.
      - SoVolumeRenderingQuality#preIntegrated is not taken into account.

# Example:

  For simple data sets, a basic VolumeViz rendering could be achieved
  with only a few nodes: minimally an SoVolumeData node to identify
  the data set and one rendering node.  However most data sets need
  at least some of the additional nodes shown here in order to get
  a correct and useful rendering. Most applications will need
  additional nodes to take advantage of region of interest, interaction,
  clipping and other VolumeViz features.  Please consider
  the code shown here as simply a guideline and a starting point for
  exploring the many powerful features available in Open Inventor.
  
  Note that some of the property nodes (data, material, color map,
  etc) will typically be shared by multiple rendering nodes. For example
  the volume data usually only needs to be loaded once, using a single
  SoVolumeData node.  Multiple slices and/or regions can be rendered
  using that data node and they may use the same transfer function or
  each have their own.

  Also note that this example is for a data volume, not a label volume.
  Please see the notes about label volumes following the code block.
\if_cpp
\code
  // Keep volume viz separate from geometry
  SoSeparator* volSep = new SoSeparator();
    root->addChild( volSep );

  // Decrease the quality while moving to have better interactivity
  SoInteractiveComplexity* interact = new SoInteractiveComplexity();
    // Decrease the "number of samples"
    interact->fieldSettings.set1Value( 0, "SoComplexity value 0.2 0.5" );
    // Decrease interpolation quality. 
    interact->fieldSettings.set1Value( 1, "SoVolumeRender interpolation LINEAR CUBIC" );
    // Don't wait before returning to full quality rendering.
    interact->refinementDelay = 0;
    volSep->addChild( interact );

  // Complexity node for the interact node to control.
  SoComplexity* volComp = new SoComplexity();
    volSep->addChild( volComp );

  // Load volume data
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHead.vol";
    volSep->addChild( volData );

  // Set range of data values to visualize.
  // Not required for 8-bit voxels, critical for larger data types.
  // The getMinMax() call may be expensive for non-LDM file formats.
  SoDataRange* volRange = new SoDataRange();
    if (volData->getDatumSize() > 1) {
      double minVal, maxVal;
      volData->getMinMax( minVal, maxVal );
      volRange->min = minVal;
      volRange->max = maxVal;
    }
    volSep->addChild( volRange );

  // Load constant intensity with alpha ramp
  SoTransferFunction* volTF = new SoTransferFunction();
    volTF->predefColorMap = SoTransferFunction::GRAY;
    volTF->minValue = 10; // Make "noise" voxels transparent
    volSep->addChild( volTF );

  // Display volume at full intensity
  SoMaterial* volMat = new SoMaterial();
    volMat->diffuseColor.setValue( 1, 1, 1 );
    volSep->addChild( volMat );

  // Volume rendering settings
  SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
    // Remove tile boundary artifacts while moving. 
    volQual->interpolateOnMove = TRUE;
    // Higher quality rendering
    volQual->preIntegrated = TRUE;
    // Optional: Enable screen space lighting
    volQual->deferredLighting = TRUE;
    // Optional: If using gradient lighting, increase quality
    volQual->surfaceScalarExponent = 5;
    volSep->addChild( volQual );

  // Display volume rendering
  SoVolumeRender* volRend = new SoVolumeRender();
    // Let Inventor compute best number of slices
    volRend->numSlicesControl = SoVolumeRender::AUTOMATIC;
    // Optional: Use lower screen resolution while moving.
    volRend->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
    volRend->lowScreenResolutionScale = 2;
    // Remove "slicing" artifacts
    volRend->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volSep->addChild( volRend );
\endcode
\endif
\if_dotnet
\code
  // Keep volume viz separate from geometry
  SoSeparator volSep = new SoSeparator();
    root.AddChild( volSep );

  // Decrease the quality while moving to have better interactivity
  SoInteractiveComplexity interact = new SoInteractiveComplexity();
    // Decrease "number of samples"
    interact.fieldSettings[0] = "SoComplexity value 0.2 0.5";
    // Decrease interpolation quality. 
    interact.fieldSettings[1] = "SoVolumeRender interpolation LINEAR CUBIC";
    // Don't wait before returning to full quality rendering.
    interact.refinementDelay.Value = 0;
    volSep.AddChild( interact );

  // Complexity node for the interact node to control.
  SoComplexity volComp = new SoComplexity();
    volSep.AddChild( volComp );

  // Load volume data
  SoVolumeData volData = new SoVolumeData();
    volData.fileName.Value = "$OIVNETHOME/src/demos/data/VolumeViz/3DHead.ldm";
    volSep.AddChild( volData );

  // Set range of data values to visualize.
  // Not required for 8-bit voxels, critical for larger data types.
  // The getMinMax() call may be expensive for non-LDM file formats.
  SoDataRange volRange = new SoDataRange();
  if (volData.GetDatumSize() > 1)
  {
      double minVal, maxVal;
      volData.GetMinMax( out minVal, out maxVal);
      volRange.min.Value = minVal;
      volRange.max.Value = maxVal;
  }
  volSep.AddChild( volRange );

  // Load constant intensity with alpha ramp
  SoTransferFunction volTF = new SoTransferFunction();
    volTF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.GRAY;
    volTF.minValue.Value = 10; // Make "noise" voxels transparent
    volSep.AddChild( volTF );

  // Display volume at full intensity
  SoMaterial volMat = new SoMaterial();
    volMat.diffuseColor.SetValue(1, 1, 1);
    volSep.AddChild( volMat );

  // Volume rendering settings
  SoVolumeRenderingQuality volQual = new SoVolumeRenderingQuality();
    // Remove border artifacts while moving. 
    volQual.interpolateOnMove.Value = true;
    // Higher quality rendering
    volQual.preIntegrated.Value = true;
    // Optional: Enable screen space lighting
    volQual.deferredLighting.Value = true;
    // Optional: If using gradient lighting, increase quality
    volQual.surfaceScalarExponent.Value = 5;
    volSep.AddChild( volQual );

  // Display volume rendering
  SoVolumeRender volRend = new SoVolumeRender();
    // Let Inventor compute best number of slices
    volRend.numSlicesControl.Value = SoVolumeRender.NumSlicesControls.AUTOMATIC;
    // Optional: Use lower screen resolution while moving.
    volRend.lowResMode.Value = SoVolumeRender.LowResModes.DECREASE_SCREEN_RESOLUTION;
    volRend.lowScreenResolutionScale.Value = 2;
    // Remove "slicing" artifacts
    volRend.samplingAlignment.Value = SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED;
    volSep.AddChild( volRend );
\endcode
\endif
\if_java
\code
  // Keep volume viz separate from geometry
  SoSeparator volSep = new SoSeparator();
    root.addChild(volSep);

  // Decrease the quality while moving to have better interactivity
  SoInteractiveComplexity interact = new SoInteractiveComplexity();
    // Decrease the "number of samples"
    interact.fieldSettings.set1Value( 0, "SoComplexity value 0.2 0.5" );
    // Decrease interpolation quality. 
    interact.fieldSettings.set1Value( 1, "SoVolumeRender interpolation LINEAR CUBIC" );
    // Don't wait before returning to full quality rendering.
    interact.refinementDelay.setValue( 0 );
    volSep.addChild( interact );

  // Complexity node for the interact node to control.
  SoComplexity volComp = new SoComplexity();
    volSep.addChild( volComp );

  // Load volume data
  SoVolumeData volData = new SoVolumeData();
    volData.fileName.setValue( "$OIVJHOME/data/VolumeViz/3DHead.vol" );
    volSep.addChild( volData );

  // Set range of data values to visualize.
  // Not required for 8-bit voxels, critical for larger data types.
  // The getMinMax() call may be expensive for non-LDM file formats.
  SoDataRange volRange = new SoDataRange();
    if (volData.getDatumSize() > 1) {
      double[] minmax;
      minmax = volData.getDoubleMinMax();
      volRange.min.setValue( minmax[0] );
      volRange.max.setValue( minmax[1] );
    }
    volSep.addChild( volRange );

  // Load constant intensity with alpha ramp
  SoTransferFunction volTF = new SoTransferFunction();
    volTF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.GRAY );
    volTF.minValue.setValue( 10 ); // Make "noise" voxels transparent
    volSep.addChild( volTF );

  // Display volume at full intensity
  SoMaterial volMat = new SoMaterial();
    volMat.diffuseColor.setValue( 1, 1, 1 );
    volSep.addChild( volMat );

  // Volume rendering settings
  SoVolumeRenderingQuality volQual = new SoVolumeRenderingQuality();
    // Remove border artifacts while moving. 
    volQual.interpolateOnMove.setValue( true );
    // Higher quality rendering
    volQual.preIntegrated.setValue( true );
    // Optional: Enable screen space lighting
    volQual.deferredLighting.setValue( true );
    // Optional: If using gradient lighting, increase quality
    volQual.surfaceScalarExponent.setValue( 5 );
    volSep.addChild( volQual );

  // Display volume rendering
  SoVolumeRender volRend = new SoVolumeRender();
    // Let Inventor compute best number of slices
    volRend.numSlicesControl.setValue( SoVolumeRender.NumSlicesControls.AUTOMATIC );
    // Optional: Use lower screen resolution while moving.
    volRend.lowResMode.setValue( SoVolumeRender.LowResModes.DECREASE_SCREEN_RESOLUTION );
    volRend.lowScreenResolutionScale.setValue( 2 );
    // Remove "slicing" artifacts
    volRend.samplingAlignment.setValue( SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED );
    volSep.addChild( volRend );
\endcode
\endif

# Label volumes
  
  A label volume, also known as a label field, is usually the result of doing
  some sort of segmentation on a data volume.  Each voxel value is an integer
  label (id) identifying which material, object, etc that the voxel belongs to.
  There could be 100's or 1000's of labels, but there might be as few as 8 label
  values. For example, a simple label volume might have 7 opaque materials plus
  plus an "exterior" material which is completely transparent.
  Conceptually, there is one big difference between a (typical) data volume and
  a label volume.  A data volume is conceptually a set of discrete samples taken
  from a continuous scalar field.  So we know the exact value at the center of
  each voxel and interpolate between those values to get the value at any position
  in between voxels.  In a label volume we normally consider each voxel to belong
  completely to one material, so the value is constant until we cross the boundary
  into the next voxel. Therefore we do not want to interpolate the label values.
  
  When rendering a label volume, make the following changes to the above example:
  - Set the field SoVolumeRender#interpolation to NEAREST and
  - Leave the field SoVolumeRenderingQuality#preintegrated field set to FALSE.
  
  If rendering isosurfaces (SoVolumeIsosurface), set the field
  SoVolumeRenderingQuality#segmentedInterpolation to TRUE.
  
  It is also important to set the data range, texture precision and color map
  size carefully. Please see the label volume discussion in SoTransferFunction.


@FILE_FORMAT_DEFAULT
   VolumeRender {
   @TABLE_FILE_FORMAT
      @TR interpolation      @TD LINEAR
      @TR lighting           @TD FALSE
      @TR lightDirection     @TD -1, -1, -1
      @TR lightIntensity     @TD 1
      @TR numSlices          @TD 0
      @TR numSlicesControl   @TD AUTOMATIC
      @TR samplingAlignment  @TD BOUNDARY_ALIGNED
      @TR lowResMode         @TD DECREASE_NONE
      @TR lowScreenResolutionScale @TD 1
      @TR subdivideTile      @TD FALSE
      @TR projectedTileSubdivision @TD 1
      @TR fixedNumSlicesInRoi @TD FALSE
      @TR opacityCorrection  @TD TRUE
      @TR renderMode         @TD VOLUME_RENDERING
      @TR dataSetIds         @TD []
   @TABLE_END
   }


@ACTION_BEHAVIOR
   SoGLRenderAction  
      Draws a volume-rendered image based on current SoVolumeData.

   SoGetBoundingBoxAction  
      Computes the bounding box that encloses the volume.

   SoRayPickAction  
     Picking always returns the first
     non-transparent voxel intersected by the pick ray. The old behavior
     can be restored by using an SoPickStyle node set to BOUNDING_BOX.

@SEE_ALSO
   SoVolumeData,
   SoTransferFunction,
   SoROI,
   SoVolumeShader,
   SoVolumeIsosurface,
   SoVolumeRenderingQuality,
   SoProjection,
   SoVolumeRendering,
   SoUniformGridClipping,
   SoVolumeClippingGroup,
   SoInteractiveComplexity

 */
class VOLUMEVIZ_API SoVolumeRender : public SoVolumeShape {
  SO_NODE_HEADER( SoVolumeRender );

 public:

   /**
    * Method to use when moving in low resolution. Used with #lowResMode field.
    */
   enum LowResMode
   {
  /**
   * No low resolution mode when moving.
   */
     DECREASE_NONE = 1,
  /**
   * Decrease the number of samples according to #SoComplexity::value
   * when moving. It has no effect if #numSlicesControl
   * is set to AUTOMATIC because this mode always uses
   *  the #SoComplexity node to compute the number of samples to draw.
   */
     DECREASE_SLICES = 1 << 1,
  /**
   * Downscale the screen resolution of the volume when
   * moving by the factor defined in #lowScreenResolutionScale.
   */
     DECREASE_SCREEN_RESOLUTION = 1 << 2
   } ;


  /** Number of samples control mode. Used with #numSlicesControl field. */
    enum NumSlicesControl {
    /**
     *  Use all samples.
     *  The number of samples depends on the viewing direction.
     */
    ALL,
    /**
     *  Use the number of samples specified by the #numSlices field.
     *  The number of samples does not depend on the viewing direction.
     */
    MANUAL,
    /**
     * (Recommended) Use a number of samples computed as follows: @BR
     * If @I numSlices @i is greater than zero, then @BR
     * n = @I complexity @i * 2 * @I numSlices @i @BR
     * where @I complexity @i comes from the value of field #SoComplexity::value
     * and @I numSlices @i comes from the #numSlices field.
     * The number of samples does not depend on the viewing direction.
     *
     * If @I numSlices @i is any value less than or equal to 0, the dimension of
     * the volume data is used instead of numSlices, so @BR
     * n = @I complexity @i * 2 * @I volumeDimension @i @BR
     * The number of samples depends on the viewing direction.
     *
     * The factor 2 is used because by default complexity is equal to
     * 0.5. So, by default, the behavior for numSlices greater than zero is the
     * same as MANUAL and the default behavior for numSlices less than or equal
     * to zero is the same as ALL. @BR
     * You can increase the complexity up to 1.0, which would double the number of
     * samples used.  You can decrease the complexity to reduce the number of
     * samples used.  The global alpha is corrected so that the brightness looks
     * the same.
     */
    AUTOMATIC,
    /**
     * Use a number of samples computed as follows: @BR
     * n = @I complexity @i * 2 * @I volumeDataDimension[mainVisibleAxis] @i @BR
     * where @I complexity @i comes from the value of field #SoComplexity::value. 
     * The number of samples depends on the viewing direction. @BR
     * @ENUM_SINCE_OIV 8.1
     */
    MAIN_AXIS
  };

  /**
   * Specifies the list of volumes on which volume rendering is applied.
   *
   * This field specifies a list of datasets using their dataset id
   * (see SoDataSet::dataSetId and SoDataSetId::id).
   * It defines the geometry of the volume rendering as the union of the extents
   * of each dataset specified in this list.
   *
   * For example, when equal to [1, 2], volume rendering is applied on the union
   * of the extents of the datasets of ids 1 and 2.
   *
   * Notes:
   *   - The SoROI node affects the union of the extents.
   *   - This field is useful only when datasets with different extents are present
   *     inside a SoMultiDataSeparator (see @ref MULTIPLE_INDEPENDENT_VOLUMES section).
   *   - A value of 0 in the list identifies the last SoVolumeData node on state.
   *   - If this field is empty, the volume rendering is applied to the union of the extents
   *     of each dataset on state.
   *
   * @anchor LIMITATIONS Limitations:
   * This field is ignored and only the last dataset on state is used for the volume rendering when:
   *   - Rendering using SoVolumeGroup
   *   - Performing an extraction using SoOffscreenVolumeRender
   *   - An SoProjection is applied to the volume rendering
   *   - SoVolumeRenderingQuality::voxelizedRendering is set to TRUE.
   *
   * The default value is an empty array.
   *
   * @FIELD_SINCE_OIV 10.12.0
   */
  SoMFInt32 dataSetIds;

  /**
   * Controls how the number of samples along each ray is determined. @BR
   * @useenum{NumSlicesControl}. Default is AUTOMATIC.
   * Generally increasing the number of samples will increase
   * the image quality, but decrease the performance (frames per second).
   *
   * We recommend to set the #numSlicesControl field to AUTOMATIC
   * and the #numSlices field to 0. The number of samples will be computed based on the 
   * dimensions of the volume (number of voxels on each axis), the #SoComplexity::value
   * setting and the viewing direction. If the viewing direction changes, the number of 
   * samples will be automatically adjusted.
   */
  SoSFEnum  numSlicesControl;

  /**
   * Specifies the number of samples along each ray. @BR
   * The default is -1, which means to use the volume dimensions (number of voxels on
   * each axis) when the #numSlicesControl field is set to AUTOMATIC.
   *
   * NOTE: This value is not used if the #numSlicesControl field is set to ALL
   * (the default for that field).
   */
  SoSFInt32 numSlices;

  /**
   * Sets the method to use when moving in low resolution. 
   * @useenum{LowResMode}. Default is DECREASE_NONE.
   *
   * - DECREASE_NONE: Default. Do not use low resolution mode when moving (i.e., when the
   *   viewer is moving the camera).
   *
   * - DECREASE_SLICES: Decrease the number of samples according to #SoComplexity::value when moving.
   *  It has no effect if #numSlicesControl is set to AUTOMATIC because in this case, VolumeViz always uses
   *  the SoComplexity node to compute the number of samples.
   *
   * - DECREASE_SCREEN_RESOLUTION: Downscale the screen resolution of the volume when
   *   moving by the factor specified in #lowScreenResolutionScale. This is the recommended
   *   setting when rendering performance is too low.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBitMask lowResMode;

  /**
   * If #lowResMode is DECREASE_SCREEN_RESOLUTION, render the volume at a lower screen resolution.
   * when moving. The resolution used is the current screen resolution divided by
   * lowScreenResolutionScale. Default is 1.  A value of 2 or 4 is recommended if using this option.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFInt32 lowScreenResolutionScale;


  /**
   * If true, LDM tiles will be subdivided for rendering. Fully transparent sub-tiles won't
   * be rendered, thus (potentially) increasing the speed of the rendering if an
   * expensive shader is being used and significant regions of the volume are fully
   * transparent.  However using false is faster if the user will frequently change
   * the data range (e.g. window center/width in medical applications).
   * SubTileDimension can be changed using SoVolumeData node's @I ldmResourceParameters@i field.
   * Default is FALSE.
   *
   * See the 'subtileDimension' field in SoLDMResourceParameters.  If the tileDimension is
   * larger than the default value, then the subtileDimension should also be larger to
   * avoid performance issues.
   *
   * @BR @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBool subdivideTile;
  
  /**
   * When this field is set to FALSE (the default), the number of samples
   * set by #numSlices is the number of samples used for the region defined by the current ROI.
   * Therefore the number of samples may change when the ROI size changes.
   * When TRUE, #numSlices is the number of samples for the whole volume.
   * In this case the sample density is constant, independent of the ROI size.
   * Default is FALSE.
   * @BR @BR
   * @FIELD_SINCE_OIV 7.1
   */
  SoSFBool fixedNumSlicesInRoi;

  /**
   * When doing volume projection (see SoProjection), only the geometry (corner
   * vertices) of the LDM tiles are projected, not the individual voxels.
   * This can produce an imprecise projected volume when using large LDM tiles or
   * low resolution levels (where the LDM tiles are larger).
   *
   * This field controls how many times the tile geometry will be subdivided
   * (producing more vertices) before being projected. Subdivision gives a smoother,
   * more accurage shape, but requires much more computational power and may
   * reduce rendering performance.  Default is 1 (subdivide once).
   *
   * NOTE: This field is ignored in raycasting mode (the default).
   *
   * @BR @BR
   * @FIELD_SINCE_OIV 7.1
   */
  SoSFInt32 projectedTileSubdivision;

  /**
   * Controls whether opacity correction is done. @BR
   * If TRUE, opacity is automatically adjusted to give similar appearance
   * no matter how many samples are used. If FALSE, opacity is not corrected
   * and increasing the number of samples will increase the opacity. Default is TRUE.
   *
   * Generally this field should always be set to true.
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFBool opacityCorrection;

  /**
   * Composition mode. Used with the #renderMode field.
   *
   * @ENUM_SINCE_OIV 9.1
   */
  enum RenderMode
  {
    /** Alpha compositing (Default). Also called Direct Volume Rendering (DVR) it blends the R, G, and B components
     *  for each data value along the sampling ray based on the alpha value (alpha blending). */
    VOLUME_RENDERING = 0,

    /** Minimum intensity projection (MinIP).
     *  The minimum value found along each ray is used to determine color. */
    MIN_INTENSITY_PROJECTION,

    /** Maximum intensity projection (MIP).
     *  The maximum value found along each ray is used to determine color. */
    MAX_INTENSITY_PROJECTION,

    /** Ray sum intensity projection (RSP).
     *  The values found along each ray are accumulated. Then the total value is
     *  normalized by the nominal number of samples for every ray (see #numSlices),
     *  which is the same for all rays. The values seen in shader functions, e.g.
     *  VVizComputeFragmentColor are still in the range 0..1, but the range of values
     *  that represents is quite different from the voxel data range. */
    SUM_INTENSITY_PROJECTION,

    /** Average Intensity projection (AIP).
     *  The values found along each ray are accumulated. The total value is
     *  divided by the actual number of samples along the current ray.  The actual
     *  number of samples may be different for each ray. The resulting data range
     *  will be similar to the voxel data range but not necessarily the same. If
     *  there are many low, e.g. 0, valued voxels, the average may be very small. */
    AVERAGE_INTENSITY_PROJECTION,

    /** Maximum Intensity Difference Accumulation (MIDA).
     *  Assuming the important features of the volumetric datasets are of high scalar values, this mode changes the
     *  compositing behavior by decreasing the accumulated color and opacity when the incoming sample's scalar
     *  intensity is higher than any of those before it along the viewing ray. In this way, the high scalar value features
     *  in the volumetric datasets won't be occluded as much as in a standard compositing (i.e. VOLUME_RENDERING).
     *
     *  See @B Instant volume visualization using maximum intensity difference accumulation. @b @I Stefan Bruckner and
     *  M. Eduard Groller. 2009. In Proceedings of the 11th Eurographics / IEEE - VGTC conference on Visualization (EuroVis'09) @i
     *  for details.
     */
    MAX_INTENSITY_DIFFERENCE_ACCUMULATION,

    /** Intensity Difference Accumulation.
     *  This mode acts as MIDA but only the opacity of the accumulated color is affected and the modulation is based
     *  on the intensity difference of two consecutive samples. This mode should emphase the difference between two
     *  different structures with different scalar intensities.
     *
     *  See @B Fast Volumetric Data Exploration with Importance-Based Accumulated Transparency Modulation.
     *  @b @I Wan Yong and Hansen Chuck, 2010, IEEE/ EG Symposium on Volume Graphics @i for details.
     */
    INTENSITY_DIFFERENCE_ACCUMULATION,

    /** Maximum Gradient Difference Accumulation.
     *  This mode acts as INTENSITY_DIFFERENCE_ACCUMULATION but the composition is affected when the incoming
     *  sample's gradient is higher than any of those before it along the viewing ray. It aims at revealing
     *  important features (i.e. higher gradient transitions) all along the ray direction.
     */
    MAX_GRADIENT_DIFFERENCE_ACCUMULATION,

    /** Gradient Difference Accumulation.
     *  This mode acts as MAX_GRADIENT_DIFFERENCE_ACCUMULATION but the opacity modulation is based on the gradient
     *  difference of two consecutive samples. It leads to a more transparent result, as more samples along the
     *  viewing ray contribute to the final result.
     */
    GRADIENT_DIFFERENCE_ACCUMULATION
  };

  /** 
  * Specifies how the voxels along each sampling ray are combined to form the final image.
  * @useenum{RenderMode}. Default is VOLUME_RENDERING (alpha blending).
  *
  * @FIELD_SINCE_OIV 9.1
  */
  SoSFEnum renderMode;

  /**
   * Sampling alignment. Used with the #samplingAlignment field.
   * @ENUM_SINCE_OIV 9.1
   */
  enum SamplingAlignment
  {
    /** Samples are located on planes perpendicular to the view direction. */
    VIEW_ALIGNED = 0,
    /** Samples are located on planes perpendicular to one axis of the volume. */
    DATA_ALIGNED,
    /** Samples are located on shells aligned with the volume's internal "boundary".
     *  Each ray begins sampling at the first intersected voxel that has an alpha value > #opacityThreshold. */
    BOUNDARY_ALIGNED,
    /** 
     * Similar to BOUNDARY_ALIGNED but uses a cubic interpolation to compute the boundary,
     * giving smoother results when using #SoVolumeRenderingQuality::deferredLighting.
     *
     * When using linear interpolation and deferred lighting, BOUNDARY_ALIGNED can generate
     * "flat shading" due to linear interpolation between voxels. SMOOTH_BOUNDARY_ALIGNED
     * will generate a smooth shading using a cubic interpolation to find the volume's boundary
     * and use a linear interpolation for the rest of volume. This is a good compromise between 
     * result and performances of a full tricubic interpolation.
     * Please refer to #SoVolumeRenderingQuality::deferredLighting and #SoVolumeShape::interpolation for details.
     * @B LIMITATIONS @b
     * SMOOTH_BOUNDARY_ALIGNED is not supported in the following case:
     * - With isosurface rendering
     * - When using gradients computed from the voxel values (SoVolumeRenderingQuality::lighting for instance).
     * - When using SoUniformGridClipping nodes.
     * In one of the above case, BOUNDARY_ALIGNED is used.
     * @warning Possibly heavy GPU usage
     */
    SMOOTH_BOUNDARY_ALIGNED,
  };

  /** 
   * Specifies which technique to use to align rayCast samples.
   * @useenum{SamplingAlignment}. Default is BOUNDARY_ALIGNED.
   *
   * VIEW_ALIGNED: Samples are located on planes perpendicular to the view direction.
   *
   * @IMAGE samplingAlignment_VIEW_ALIGNED.png
   *
   * DATA_ALIGNED: Samples are located on planes perpendicular to one axis of the volume. @BR
   *
   * @IMAGE samplingAlignment_DATA_ALIGNED.png
   *
   * BOUNDARY_ALIGNED: Samples are located on shells aligned with the volume's internal "boundary". 
   * Each ray begins sampling at the first intersected voxel that has an alpha value > #opacityThreshold.
   * This technique greatly decreases "slicing" artifacts even with a relatively small number of slices.
   * It is strongly recommended to enable this mode when using SoVolumeRenderingQuality::ambientOcclusion.
   *
   * @IMAGE samplingAlignment_BOUNDARY_ALIGNED.png
   *
   * SMOOTH_BOUNDARY_ALIGNED: Similar to BOUNDARY_ALIGNED but uses a cubic interpolation to compute the boundary,
   * giving smoother results when using #SoVolumeRenderingQuality::deferredLighting.
   *
   * NOTE: If an SoVolumeGroup or SoProjection node
   * applies to this node, the field is ignored and VIEW_ALIGNED is used.
   *
   * @FIELD_SINCE_OIV 9.1
   */
  SoSFEnum samplingAlignment;

  /**
   * Specifies a threshold opacity (alpha) value that defines voxels considered to be "solid" (non-transparent).
   * Many volume data sets are composed of objects of interest floating in an approximately transparent "fog",
   * depending on opacity values in the transfer function. Several effects like BOUNDARY_ALIGNED sampling,
   * ambientOcclusion and deferredLighting need to locate the boundary between solid objects and ambient fog.
   * This field defines a threshold in the range [0, 1]. Voxels are considered solid if their alpha value is
   * greater than opacityThreshold.
   *
   * A value of 0.0 generally works well with a transfer function containing "binary" opacity (a transfer
   * function with only fully transparent or fully opaque colors). If this is not the case, you should set
   * this value according to your visualization parameters. Generally between 0.1 and 0.5.
   *
   * Default is 0.1, Ignore low visibility voxels.
   *
   * @FIELD_SINCE_OIV 9.3
   */
  SoSFFloat opacityThreshold;

  /**
   * Specifies the size of the screen tiles used for rendering. Default is (-1, -1), it means no screen tiling rendering.
   */
  SoSFVec2i32 stillScreenTileSize;

  /**
   * Constructor.
   */
  SoVolumeRender();

  /**
   * Abort code for callback
   */
  enum AbortCode {
    /**
     *  Continue rendering as usual.
     */
    CONTINUE,
    /**
     *  The render action of the SoVolumeRender node is aborted. The render action
     * continues for the remaining part of the scene graph.
     */
    ABORT,
    /**
     *  The current slice is not drawn. Rendering continues with the next slice.
     */
    SKIP
  };

  /**
   * [OIV-WRAPPER NAME{RenderAbortCB}]
   */
  typedef AbortCode SoVolumeRenderAbortCB(int totalElems, int thisElem, void *userData);

  /**
   * Sets \if_dotnet delegate \else callback \endif to call during texture map rendering to test for an abort
   * condition. When not in raycasting mode, it will be called for each element that is rendered.
   *
   * In the case of LDM, an element is a tile, @B totalElems @b is the number of tiles that
   * will be drawn, and @B thisElem @b is the number (counting from one) of the next tile
   * to be drawn. The quotient @B thisElem @b / @B totalElems @b represents the progress of
   * the volume rendering process.
   *
   * In the non-LDM case, an element is a slice, @B totalElems @b is the number of slices
   * that will be drawn, and @B thisElem @b is the number (counting from one) of the next
   * slice to be drawn. The quotient @B thisElem @b / @B totalElems@b represents the progress
   * of the volume rendering process.
   *
   * This allows applications to terminate rendering of this node prematurely if some
   * condition occurs. It does not terminate traversal of the scene graph. (For that
   * use the SoGLRenderAction abort callback.) The callback function should return
   * one of the AbortCode codes to indicate whether rendering should continue.
   */
  void setAbortCallback( SoVolumeRenderAbortCB *func, void *userData = NULL);


#if SoDEPRECATED_BEGIN(8500)

  /**
   * Indicates if lighting is required. Default is FALSE. @BR
   * NOTE: Better performance for lighting can be obtained using the SoVolumeRenderingQuality
   * node. Using SoVolumeRenderingQuality, lighting is determined by the first light
   * in the scene graph (similar to other geometry) and the lighting computation is done on
   * the GPU (therefore SoVolumeRenderingQuality requires programmable shaders).
   *
   * Using the #lighting field, lighting is determined by the #lightDirection field and
   * the lighting computation is done on the CPU. This requires RGBA textures to be loaded
   * on the GPU which uses more texture memory and requires more time to modify the
   * transfer function (color map) because textures must be reloaded.  Note that activating
   * or deactivating lighting will also normally force
   * the textures to be recreated, which may be slow.
   *
   * NOTE: Only set the lighting field to TRUE in SoVolumeRenderingQuality or
   * SoVolumeRender. Do not set both #lighting fields to TRUE.
   */
  SoDEPRECATED_MEMBER_NOWARN(8500,"Use SoVolumeRenderingQuality::lighting field instead.")
  SoSFBool  lighting;

  /**
   * Light direction (relative to the volume). The default is (-1,-1,-1).
   * Only affects CPU computed lighting (i.e. when the #lighting field is TRUE).
   */
  SoDEPRECATED_MEMBER_NOWARN(8500,"Use SoVolumeRenderingQuality::lighting field instead.")
  SoSFVec3f lightDirection;

  /**
   * Light intensity in the range [0-1]. The default is 1.
   * Only affects CPU computed lighting (i.e. when the #lighting field is TRUE).
   */
  SoDEPRECATED_MEMBER_NOWARN(8500,"Use SoVolumeRenderingQuality::lighting field instead.")
  SoSFFloat lightIntensity;

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9100)

  /**
   * Indicates if samples should be computed in a view-aligned manner. Default is TRUE.
   *
   * NOTE: If an SoVolumeIsosurface, SoVolumeRenderingQuality or SoProjection node
   * applies to this node, this field is ignored and view-aligned samples are used.
   */
  SoDEPRECATED_MEMBER_NOWARN(9100,"Use samplingAlignment field instead.")
  SoSFBool viewAlignedSlices;

#endif /** @DEPRECATED_END */

/*****************************************************************************/
SoEXTENDER public:
  /** @copydoc SoVolumeShape::computeBBox */
  virtual void computeBBox(SoAction *, SbBox3f &box, SbVec3f &center);

  /** @copydoc SoVolumeShape::rayPick */
  virtual void rayPick(SoRayPickAction *action);

/*****************************************************************************/
SoINTERNAL public:

  /** @copydoc SoVolumeShape::initClass */
  static void initClass();

  /** @copydoc SoVolumeShape::exitClass */
  static void exitClass();

  /**
   * Return the volume rendering quality currently in the state if any.
   */
  SoVolumeRenderingQuality* getVolumeRenderingQuality( SoState* state ) const;

  /**
   * Return the volume offscreen implementation.
   */
  SoVolumeOffscreenImpl* getVolumeOffscreenImpl() const;

#if SoDEPRECATED_BEGIN(9630)

  SoDEPRECATED_METHOD( 9630, "Slabbing is not useful anymore." )

#endif /** @DEPRECATED_END */
   /** Return true if we need the depth texture of current OIV scene with the 
    * current rendering parameters..
    * If true, internalFormat is the needed texture format. */
   bool needOivDepth(SoState* state);

   /** Return true if we need COLOR_ATTACHEMENT0 RTT to
    * apply the current rendering parameters.
    * If true, internalColorFormat is the needed internal
    * texture format (GL_RGBA32F, ...). */
   bool needRttColor(SoState* state) const;

   /** Return true if we need RTT EDGE_DETECT to apply the current EdgeDetect effect.
    * If true, internalFormat is the needed texture format. */
   bool needRttEdgeDetect(SoState* state) const;

   /** Return true if we need RTT BOUNDARY_DEPTH to apply current imageSpace effect.
    * If true, internalFormat is the needed texture format. */
   bool needRttBoundaryDepth(SoState* state) const;

   /** Return true if we need RTT COLOR_UNTIL_BOUNDARY to apply current imageSpace effect.
    * If true, internalFormat is the needed texture format. */
   bool needRttBoundaryColor(SoState* state) const;

   /** Return sampling alignment according to SoInteractiveComplexity */
   SamplingAlignment getSamplingAlignment(SoState* state) const;

   /** Remove given dataset from internal list of managed ds */
   void removeDsFromRegList(SoDataSet* pDs);
   
   /** Keep composition/renderMode field synchronised for compat purpose */
   virtual void notify(SoNotList *list);

   /**
   * Called when user stop or start interaction
   */
   void onUserInteractionChange(SoState*state, bool stopMoving);
   
/*****************************************************************************/
 protected:
   // Destructor
   virtual ~SoVolumeRender();

   void createVolumeRender( SoState* state );

   void generatePrimitives(SoAction *action);

   void doRendering(SoGLRenderAction *action);

   void setupVolumeRenderInterface(SoState* state);

   SoDetail *createTriangleDetail(SoRayPickAction *,
                                  const SoPrimitiveVertex *v1,
                                  const SoPrimitiveVertex *,
                                  const SoPrimitiveVertex *,
                                  SoPickedPoint *);

   /**
    * Begin offscreen rendering:
    * Initialise RTT and fbo if needed.
    */
   void beginOffscreenRendering( SoGLRenderAction* action );

   /**
    * End offscreen rendering:
    * release RTT and fbo if needed.
    */
   void endOffscreenRendering( SoGLRenderAction* action );

   /**
    * Activate front to back rendering
    */
   void setupFrontToBackGlState(SoState* state);

  /**
   * Returns TRUE if we are doing custom depth peeling rendering.
   * In this case it means that the volume should not be rendered.
   */
  SbBool shouldGLRender(SoGLRenderAction *renderAction, SbBool isPointsOrLines);

/*****************************************************************************/
SoINTERNAL protected:

  /** Implement the SoLdmValuationAction */
  virtual void ldmAction(SoLdmValuationAction* action);

/*****************************************************************************/
private:

#if SoDEPRECATED_BEGIN(9630)
  SoDEPRECATED_FIELD( 9630, "No longer used. Only raycasting mode is supported." )
  SoSFBool useEarlyZ;
  SoDEPRECATED_FIELD( 9630, "No longer used. Only raycasting mode is supported." )
  SoSFInt32 numEarlyZPasses;
  SoDEPRECATED_FIELD( 9630, "No longer used. Only raycasting mode is supported." )
  SoSFBool gpuVertexGen;
#endif  /** @DEPRECATED_END */

   /** Fill detail with all necessary infos*/
   void fillDetail(SoRayPickAction* rpa, SoVolumeRenderDetail* detail, const SoLDM::DataSetIdPair& idPair);

   void genPrimDataSet(SoAction* action, const SoLDM::DataSetIdPair& idPair);

   /** Return true if subtiles can be used */
   bool canSubdivideTiles(SoState* state);

   /** Return true if raycasting mode has to be used */
   bool isRaycastingEnabled(SoState* state) const;

  SoVolumeRenderAbortCB	*m_abortCBfunc;
  void  *m_abortCBuserData;
  void   GLRenderTex2D3D(SoGLRenderAction* action);

   /**
    * Do volume rendering in a fbo if needed (low res move)
    */
   void useOffscreenRendering(SoGLRenderAction* action);

  /**
   * Create internal shaders
   */
   void initInternalShaders();

  void setupDepthPeelingTextures(SoGLRenderAction *action);

  void rayPickDataSet(SoRayPickAction* action, const SoLDM::DataSetIdPair& idPair);

  /** Use to apply post-process effects (edgeDetect, AO, etc...). */ 
  SoVRImageSpaceEffects* m_imageSpaceEffects;

  /** Render at floor(1/m_vpScale of the viewport size). 
   * Scale that is currently used to render the viewport. it is equal to
   * m_lowScreenResScale if rendering in INTERACTIVE and using lowResMove, and is 1
   * if rendering in STILL mode. 
   * Note that fullResSize/lowResSize may not be exactly equal to m_vpScale
   * if fullResSize cannot be divided by lowResSize. */
  int m_vpScale;

  /** The requested lowScreen resolution scale. Note that this value is always 
   * equal to lowScreenResolutionScale field, even if we are in STILL mode (full res). */
  int m_lowScreenResScale;

  SoShaderParameter1i *m_opacityTex;
  SoGLProgramGLSL* m_binaryOpacityComposeShader;

  SoShaderProgram* m_writeFragShader;

  SoVolumeIsosurface* m_volumeIsosurface;

  /**
   * Delete all shaders, fbo and textures used for some effects
   */
  void deleteShadersFboTextures();

  /**
   * Update texture size if lowScreenResolutionScale has changed
   */
  void updateLowResScale( SoState* state );

  /** 
  * register a given contextId to a list of DataSet
  */
  void registerCtxId(const SoLDM::DsVector &dsVector, int ctxId); 

  /** 
  * unregister a given contextId to a list of DataSet
  */
  void unregisterCtxId(const SoLDM::DsVector &dsVector, int ctxId);

  /** true if using lowResMove while moving. Note that this value can be true even if we don't move. */
  bool m_useLowResMove;

  /** true if using low slice move while moving. Note that this value can be true even if we don't move. */
  bool m_useLowNumSliceMove;

  SoVolumeRenderLdm* m_vriLdm;
  SoVolumeRenderRaycast* m_vriRaycast;
  SoVolumeRenderInterface* m_vri;

  bool m_frontToBackComposition;

  // List of dataset gathered in previous valuation action
  SoLDM::DsVector m_prevDsList;

  // Callback to manage deletion of dataset
  static void volDataDeleteCB(void* data, SoSensor *sensor);

  // ctxId attached to this volumeRender. Set by ValuationAction.
  int m_prevRegisteredCtxId;

  SoVolumeStateVr* m_vs;

  SoVolumeOffscreenImpl* m_volumeOffscreenImpl;

  // Associate a Node sensor to each DataSet managed
  std::map<SoDataSet*, SoNodeSensor*> m_mapDsSensor;

  friend class SoVolumeRenderLdm;
  friend class SoVolumeRenderRaycast;

  SoRef<SoGroup> m_depthPeelingSG;
  SoRef<SoGroup> m_depthPeelingColorTexGroup;
  SoRef<SoTextureUnit> m_depthPeelingColorTexUnit;
  SoRef<SoGroup> m_depthPeelingDepthTexGroup;
  SoRef<SoTextureUnit> m_depthPeelingDepthTexUnit;
};
