/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Aug 2005)
**=======================================================================*/


#ifndef _SO_VOLUME_QUALITY_H_
#define _SO_VOLUME_QUALITY_H_


#include <Inventor/SbBasic.h>
#include <Inventor/fields/SoMFNode.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoSubNode.h>
#include <Inventor/nodes/SoGroup.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

/*----------------------------------------------------------------------------*/
class SoFragmentShader;
class SoVertexShader;
class SoTransferFunction;
class SoVolumeShader;
class SoNodeDependencies;

/**

 @VREXT Volume rendering quality property mode

 @ingroup VolumeVizNodes

 @DESCRIPTION
 This property node causes subsequent SoVolumeRender nodes to be drawn with different
 rendering effects and/or levels of quality.

# NOTES
 - Only one (or none) of the following nodes can be active at a time:
   SoVolumeShader or SoVolumeRenderingQuality. However since Open Inventor 7.1, 
   both SoVolumeRenderingQuality and SoVolumeIsosurface may applied to the same
   SoVolumeRender node.

 - If the application simply needs to replace one or more of the VolumeViz shader
   rendering functions with a customized shader, you can use either SoVolumeShader
   or SoVolumeRenderingQuality. However if the application needs the advanced
   rendering features of SoVolumeRenderingQuality, e.g. lighting, in addition to
   the customized behavior, then you should use this node (which is derived from
   SoVolumeShader). Using the shader rendering framework is explained on the
   SoVolumeShader page.

 - The SoVolumeRenderingQuality node must be before the SoVolumeRender and after the
   SoTransferFunction.

 - This is a shader node!  
   The effect will usually be undesirable if it is
   applied to standard geometry (polygons, lines, etc).  Therefore applications
   should generally keep the volume visualization nodes and standard geometry
   nodes separate in the scene graph (i.e. under different SoSeparator nodes).

 - Please read the comments for each field.  
   Some options only apply to gradient lighting and have no effect on deferred lighting.

# Pre-integrated and lighted rendering
 Various enhanced rendering modes for volume rendering are available:
  - Pre-integrated volume rendering
  - Lighted volume rendering

 The pre-integrated mode (#preIntegrated field) generally provides higher image
 quality for the same number of slices. Most applications should enable this field.
 However, note that pre-integration integrates between color map values.
 For typical scalar data containing sampled values, this provides a beneficial smoothing.
 It is especially useful when the color changes sharply between adjacent color map
 entries.  However, when the color changes sharply between adjacent voxels, it can
 can cause values that are not actually in the original data to be displayed. 
 This is undesirable for some data, for example "label" volumes resulting from
 segmentation.  Pre-integration is not recommended for such data.

 Also note:
  - The #preIntegrated field has no effect on SoVolumeIsosurface rendering.
  - The #preIntegrated field is only considered when SoVolumeRendering::renderMode
    is set to VOLUME_RENDERING (the default).

 When lighting is enabled for volume rendering, VolumeViz applies the same lighting
 equation used for polygonal geometry, including (optionally) specular highlights.
 The base voxel color comes from the SoTransferFunction node or (optionally) from a
 custom shader function provided by the application.  This color is modified using the
 current material (SoMaterial), a vector simulating a "normal vector" and one or more
 directional light nodes (SoDirectionalLight) found in the scene graph.  Voxels can
 also cast and receive shadows (see SoShadowGroup).

*  **Limitation**: Pre-integrated and lighted rendering are not supported if using custom
 fragment shaders and redefining FRAGMENT_COMPUTE_COLOR slot.

 VolumeViz supports two lighting modes using either the #lighting field or the 
 #deferredLighting field. In both cases lighting is computed on the GPU as part of the
 rendering process.
  - **Gradient lighting**  
    The #lighting field enables gradient based lighting, where the "normal vector"
    at each sample along the ray is a gradient vector computed from the data values
    in the volume. Gradient based lighting only supports a single light source and
    only the first light in the scene graph (typically the viewer's "headlight") is
    used. No other lights affect the volume in this case.
    The #gradientQuality field controls the algorithm used to
    compute gradient vectors. Several other fields affect the gradient computation
    including #gradientThreshold, #surfaceScalarExponent and #unnormalizedGradientExponent.
    Setting surfaceScalarExponent to a small value, for example 2.0, is recommended.

  - **Deferred lighting**  
    The #deferredLighting field enables screen space lighting, where the "normal vector"
    is computed from the final image depth buffer.  Deferred lighting is faster and
    supports multiple light sources, but works best when the transfer function makes each
    data value either opaque or transparent.  Deferred lighting is not affected by gradient
    related fields.

 Unlike other primitives
 (including other VolumeViz primitives), volume lighting is not affected by an
 SoLightModel node. Also unlike other primitives, if lighting is enabled and there are
 no light nodes in the scene graph, the voxel color is taken from the transfer function
 or custom shader function "as is" (other primitives would be black in this case).

 Each light node's direction and intensity fields are used, but the color field is not currently used.
 The current  SoMaterial specifies the ambient, specular, diffuse, and emissive color values for
 the lighting equation. Note that the default diffuse color is "light gray" (0.8), not
 full white. This allows specular lighting to push the color toward full white (as expected).
 If specular lighting is not desired, then changing this to full white (1.0) is recommended
 in order to see the true colors specified in the transfer function.

   The following figures show the same volume data:

      @TABLE_1B
         @TR Default volume rendering @TD Pre-integrated volume rendering @TD Lighted pre-integrated volume rendering
         @TR @IMAGE quality_volume.jpg
                   @TD @IMAGE quality_preintegrated.jpg
                   @TD @IMAGE quality_light.jpg
      @TABLE_END

 # Quality enhancement parameters
 
  - **Jittering**  
     When #jittering is set to true, a random offset is added to texture
     coordinates in order to decrease "ring" artifacts without the
     cost of drawing a higher number of slices.
     Note that this creates a "noisy" image.
     Instead we recommend setting the SoVolumeRender::samplingAlignment field to BOUNDARY_ALIGNED.

      @TABLE_1B
         @TR No Jittering @TD With Jittering
         @TR @IMAGE volume_nojitter.jpg
         @TD @IMAGE volume_jitter.jpg
      @TABLE_END

  - **Gradient quality**  
      When gradient lighting (#lighting field) is enabled, the
     #gradientQuality field allows you to choose between
     various gradient computation techniques. The computational cost increases
     with the quality.  Has no effect on deferred lighting.

      @TABLE_1B
         @TR Low quality @TD Medium quality @TD High quality
         @TR @IMAGE volume_forwarddiff.jpg
         @TD @IMAGE volume_centraldiff.jpg
         @TD @IMAGE volume_sobel.jpg
      @TABLE_END

  - **Surface scalar**  
     When gradient lighting (#lighting field) is enabled or #edgeDetect2DMethod
     is GRADIENT, the #surfaceScalarExponent field disables lighting (or edge detection)
     on uniform surfaces in order to avoid noise in these area. This
     field should not be mixed with #unnormalizedGradientExponent. The default value is zero,
     but a small value, for example 2.0, is recommended.  Has no effect on deferred lighting.

      @TABLE_1B
         @TR Surface Scalar disabled  @TD Surface Scalar enabled
         @TR @IMAGE surfaceScalarOff.jpg
         @TD @IMAGE surfaceScalarOn.jpg
      @TABLE_END

  - **Unnormalized gradient**  
     When gradient lighting (#lighting field) is enabled, if
     #unnormalizedGradientExponent is not 0, voxels with small gradients
     will get more contribution from the ambient light than voxels
     with high gradients. It is similar to #surfaceScalarExponent but
     uses the ambient light instead of the transfer function color for
     uniform surfaces.  Has no effect on deferred lighting.

  - **Gradient threshold**  
      When gradient lighting is enabled, gradients with a
      length less than #gradientThreshold are ignored during the lighting
      computation. This avoids doing lighting on noise while still lighting
      important data. In the following screenshots, setting a threshold of 0.1
      removed lighting on the noise surrounding the spheres.  Has no effect on deferred lighting.

      @TABLE_1B
         @TR With #gradientThreshold set to 0 @TD With #gradientThreshold set to 0.1
         @TR @IMAGE volume_gradientThr0.jpg
         @TD @IMAGE volume_gradientThr1.jpg
      @TABLE_END

 # Image enhancement parameters

 Various image enhancement techniques are available in this node and
 in SoTransferFunction.

  - **Ambient occlusion**  
      The #ambientOcclusion field enables a rendering technique that simulates self-shadowing
      of the volume. In other words, the amount of ambient (global) light in the scene reaching
      each sample is reduced by neighboring voxels.  This effect makes it much easier to see
      the relative depth of features in the volume. Generally we recommend using this effect
      rather than the following effects.

      Both lighting and/or shadow casting may also be enabled, but neither is required to use ambient
      occlusion.  This effect works best when the volume data contains surfaces (region boundaries
      with relatively sharp gradients) or in voxelized rendering mode.  In both cases it works best
      when voxels are either transparent or nearly opaque. If you use ambientOcclusion,
      you should set the SoVolumeRender::samplingAlignment field to BOUNDARY_ALIGNED to reduce "slicing"
      artifacts.

  - **Boundary opacity**  
      #boundaryOpacity increases opacity depending on the length of the gradient vector.
      Areas with large gradient changes will have their opacity increased according to
      the #boundaryOpacityIntensity.  Note that this option has a significant performance
      penalty.

  - **Edge coloring**  
      When #edgeColoring is on, the color of each voxel may be mixed
      with the #edgeColor. Areas where the normal (computed from the gradient)
      is facing the camera will have an unmodified color, whereas areas where the normal
      is more perpendicular to the view direction will tend towards #edgeColor.

  - **Edge detection**  
      When #edgeDetect2D is enabled, an image space filter is applied on
      the volume rendering image in order to detect edges, which will
      be highlighted. The results are affected by the #edgeDetect2DInnerThreshold
      and #edgeDetect2DOuterThreshold fields. The #edgeDetect2DMethod bitmask allows to
      apply the edge detection on the image's luminance, depth and/or gradient. The gradient
      method may give noisy result, the #surfaceScalarExponent may help to improve the result
      in this case.  The gradient method has a significant performance penalty.

      The following table shows the available edge detection techniques
      (explanation of faux shading is in SoTransferFunction):
      @TABLE_1B
         @TR No Edges @TD Boundary Opacity @TD Edge 2D
         @TR     @IMAGE volume_noedges.jpg
             @TD @IMAGE boundary.jpg
             @TD @IMAGE edge2d.jpg
         @TR Edge Coloring @TD Faux Shading (see SoTransferFunction)
             @TR @IMAGE edgecoloring.jpg
             @TD @IMAGE fauxshading.jpg
      @TABLE_END

 Because this node is derived from SoVolumeShader, IVVR_FIRST_RESERVED_TEXTURE_UNIT applies
 to it. See SoVolumeShader for more information.

# Volume projection

 Volume projection (SoProjection or derived classes) is incompatible with some 
 options enabled by this node.

 Do not enable the preIntegrated, jittering or edgeDetect2D fields when using projection.

 @FILE_FORMAT_DEFAULT
    VolumeRenderingQuality {
    @TABLE_FILE_FORMAT
       @TR lighting           @TD FALSE
       @TR preIntegrated      @TD TRUE
       @TR jittering          @TD FALSE
       @TR gradientThreshold  @TD 0.0001
       @TR edgeColoring       @TD FALSE
       @TR edgeColor          @TD (0, 0, 0)
       @TR edgeThreshold      @TD 0.2
       @TR boundaryOpacity    @TD FALSE
       @TR boundaryOpacityIntensity @TD 1.5
       @TR boundaryOpacityThreshold @TD 1.5
       @TR edgeDetect2D @TD FALSE
       @TR edgeDetect2DInnerThreshold @TD 0.1
       @TR edgeDetect2DOuterThreshold @TD 0.1
       @TR edgeDetect2DMethod         @TD LUMINANCE
       @TR gradientQuality @TD MEDIUM
       @TR colorInterpolation @TD TRUE
       @TR unnormalizedGradientExponent @TD 0
       @TR surfaceScalarExponent @TD 5
       @TR segmentedInterpolation @TD FALSE
       @TR segmentedInterpolationThreshold @TD 0.5
       @TR voxelizedRendering @TD FALSE
       @TR voxelOutline       @TD FALSE
       @TR voxelOutlineThreshold  @TD 2.
       @TR voxelOutlineWidth  @TD 2.
       @TR voxelOutlineColor  @TD (0, 0, 0)
       @TR ambientOcclusion   @TD FALSE
       @TR deferredLighting   @TD TRUE
       @TR interpolateOnMove  @TD TRUE
    @TABLE_END
    }

 @ACTION_BEHAVIOR
 SoGLRenderAction  
 Sets volume rendering quality parameters in the traversal state.

 @SEE_ALSO
  SoVolumeRender,
  SoVolumeShader,
  SoVolumeIsosurface,
  SoTransferFunction


 */
class VOLUMEVIZ_API SoVolumeRenderingQuality : public SoVolumeShader {

  SO_NODE_HEADER(SoVolumeRenderingQuality);

public:
  /** Gradient Quality mode */
  enum GradientQuality
  {
    /**
     *
     * Use the forward difference technique to compute the gradient.
     * This is a fast technique but it gives a lower quality gradient.
     */
    LOW = 0,

    /**
     *
     * Use the central difference technique to compute the gradient.
     * This is a compromise between quality and speed.
     */
    MEDIUM,

    /**
     *
     * Use a Sobel filter to compute the gradient.
     * This gives hiqh quality gradients at the expense of speed.
     */
    HIGH,
#ifndef HIDDEN_FROM_DOC
    LAST_GRADIENT_QUALITY
#endif
  };

  /**
   * The edge detection algorithm used when #edgeDetect2D is TRUE, can work
   * on the image luminance, depth buffer and/or gradient. These enums can be combined
   * in the field #edgeDetect2DMethod.
   */
  enum EdgeDetect2DMethod {
    /** Edge detection will use the image's luminance. This method has a very little impact on speed. */
    LUMINANCE = 1,

    /** Edge detection will use the image's depth buffer. */
    DEPTH = 1 << 1,

    /** Edge detection will use the volume's gradient. These gradients are affected by
     * the #surfaceScalarExponent fied. This method is the most computationally expensive.
     */
    GRADIENT = 1 << 2
  };

  /**
   * Constructor
   */
  SoVolumeRenderingQuality();

  /**
  * Enable gradient based lighting (computed on the GPU).
  * Default is FALSE.
  * Gradient based lighting is computed using the direction and magnitude of gradient vectors
  * computed from the data values in place of "normal vectors" in the lighting equation. A
  * gradient vector is computed for each sample point along the rays cast through the volume.
  *
  * VolumeViz also supports screen space lighting (see the #deferredLighting field).
  *
  * Notes:
  * - The result of the gradient lighting computation is affected by how the gradient
  *   vectors are computed.  See the #gradientQuality field.
  *
  * - Gradient lighting may produce "noisy" results when the gradient magnitudes are
  *   small and/or the gradient directions are not consistent. This often happens, for
  *   example, in relatively homogeneous regions of a volume.  Several fields are
  *   provided to compensate for this effect.  See #gradientThreshold and #surfaceScalarExponent.
  *
  * - Both gradient lighting and deferred lighting may be enabled at the same time,
  *   but the intent is that the application should use one or the other.
  *
  * - In any case, do not use the SoVolumeRender::lighting field.
  * 
  * @B LIMITATIONS @b
  * - Only directional lights (SoDirectionalLight) are supported.
  *
  * - Only one light source, the first one traversed in the scene graph, is supported.
  */
  SoSFBool lighting;

  /**
  * Enable pre-integrated volume rendering. @BR
  * Pre-integrated rendering can significantly increase image quality
  * (at the cost of slightly lower performance). Default is TRUE.
  *
  * Limitations:
  * - Pre-integration integrates between color map values.
  *   For typical scalar data containing sampled values, this provides a beneficial smoothing.
  *   When the color changes sharply between adjacent voxels, it can can cause values that
  *   are not actually in the original data to be displayed. 
  *   This is undesireable for some data, for example "label" volumes resulting from
  *   segmentation.  Pre-integration is not recommended for such data.
  *
  * - The #preIntegrated field has no effect on SoVolumeIsosurface rendering.
  *
  * - The #preIntegrated field is only considered when SoVolumeRendering::renderMode
  *   is set to VOLUME_RENDERING (the default).
  */
  SoSFBool preIntegrated;

  /**
   * Enable jittering. @BR
   * If set to TRUE, a random offset is added to texture
   * coordinates in order to decrease ringing artifacts (boundary edge artifacts) without the
   * cost of drawing a higher number of slices.
   * Note that this creates a "noisy" image.
   * Instead we recommend setting the SoVolumeRender::samplingAlignment field to BOUNDARY_ALIGNED.
   * Note: Jittering is only available when pre-integrated rendering is enabled.
   * Default is FALSE.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBool jittering;

  /**
   * Ignore all gradients with a magnitude less than the specified threshold. @BR
   * Default is 0.0001, meaning that all gradients are used in the lighting computation.
   * Maximum useful value is 1.0, because gradient vectors are normalized.
   * Only affects gradient based lighting (#lighting field).
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat gradientThreshold;

  /**
   * Enable edge coloring. @BR
   * If TRUE, changes the color based on the gradient direction (normal).
   * Edges will be highlighted with the color specified in #edgeColor.
   * Areas where the normal (computed from the gradient) is facing the camera
   * will have an unmodified color, whereas areas where the normal
   * is more perpendicular to the view direction will tend towards
   * #edgeColor.  Default is FALSE.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBool edgeColoring;

  /**
   * Color used to draw edges when #edgeColoring or #edgeDetect2D is TRUE.
   * Default is black (0,0,0).
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFColor edgeColor;

  /**
   * If this value is low, more edges will be detected.
   * Default is 0.2.
   * Min value is 0.
   * There is no max, but most of the time a value between 0 and 1 is good.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat edgeThreshold;

  /**
   * Enable boundary opacity. @BR
   * If TRUE, increases boundary opacity based on the gradient magnitude.
   * Default is FALSE.
   *
   * This effect has a significant performance penalty.
   *
   * SoVolumeRender::subdivideTile will be disabled if set to TRUE.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBool boundaryOpacity;

  /**
   * Applies when #boundaryOpacity is TRUE. If > 1, this increases the boundary opacity globally. If < 1, decreases it.
   * Default is 1.5.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat boundaryOpacityIntensity;

  /**
   * Applies when #boundaryOpacity is TRUE.
   * If this value is low (near 0), only regions with a high gradient will be enhanced.
   * Default is 1.5.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat boundaryOpacityThreshold;

  /**
   * Enable 2D edge detection. @BR
   * If this field is TRUE, a 2D edge detection algorithm is used
   * to highlight edges in the rendered image of the volume. Default is FALSE.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFBool edgeDetect2D;

  /**
   * Increase this value to decrease noise on silhouette edges.
   * 1 = remove all edges, 0 = remove nothing.
   * Default is 0.1
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat edgeDetect2DInnerThreshold;

  /**
   * Increase this value to decrease noise on edges in the volume.
   * 1 = remove all edges, 0 = remove nothing.
   * Default is 0.1
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFFloat edgeDetect2DOuterThreshold;

  /**
   * Method used for 2D edge detection. @BR
   * Specifies the edge detection algorithm used when #edgeDetect2D is TRUE.
   * Edge detection can work on the image luminance, depth buffer and/or gradient.
   * For example, gradients may be too noisy to give interesting edges
   * and may be ignored with this field. The gradient method is also the slowest
   * (luminance being the fastest) because it needs more texture fetchs.
   * See #EdgeDetect2DMethod. Default is LUMINANCE and GRADIENT. @BR
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFBitMask edgeDetect2DMethod;

  /**
   * Specifies the algorithm used to compute gradients, for example,
   * normals for lighting.
   * Only affects gradient based lighting (#lighting field).
   *
   * @useenum{GradientQuality}. Default is MEDIUM. @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFEnum gradientQuality;

  /**
   * Controls interpolation of data values used for color lookup. @BR
   * If FALSE, interpolation is not done on data values used to access the colormap
   * but is still done when computing gradient for lighting. This may be used
   * to render a segmented dataset where interpolation is not appropriate.
   * Default is TRUE.
   * Only affects gradient based lighting (#lighting field).
   * 
   * Generally it's better to set the SoVolumeRender::interpolation field to
   * NEAREST, unless you really want to use gradient based lighting.
   *
   * @FIELD_SINCE_OIV 7.2
   */
  SoSFBool colorInterpolation;

  /**
   * If this field is greater than or equal to 1, a surface with a small gradient will be less lighted
   * than a surface with a high gradient. The resulting color is a mix between the lighted voxel color and
   * the non-lighted voxel color (ie: the corresponding entry in the colormap).
   * If the value is less than 1, the field is ignored.
   * Values higher than 256 will apply lighting on almost all surfaces. For most datasets,
   * values between 2 and 16 should be enough.  Default is 5. @BR
   * Only affects gradient based lighting (#lighting field).
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFFloat surfaceScalarExponent;

  /**
   * When unnormalizedGradientExponent is not 0, surfaces with high gradients will
   * be more lighted than surfaces with small gradients.
   * Voxels with small gradients will have less diffuse and specular than other with high gradients
   * (ie: ambient color will be the biggest contributor).
   * This lighting contribution
   * can be controlled with this field. If this value is high, smaller gradients will
   * contribute to lighting.
   * Values higher than 256 will apply lighting on almost all surfaces.
   * For most dataset, values between 2 and 16 should be enough.  Default is 0. @BR
   * Only affects gradient based lighting (#lighting field).
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFFloat unnormalizedGradientExponent;

  /**
   * Allow correct interpolation when rendering segmented data.
   * NOTE: Only available on SoVolumeIsosurface. @BR
   * @FIELD_SINCE_OIV 8.5
   */
  SoSFBool segmentedInterpolation;

  /** 
   * Defines the iso distance used when #segmentedInterpolation is true. @BR
   * Default is 0.5. value must be in the range [0 .. 1]
   * NOTE: Only available on SoVolumeIsosurface. @BR
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFFloat segmentedInterpolationThreshold;

  /**
   * If true, SoVolumeRender displays voxels as individual cubes.
   * Default is false.
   *
   * Limitations:
   * - When using multiple independent volumes (different dimensions or extents) inside a SoMultiDataSeparator,
   *   this feature is not supported. Only the last volume in the scenegraph will be displayed.
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFBool voxelizedRendering;

  /**
   * If true, draw the outline of voxels (default is false).
   *
   * Notes:
   * - Since Open Inventor 9.6, this mode applies to volume slice primitives (SoOrthoSlice etc)
   *   in addition to volume rendering.
   *
   * - In the case of SoObliqueSlice, you may see triangles, quadrilaterals and even
   *   irregular pentagons in some cases. These lines are correct and correspond to the
   *   intersections between the voxels and the oblique slice.  For example:
   * @BR
   * @IMAGE outline_oblique_slice.jpg
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFBool voxelOutline;

  /**
   * When #voxelOutline is TRUE, this threshold specifies
   * the minimum size of a voxel, in screen pixels, for the voxel outline to be visible.
   * Default is 2, meaning that outline is visible only if a single voxel is
   * bigger than 2 screen pixels.
   *
   * @FIELD_SINCE_OIV 9.2
   */
  SoSFFloat voxelOutlineThreshold;

  /**
   * When #voxelOutline is TRUE, this value specifies the voxel outline width in pixels.
   * Default is 2 pixels.
   *
   * @FIELD_SINCE_OIV 9.6
   */
  SoSFFloat voxelOutlineWidth;

  /**
  * When #voxelOutline is TRUE, this value specifies the voxel outline color.
  * Default is black : (0, 0, 0).
  *
  * @FIELD_SINCE_OIV 9.6
  */
  SoSFColor voxelOutlineColor;

  /** 
  * If true, apply an ambient occlusion effect (default is FALSE).
  * Ambient occlusion is an shading effect that approximates
  * attenuation of light due to neighboring voxels. It works best when the volume
  * data contains surfaces (region boundaries with relatively sharp gradients),
  * or in voxelized rendering mode, and those voxels are opaque.
  *
  * If you use ambientOcclusion, you should set the SoVolumeRender::samplingAlignment field to BOUNDARY_ALIGNED
  * to reduce "slicing" artifacts.
  *
  * Notes:
  * - This effect is only applied on voxels that are considered to be "solid" (non-transparent). @BR
  *   The "solid" transparency threshold is controlled by the value of SoVolumeRender::opacityThreshold.
  *
  * @FIELD_SINCE_OIV 9.1
  */
  SoSFBool ambientOcclusion;

  /** 
  * Enable screen space lighting (computed on the GPU).
  * Default is TRUE.
  * Deferred lighting is computed based on the final image depth buffer instead using the data gradients.
  * It is much faster than gradient based lighting (see the #lighting field) and supports multiple
  * light sources (up to 8). Because it is not based on data gradients, it does not have problems with small/random
  * gradients and it gives a much better result when using clipping nodes like SoVolumeClippingGroup or
  * SoUniformGridClipping.  Light sources are defined by Open Inventor SoDirectionalLight nodes.
  *
  * Notes:
  * - Since Open Inventor 9.3, deferred lighting supports up to 8 light sources.
  *
  * - Both deferred lighting and gradient lighting may be enabled at the same time, but the intent
  *   is that the application should use one or the other.
  *
  * - Gradient related fields, e.g. surfaceScalarExponent, have no effect on deferred lighting.
  *
  * - In any case, do not use the SoVolumeRender::lighting field.
  * 
  * - The lighting model used by deferred lighting is the Phong reflection model.
  *   Its parameters are defined by the current SoMaterial on state.
  *
  * - This effect is only applied on voxels that are considered to be "solid" (non-transparent). @BR
  *   The "solid" transparency threshold is controlled by the value of SoVolumeRender::opacityThreshold.
  *
  * @B LIMITATIONS @b
  * - Only directional lights (SoDirectionalLight) are supported.
  *
  * - If you use deferredLighting, you should set the SoVolumeRender::samplingAlignment field to BOUNDARY_ALIGNED
  *   or SMOOTH_BOUNDARY_ALIGNED to reduce "slicing" artifacts.
  *
  * - The light color is taken into account, but not the light intensity field.
  *   You can get the same effect as reduced intensity by reducing the light color values.
  *
  * - Deferred lighting works best when
  *     - The volume data can be considered to contain "surfaces", for example bones (etc)
  *       in medical data or metal parts in NDT data, and
  *     - The transfer function makes the surface material
  *       relatively opaque and the surrounding material relatively transparent.
  *
  * @FIELD_SINCE_OIV 9.2
  */
  SoSFBool deferredLighting;

#if SoDEPRECATED_BEGIN(9300)

  /**
   * Enable cubic interpolation of data values. @BR
   * @warning Heavy GPU usage. Primarily useful for still image rendering.
   * Default is FALSE.
   * @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoDEPRECATED_MEMBER_NOWARN(9300,"Use SoVolumeShape::interpolation field instead.")
  SoSFBool cubicInterpolation;

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(10000)

  /** Available Lighting Model mode */
  enum SoDEPRECATED_ENUM(10000,"No longer used. OPENGL mode is always used.") LightingModel
  {
    OIV6,
    OPENGL
  };

  /**
   * Sets the lighting model.
   * For backward compatibility with Open Inventor v6.  Applications should set this field to OPENGL.
   * Only affects gradient based lighting (#lighting field).
   *
   * @useenum{LightingModel}. Default is OIV6. @BR
   * @FIELD_SINCE_OIV 7.0
   */
  SoDEPRECATED_MEMBER_NOWARN(10000,"No longer used. OPENGL mode is always used.")
  SoSFEnum lightingModel;

#endif /** @DEPRECATED_END */

#ifndef HIDDEN_FROM_DOC
  SoINTERNAL public:

  /** Shadowing techniques */
  enum ShadowingMethod
  {
    /** The standard shadowmap-based shadows. See SoShadowGroup::method. */
    SHADOW_MAP,

    /**
     * Shadows using a raycasting based algorithm.
     * This type provides high quality shadows, but can have a big impact on performance.
     * It supports SoROI clipping, but does not support SoClipPlane, SoVolumeClippingGroup,
     * SoUniformGridClipping or SoUniformGridProjectionClipping.
     */
    RAYCASTING,
  };

  /**
   * Shadowing method used for shadows. @BR
   * Default is SHADOW_MAP. @BR
   */
  SoSFEnum shadowingMethod;

  /** Radius of Ambient Occlusion, in voxel number */
  SoSFFloat AORadius;
  /** Number of sample used for AO */
  SoSFFloat AONumSample;
  /** Size of AO kernel (see SoScreenSpaceAmbientOcclusion for details) */
  SoSFFloat AOKernelSize;
  /** Use normal to compute AO */
  SoSFBool AOUseNormal;
  /** Show AO */
  SoSFBool AOShowAO;
  /** Show surface normal */
  SoSFBool AOShowNormal;

  void GLRender(SoGLRenderAction *action);
  virtual void doAction(SoAction *action);
  virtual void notify(SoNotList* list);

  // Initializes the classes.
  static void initClass();
  static void exitClass();

  /** Return gradient quality according to SoInteractiveComplexity */
  GradientQuality getGradientMethod(SoState* state) const;

  /** Return edgeDetect2D field value according to SoInteractiveComplexity */
  SbBool hasEdgeDetect2d(SoState* state) const;

  /** Return ambient occlusion field value according to SoInteractiveComplexity */
  SbBool hasAmbientOcclusion(SoState* state) const;

  /** Return deferred light field value according to SoInteractiveComplexity */
  SbBool hasDeferredLighting(SoState* state) const;

  /** Return cubic interpolation field value according to SoInteractiveComplexity */
  SbBool hasCubicInterpolation(SoState* state) const;

  /** Return boundaryOpacity field value according to SoInteractiveComplexity */
  SbBool hasBoundaryOpacity(SoState* state) const;

  /** Install textures needed for shaders */
  virtual void installTextures(SoGLRenderAction* action);
  
  /** @copydoc SoVolumeShader::isRaycastingEnabled */
  virtual SbBool isRaycastingEnabled(SoState* state) const;

  /** Reserve needed texture units */
  virtual void allocateTextureUnit(SoGLRenderAction *action) const;

  /** Return TRUE if render mode technique is Voxelized  */
  SbBool isVoxelizedRenderingEnabled(SoState* state) const;

  /** Return true if interpolation on border must be enabled */
  virtual bool isInterpolationActive(SoGLRenderAction* action);

  /** Return true if colorInterpolation must be enabled */
  virtual bool isColorInterpolationEnabled(SoState* state);

  /** Return TRUE if preIntegration is enabled  */
  bool isPreintegratedEnabled(SoState* state) const;

  /** Set whether or not we want to reset the physical quality element to NULL when traversing this node. */
  void setRemovePhysicalQuality( bool removePhysicalQuality );

  /** Sets the current render mode used by the VolumeRender node */
  void setRenderMode( SoVolumeRender::RenderMode renderMode ) { m_renderMode = renderMode; }
protected:

  virtual ~SoVolumeRenderingQuality();

#endif

private:

  static const float DEFAULT_INNER_EDGE_THRESHOLD;
  static const float DEFAULT_OUTTER_EDGE_THRESHOLD;

  /** Disable edge2D if not supported */
  void checkAndUpdateEdge2DSupport(SoGLRenderAction* action);

  /** Return edgeColoring field value according to SoInteractiveComplexity */
  SbBool hasEdgeColoring(SoState* state) const;

  /**
   * Put edgeDetect2DInnerThreshold and edgeDetect2DOuterThreshold
   * between 0. and 1. if needed
   */
  void clampEdgeDetect2dFields();

  SoNodeDependencies* m_transferFunctionCache;

  /**
   * This group store internal scene used to render pre-integrated texture
   */
  SoRef<SoGroup> m_preIntegratedScene;
  SoGroup* m_preIntegratedTexPlaceHolder;
  SoTextureUnit* m_preIntegratedTexUnit;

  /** True if we want to set the physical quality element to NULL when traversing this node. */
  bool m_removePhysicalQuality;

  /** Current VolumeRender render mode */
  SoVolumeRender::RenderMode m_renderMode;
};
/*----------------------------------------------------------------------------*/

#endif /*_SO_VOLUME_QUALITY_H_*/


