/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef  _SO_VOLUME_SKIN_
#define  _SO_VOLUME_SKIN_

#include <Inventor/nodes/SoShape.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/SbBox.h>

#include <VolumeViz/nodes/SoSlice.h>

class SoVertexShader;
class SoVolumeSkinLdm;
class SoVolumeSkinInterface;
class SoVolumeStateSkin;
class SoGetBoundingBoxAction;

/**
 * @VREXT Draws a data volume skin.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 *   This node draws the "skin" (outer faces) of the data volume defined by
 *   the current SoVolumeData node.  This skin is a set of textured polygons
 *   similar to ortho slices.
 *
 *   The current SoVolumeData node can be specified with #dataSetId. When this field is set to 0,
 *   the last SoVolumeData node on state is used.
 *
 *   For a non-RGBA (scalar valued) volume,
 *   each voxel's RGBA value is determined by the current SoDataRange and SoTransferFunction.
 *   The current @B diffuse color @b and @B transparency @b (set, for example, with
 *   an SoMaterial node) modify the appearance of the voxels.  This means that, for
 *   example, the current transparency can be used as a @I global alpha @i value
 *   to modulate the overall opacity.
 *   For an RGBA volume each voxel's RGBA value comes directly from the volume data.
 *
 *   When using an opaque transfer function (no alpha values < 1), this node
 *   generates the same image as SoVolumeRender, but with much higher
 *   performance, because only slices are rendered.
 *   A common pattern is to put an SoVolumeRender node and an SoVolumeSkin node
 *   under an SoSwitch node.  This makes it easy to switch between skin rendering
 *   and volume rendering.
 *
 *   The #interpolation field controls how the texture is interpolated.
 *
 *   Optionally a bump mapping effect may be applied. Normal vectors are automatically
 *   computed from the data value gradient.  The #enableBumpMapping and #bumpScale
 *   fields control whether bump mapping is active and the intensity of the effect.
 *
 *   The entire skin is pickable, even where it is transparent as a result of the
 *   current transfer function.  The SoVolumeSkinDetail class allows you to get the
 *   voxel position and value after picking.
 *
 * @B Notes: @b
 *   - Drawing position: @BR
 *     An SoVolumeSkin is approximately the same as six ortho slices, but not exactly.
 *     The difference is that each face of the skin is drawn at the outer edge of the voxel
 *     slice it intersects. So if there is no SoROI, or the ROI is identical to the volume
 *     dimensions, the volume skin faces will be drawn at the limits of the volume extent
 *     (bounding box).
 *     For SoOrthoSlice, the geometry passes through the @I center@i of the voxels in the specified
 *     slice. So an ortho slice with sliceNumber = 0 will be drawn one-half
 *     voxel size (on the slice axis) in from the bounding box of the volume. 
 *
 *   - Transformation matrices: @BR
 *     The volume size and orientation (like geometry) can be modified by
 *     transformation nodes in the scene graph and this in turn modifies the
 *     appearance of volume rendering nodes like SoVolumeSkin.
 *     However the same transformation must be applied to the volume data node
 *     and all volume rendering nodes associated with that volume.  So effectively
 *     any transformation nodes that affect the volume must be placed @B before @b
 *     the volume data node.
 *
 *   - Picking: @BR
 *     The entire skin is pickable, even where it is transparent as a result of the
 *     current transfer function.  The SoVolumeSkinDetail class allows you to get the
 *     voxel position and value after picking.
 *
 *   - Clipping: @BR
 *     Volume primitives can be clipped using a region of interest (SoROI), geometry
 *     (SoVolumeClippingGroup) and/or height fields (SoUniformGridClipping).
 *     They are also clipped by OpenGL clipping planes (SoClipPlane), but we recommend
 *     using the VolumeViz clipping nodes instead. @BR @BR
 *     Note that, in general, clipping does not work well with SoVolumeSkin because the
 *     clipped portions of the skin are not "closed", allowing the user to see inside the skin.
 *     However the SoROI node's EXCLUSION_BOX mode does work well and can be used to exclude
 *     a sub-region from the skin, forming what is sometimes called a "chair cut".
 *
 *   - Dragging: @BR
 *     It is possible to interactively translate and resize a skin that is limited to
 *     a subset of the volume (sometimes called a "volume probe").  This is done using
 *     an Open Inventor dragger, e.g. SoTabBoxDragger, to adjust an SoROI node.  This
 *     is such a common operation that Open Inventor provides the SoROIManip node.
 *
 *   - Interpolation: @BR
 *     Interpolation is specified using the #interpolation field. The default (LINEAR)
 *     does bi-linear interpolation between voxel values. The NEAREST value can be used
 *     to display individual voxels. For best image quality we recommend using the
 *     TRILINEAR or MULTISAMPLE_12 value.
 *
 *   - Material: @BR
 *     The color of each voxel is modulated by the current diffuse color in the traversal
 *     state. The default diffuse color is 0.8,0.8,0.8. This results in full intensity
 *     values in the color map being displayed as 80% intensity. Therefore we recommend
 *     adding an SoMaterial node before the skin and setting its diffuseColor field to
 *     full white (1,1,1).
 *
 *   - Transparency: @BR
 *     Typically the color map (SoTransferFunction) used for volume rendering (SoVolumeRender)
 *     assigns transparency (alpha < 1) to some voxel values.  If you want to use the same
 *     color map for skin rendering, but render the slice completely opaque, set the #alphaUse
 *     field to ALPHA_OPAQUE. This overrides the alpha values in the color map (or an RGBA
 *     volume). However it does not affect transparency assigned using an SoMaterial node.
 *
 *   - Voxel edges: @BR
 *     The edges of the voxels can also be rendered.
 *     See options in the SoVolumeRenderingQuality node.
 *
 *   - Custom shaders: @BR
 *     The current SoVolumeShader node, if any, allows custom shaders to be defined for
 *     special computation or rendering effects, including blending multiple volumes.
 *
 *   - Composition with Multiple Data: @BR
 *     It is possible to compose datasets that have different dimensions,
 *     tile sizes and transformations. @BR
 *     In order to help fetch the correct data values in custom shaders,
 *     texture coordinates conversion functions are provided in the
 *     @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *     For instance,
 *     \code
 *     vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *     \endcode
 *     can be used to convert texture coordinates related to one dataset to
 *     texture coordinates related to another dataset. @BR
 *     The conversion is based solely on the transformations applied to each
 *     dataset, which are defined by their model matrix and their extent. @BR
 *     Please note that the model matrix of a dataset is defined by to the
 *     SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *     the order of the traversal.
 *
 *   - Large Slice mode: @BR
 *     When the "large slice" mode is enabled (see SoSlice::largeSliceSupport), if all the
 *     required full resolution tiles have already been loaded, then
 *     the slice data is taken from LDM system memory cache as usual. But if some required 
 *     tiles are not currently in memory, the required slice data will be loaded directly 
 *     from the volume reader @I without @i loading the complete tiles. This reduces disk I/O
 *     and reduces the amount of system memory required to display the slice at full resolution,
 *     so larger (or more) slices can be displayed. The required tiles are then scheduled to be
 *     loaded asynchronously in case adjacent slices are displayed later. 
 *     For example, loading a 1024x1024 SoOrthoSlice from an 8-bit dataset with 128x128x128
 *     tiles would normally require loading 1024x1024x128 bytes of data (as complete tiles). 
 *     With largeSliceSupport enabled, only 1024x1024 bytes (maximum) of data need to
 *     be loaded (in the worst case where no high resolution data is currently in memory).
 *
 *   - Performance: @BR
 *     - Tile size: @BR
 *       For backward compatibility, the default tile size is still only 64. This is quite
 *       small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
 *       number of tiles that must be managed by VolumeViz. This overhead can be significant,
 *       especially for operations that require reloading the data textures on the GPU, for
 *       example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
 *       it can be efficient to set the tile size large enough to contain the entire volume.
 *       For very large volumes, larger tile sizes are efficient for SoVolumeRender but
 *       somewhat inefficient for slice rendering because complete tiles must be loaded
 *       even though the slice only uses part of the data. Applications should experiment.
 *       @BR
 *       For volumes stored in LDM file format, the tile size must be specified when the
 *       volume is converted to LDM (see SoConverter and the "-t" option).  For other data
 *       data formats the tile size can be specified using the SoVolumeData node's
 *       @I ldmResourceParameters@i field, but only after setting the @I filename@i field
 *       or calling the \if_dotnet SetReader() \else setReader() \endif method.
 *
 *     - Tile cache policy:  
 *       It specifies how the tiles are stored in CPU memory. The selected policy
 *       can significantly impact the data loading performance versus the CPU memory footprint.
 *       See SoLDMResourceParameters::tileCachePolicy for detail.
 *
 *     - Compressed textures: @BR
 *       For performance reasons, SoVolumeSkin accumulates small textures into a bigger
 *       one. When using compressed RGBA textures (via #SoDataSet's field useCompressedTexture),
 *       this optimization cannot be done.
 *       If you want to favor performance rather than memory usage, you should disable compression
 *       (enabled by default if supported by the graphic card)
 *
 * @EXAMPLE
 *   For simple data sets, a basic VolumeViz rendering could be achieved
 *   with only a few nodes: minimally an SoVolumeData node to identify
 *   the data set and one rendering node.  However most data sets need
 *   at least some of the additional nodes shown here in order to get
 *   a correct and useful rendering. Most applications will need
 *   additional nodes to take advantage of region of interest, interaction,
 *   clipping and other VolumeViz features.  Please consider
 *   the code shown here as simply a guideline and a starting point for
 *   exploring the many powerful features available in Open Inventor.
 *   
 *   Note that some of the property nodes (data, material, color map,
 *   etc) will typically be shared by multiple rendering nodes.  In other
 *   words the volume usually only needs to be loaded once, using a single
 *   SoVolumeData node, then multiple slices and/or regions can be rendered
 *   using that data node.
 *
 *   Also note that this example is for a data volume, where each voxel can be
 *   considered a discrete sample from a continuous data field and interpolation
 *   should be used to compute values between voxel centers. If you are
 *   rendering a label volume, then each voxel is an "id" assigning that
 *   voxel to a specific material, object, etc.  In this case, set the
 *   @I interpolation@i field to NEAREST to disable interpolation.
 * \if_cpp
 * \code
 *   // Keep volume viz separate from geometry
 *   SoSeparator* volSep = new SoSeparator();
 *     root->addChild( volSep );
 *
 *   // Load volume data
 *   SoVolumeData* volData = new SoVolumeData();
 *     volData->fileName = "$OIVHOME/examples/data/VolumeViz/Colt.vol";
 *     volSep->addChild( volData );
 *
 *   // Set range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange* volRange = new SoDataRange();
 *     if (volData->getDatumSize() > 1) {
 *       double minVal, maxVal;
 *       volData->getMinMax( minVal, maxVal );
 *       volRange->min = minVal;
 *       volRange->max = maxVal;
 *     }
 *     volSep->addChild( volRange );
 *
 *   // Load opaque color map
 *   SoTransferFunction* volTF = new SoTransferFunction();
 *     volTF->predefColorMap = SoTransferFunction::BLUE_WHITE_RED;
 *     volSep->addChild( volTF );
 *
 *   // Display volume at full intensity
 *   SoMaterial* volMat = new SoMaterial();
 *     volMat->diffuseColor.setValue( 1, 1, 1 );
 *     volSep->addChild( volMat );
 *
 *   // Remove tile boundary artifacts while moving. 
 *   SoVolumeShader* volShader = new SoVolumeShader();
 *     volShader->interpolateOnMove = TRUE;
 *     volSep->addChild( volShader );
 *
 *   // Display volume skin
 *   SoVolumeSkin* volSkin = new SoVolumeSkin();
 *     volSep->addChild( volSkin );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   // Keep volume viz separate from geometry
 *   SoSeparator volSep = new SoSeparator();
 *     root.AddChild( volSep );
 *
 *   // Load volume data
 *   SoVolumeData volData = new SoVolumeData();
 *     volData.fileName.Value = "$OIVNETHOME/src/demos/data/VolumeViz/Colt.vol";
 *     volSep.AddChild( volData );
 *
 *   // Set range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange volRange = new SoDataRange();
 *   if (volData.GetDatumSize() > 1)
 *   {
 *       double minVal, maxVal;
 *       volData.GetMinMax( out minVal, out maxVal);
 *       volRange.min.Value = minVal;
 *       volRange.max.Value = maxVal;
 *   }
 *   volSep.AddChild( volRange );
 *
 *   // Load opaque color map
 *   SoTransferFunction volTF = new SoTransferFunction();
 *     volTF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.BLUE_WHITE_RED;
 *     volSep.AddChild( volTF );
 *
 *   // Display volume at full intensity
 *   SoMaterial volMat = new SoMaterial();
 *     volMat.diffuseColor.SetValue(1, 1, 1);
 *     volSep.AddChild( volMat );
 *
 *   // Remove tile boundary artifacts while moving. 
 *   SoVolumeShader volShader = new SoVolumeShader();
 *     volShader.interpolateOnMove.Value = true;
 *     volSep.AddChild( volShader );
 *
 *   // Display volume skin
 *   SoVolumeSkin volSkin = new SoVolumeSkin();
 *     volSep.AddChild( volSkin );
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Keep volume viz separate from geometry
 *   SoSeparator volSep = new SoSeparator();
 *     root.addChild(volSep);
 *
 *   // Load volume data
 *   SoVolumeData volData = new SoVolumeData();
 *     volData.fileName.setValue( "$OIVJHOME/data/VolumeViz/Colt.vol" );
 *     volData.ldmResourceParameters.getValue().tileDimension.setValue(128,128,128);
 *     volSep.addChild( volData );
 *
 *   // Set range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange volRange = new SoDataRange();
 *     if (volData.getDatumSize() > 1) {
 *       double[] minmax;
 *       minmax = volData.getDoubleMinMax();
 *       volRange.min.setValue( minmax[0] );
 *       volRange.max.setValue( minmax[1] );
 *     }
 *     volSep.addChild( volRange );
 *
 *   // Load opaque seismic color map
 *   SoTransferFunction volTF = new SoTransferFunction();
 *     volTF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.BLUE_WHITE_RED );
 *     volSep.addChild( volTF );
 *
 *   // Display volume at full intensity
 *   SoMaterial volMat = new SoMaterial();
 *     volMat.diffuseColor.setValue( 1, 1, 1 );
 *     volSep.addChild( volMat );
 *     
 *   // Remove tile boundary artifacts while moving.
 *   SoVolumeShader volShader = new SoVolumeShader();
 *     volShader.interpolateOnMove.setValue( true );
 *     volSep.addChild( volShader );
 *
 *   // Display volume skin
 *   SoVolumeSkin volSkin = new SoVolumeSkin();
 *     volSep.addChild( volSkin );
 * \endcode
 * \endif
 *
 * @FILE_FORMAT_DEFAULT
 *    VolumeSkin {
 *    @TABLE_FILE_FORMAT
 *       @TR dataSetId          @TD 0
 *       @TR interpolation      @TD LINEAR
 *       @TR alphaUse           @TD ALPHA_AS_IS
 *       @TR useRGBA            @TD FALSE
 *       @TR enableBumpMapping  @TD FALSE
 *       @TR bumpScale          @TD 1.0
 *       @TR faceMode           @TD FRONT
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    @B SoGLRenderAction @b @BR
 *       Draws a volume-rendered image based on current SoVolumeData.
 *
 *    @B SoGetBoundingBoxAction @b @BR
 *       Computes the bounding box that encloses the volume.
 *
 * @SEE_ALSO
 *    SoVolumeData,
 *    SoTransferFunction,
 *    SoROI,
 *    SoVolumeShader
 *
 *
 */
class VOLUMEVIZ_API SoVolumeSkin : public SoSlice
{
  SO_NODE_HEADER( SoVolumeSkin );

public:

  /**
   * Use this enum with the #faceMode field.
   */
  enum FaceMode
  {
    /** Display only front faces */
    FRONT,
    /** Display only back faces */
    BACK,
    /** Display front and back faces */
    FRONT_AND_BACK
  };

  /**
   * Constructor.
   */
  SoVolumeSkin();

  /**
   * Specifies the SoVolumeData node to use.
   *
   * This is useful when datasets of different dimensions are present in the scene graph.
   * Please see SoMultiDataSeparator for more details.
   *
   * When set to 0, the last SoVolumeData node on state is used.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 10.11.0
   */
  SoSFInt32 dataSetId;

  /**
   * Display front (facing the camera) faces or back faces. 
   * @useenum{FaceMode}. Default is FRONT.
   * Displaying only front (or back) faces reduces the amount of data
   * that must be loaded and reduces the rendering time.
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFEnum faceMode;

SoEXTENDER public:

  /**
   * Returns an alternate represention for this node. @BR
   * This is typically a scene graph that gives approximately the same
   * appearance using only core Open Inventor nodes.  For example the
   * volume skin could be represented using SoTexture2 nodes.
   * This class implements an alternate for PDF and U3D export.
   */
  virtual SoNode* getAlternateRep( SoAction* action );

  /** @copydoc SoSlice::GLRender */
  virtual void GLRender(SoGLRenderAction* action);

  /**
   * Compute bounding box
   */
  void computeBBox(SoAction *, SbBox3f &box, SbVec3f &center);

SoINTERNAL public:
  /**
   * Node init
   */
  static void initClass();
  /**
   * Node exit
   */
  static void exitClass();

  /**
   * Create a subgraph for the alternateRep field that gives approximately
   * the same appearance, but using only core Open Inventor nodes.
   * Might be useful in other cases.
   */
  virtual SoNode* createAlternateRep( SoAction *action );

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  /**
   * Returns true if the shape intersects the given tile
   */
  virtual bool intersect( SoLDMNodeFrontManager* nfm, const SoLDMTileID& tileId );

private:
  /**
   * Create and set the good rendering class
   */
  bool setupRenderInterface(SoAction* action);

  /** Interface for LDM mode */
  SoVolumeSkinLdm* m_volSkinLdm;

  /** Current active render Interface*/
  SoVolumeSkinInterface* m_volSkinInterface;

protected:
  /**
   * Destructor
   */
  virtual ~SoVolumeSkin();

  /**
   * Performs GL rendering
   */
  virtual void doRendering(SoGLRenderAction *action);

  /**
   * Generate triangles
   */
  void generatePrimitives(SoAction *action);

  /**
   * Overrides standard method to create an SoVolumeSkin instance
   * representing a picked intersection with a triangle that is half
   *  of the face of a cube.
   */
  SoDetail *createTriangleDetail(SoRayPickAction *,
                                 const SoPrimitiveVertex *v1,
                                 const SoPrimitiveVertex *,
                                 const SoPrimitiveVertex *,
                                 SoPickedPoint *);

  SoVolumeStateSkin* m_vs;

  friend class SoVolumeSkinLdm;
};
#endif // _SO_VOLUME_SKIN_


