/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : L.Taboni (Apr 2004)
**=======================================================================*/
#ifndef _SO_VR_DICOM_DATA_
#define _SO_VR_DICOM_DATA_

#include <Inventor/SbLinear.h>
#include <Inventor/SbString.h>
#include <Inventor/SbBox.h>
#include <Inventor/SbPImpl.h>

#include <VolumeViz/readers/dicom/SiDicomDataSet.h>
#include <VolumeViz/readers/dicom/SiDicomElement.h>

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoVRDicomData);

/**
 * @VREXT DICOM data.
 *
 * @ingroup VolumeVizReaders
 *
 * @DESCRIPTION
 *  Access data in a DICOM data set.
 *
 *  DICOM is a widely used format for storing medical image data (CT, MRI, etc),
 *  defined by the National Electrical Manufacturers Association (NEMA)
 *  (http://medical.nema.org).
 *
 *  This class can be used to directly open and query information from DICOM
 *  format files.  It can also be used to get DICOM specific information from
 *  a file opened by VolumeViz (see SoVRDicomFileReader).
 *
 *  Note that it @I only@i returns information for a single file, not (in general)
 *  for the complete volume.
 *
 *  To conveniently iterate over or randomly access all the DICOM attributes,
 *  see the getDicomDataSet() method.
 *
 * @SEE_ALSO
 *    SoVRDicomFileReader,
 *    SiDicomDataSet
 */
class VOLUMEVIZ_API SoVRDicomData
{
  SO_PIMPL_BASE_PUBLIC_HEADER(SoVRDicomData);
public:

  /** Constructor. */
  SoVRDicomData ();

  /** Destructor. */
  ~SoVRDicomData ();

  // header & image

  /** Reads the DICOM file header. Returns TRUE upon success.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE SbBool readDicomHeader (const char *fileName);

  /** Reads the DICOM file header. Returns TRUE upon success.
   */
  SbBool readDicomHeader (const SbString& fileName);

  /** Reads the DICOM image. Returns TRUE upon success.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE SbBool readDicomImage (const char *fileName);

  /** Reads the DICOM image. Returns TRUE upon success.
   */
  SbBool readDicomImage (const SbString& fileName);

  /**
   * Closes the underlying reader.
   */
  SbBool close();

  // info from header

  /** Returns the number of frames.
   *  Currently an SoVRDicomData object corresponds to a single DICOM file.
   *  Therefore getNumFrames() returns the number of images in the associated
   *  DICOM file, @I not@i (in general) the number of images in the volume.
   *  Typically it returns 1 because there is a single image in the
   *  file.  Note that even if the SoVRDicomData object was queried from an
   *  SoVRDicomFileReader, it does @I not@i return the number of images in the
   *  volume except in the special case where all images are stored in a single file.
   *  To get the number of images in the volume use the volume reader or SoVolumeData object.
   */
  int getNumFrames () const;

  /** Returns the image width. */
  int getImageWidth () const;

  /** Returns the image height. */
  int getImageHeight () const;

  /** Returns the image size in bytes. */
  int getSizeInBytes () const;

  /** Returns the number of bytes per pixel. */
  int getBytesPerPixel () const;

  /** Returns the number of samples per pixel. */
  int getSamplesPerPixel () const;

  /** Returns true if the transfer syntax is supported by Open Inventor. */
  SbBool supportedTransferSyntax () const;

  /**
   * Read a region of a DICOM frame.
   *
   * @param frame  The frame to read. Note there is typically only one frame per file (see getNumFrames).
   * @param subImg A region of the frame to read (see getImageWidth() and getImageHeight()).
   * @param dst    A buffer that will be filled with image data (see getSizeInBytes()).
   *
   * NOTE: The application is responsible for allocating and freeing this memory.
   */
  void readFrame( size_t frame, const SbBox2i32& subImg, uint8_t* dst ) const;

  /**
   * Returns a DICOM data object that allows accessing all the DICOM attributes.
   *
   * @return a pointer to the created data set, that must be deleted by the application.
   *         If an error occurs, returns NULL.
   */
  const SiDicomDataSet* getDicomDataSet() const;

  // ACQUISITION info

  /**
   * Returns the slice thickness in mm.
   *
   * If the thickness information is not present in the file, will return NaN.
   */
  float getSliceThicknessMM () const;

  /** Returns the slice spacing. */
  float getSliceSpacing () const;

  // RELATION info

  /** Returns the series number. */
  int getSeriesNumber () const;

  /** Returns the image number. */
  int getImageNumber () const;

  /** Returns the image location in mm.
   *  This is the DICOM ImagePosition transformed by the DICOM ImageOrientation.
   */
  float getSliceLocation () const;

  // IMAGE info

  /** Returns the X pixel size in mm. */
  float getXPixelSize () const;

  /** Returns the Y pixel size in mm. */
  float getYPixelSize () const ;

  /**
   * Returns the 3D position of the center of the image.
   * This position represents the "origin" of the image in a 3D graphics sense
   * and be used with a transform node, for example, SoTranslation.
   * It is computed from the DICOM Image Position Patient attribute, but is @I not@i
   * the same as the Image Position Patient (which is the position of the center of
   * the first voxel). If this element is not available, position will be set to (0, 0, 0).
   * @return True if file contains position
   */
  SbBool getPosition(SbVec3f& position) const;

  /**
   * Returns the position of the first frame's top left voxel center in the patient space.
   * This correspond to Dicom tag ImagePositionPatient (0020, 0032) of image stack first slice.
   */
  SbVec3f getImagePosition() const;

  /**
   * Returns the orientation of the DICOM volume.
   * This orientation can be used with a transform node, for example SoRotation
   * It is computed from the DICOM Image Orientation Patient.  If this element is not
   * available, orientation will be set to the identity matrix.
   * @return True if file contains orientation information.
   */
  SbBool getOrientation(SbMatrix& orientation) const;

  /** Returns the number of bits stored. */
  int getBitsStored () const;

  /** Returns the high bit. */
  int getHighBit () const;

  /** Returns the signed flag. */
  int signedData () const;

  /** Returns the intercept. */
  float getIntercept () const;

  /** Returns the slope. */
  float getSlope () const;

  /**
   * Generic method to get DICOM info.
   * The group and element parameters should be given using the hexadecimal tag values
   * defined by the DICOM specification.
   * For example Rescale Slope is (0028,1053) and Rescale Intercept is (0028,1052).
   * - See http://medical.nema.org/
   * - See http://www.sno.phy.queensu.ca/~phil/exiftool/TagNames/DICOM.html
   *
   * To conveniently access the DICOM attributes, see the getDicomDataSet() method.
   *
   * @param group hexadecimal group tag value to retrieve info from.
   * @param element hexadecimal element tag value to retrieve info from in the group.
   * @return value info.
   */
  SbString getDicomInfo( unsigned short group, unsigned short element) const;

private:
  // The copying of SoVRDicomData is disabled because the copy will hold the
  // same ImageReader, that is not a correct behavior and can cause data races.
  SoVRDicomData( const SoVRDicomData& other );
  SoVRDicomData& operator=( const SoVRDicomData& other );

  friend class SoVRDicomFileReader;
};

#endif /*_SO_VR_DICOM_DATA_*/
