/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : L.Taboni (Apr 2004)
**=======================================================================*/
#ifndef  _SO_VR_DICOM_FILE_READER_
#define  _SO_VR_DICOM_FILE_READER_

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/readers/SoVRDicomData.h>
#include <VolumeViz/nodes/SoVolumeData.h>

class SiDicomSliceVector;
class SoDicomSliceCache;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoVRDicomFileReader);

/**
 * @VREXT DICOM file reader.
 * 
 * @ingroup VolumeVizReaders
 * 
 * @DESCRIPTION
 * 
 * Volume reader for the DICOM file format. 
 * VolumeViz automatically selects this reader if a filename is given and the file
 * extension is ".dc3", ".dic", ".dcm", or ".dicom".
 * 
 *  DICOM is a widely used format for storing medical image data (CT, MRI, etc),
 *  defined by the National Electrical Manufacturers Association (NEMA)
 *  (http://medical.nema.org).
 *
 * SoVRDicomFileReader can load a volume from a single DICOM file or from a list of
 * DICOM files (stack of images). Loading a volume from a single DICOM file is the
 * same as loading any other format supported by VolumeViz (see setFilename(const SbString& filename)).
 * Loading a volume from a list of DICOM files can be done by:
 * - Using a list file. See setFilename(const SbString& filename).
 *
 * - Specifying an explicit list of filenames.  See setFilename(const SbString& filename).
 *
 * - Specifying a list of filenames by simple pattern.  See setFilenamePattern(const SbString& pattern).
 *
 * - Specifying a list of filenames using a sprintf-like format.  See setFilenameList(const SbString&, const int, const int, const int).
 *
 * - Specifying a file and loading all images that are part of the same "series". See setFilenameListFromSeries(const SbString& base).
 *
 * - Specifying a directory and loading all DICOM files in the directory.  See setDirectory(const SbString& folder).
 *
 * Unlike a raster stack, the position of each slice in a DICOM volume is determined by the
 * location value in its file header and not by the order of the file name in the list. Also
 * unlike a raster stack consisting of (for example) JPEG or PNG images, a DICOM file in the
 * list may contain more than one slice of the volume. The reader handles this automatically.
 * The first file in the list is considered the "reference" for the volume and all slices must
 * be compatible with this one. Specifically this means that subsequent files must have the
 * same width, height, data type and so on.
 *
 * Note when using a list file: If the file extension is not ".dc3", ".dic" or ".dicom"
 * VolumeViz will not automatically select the DICOM volume reader. You can either give the
 * list file one of the DICOM file extensions or force VolumeViz to use the DICOM reader by
 * explicitly creating an instance of SoVRDicomFileReader and calling the SoVolumeData method
 * setReader().
 *
 * The volume reader will automatically get the volume dimensions, data type, extent
 * (voxel size/spacing) and number of significant bits from the DICOM file header. The reader
 * will also apply the data adjustment (if any) specified by the RescaleSlope and RescaleIntercept
 * tags in the DICOM file header, i.e.: actualValue = slope * storedValue + intercept. As part of
 * this process the reader will automatically convert unsigned data to the corresponding signed
 * data type if necessary (in other words if the rescale calculation produces negative values).
 * The application can also explicitly specify the volume data type. This allows, for example,
 * converting float data values to more compact integer values. @BR
 * Note: The DICOM reader only uses the rescale slope and intercept values from the first file
 * in the list. It does not currently handle the (less common) case where each file contains different rescale values.
 *
 * Photometric Interpretation:
 * - MONOCHROME1 : Mapping of data values to color and opacity is defined by the SoDataRange and
 *   SoTransferFunction nodes. The application will normally use the DICOM Window Center and Window Width
 *   values to set the SoDataRange. For SoTransferFunction the application can use the predefined light
 *   to dark color map INTENSITY_REVERSED for displaying single images.
 *
 * - MONOCHROME2 : Mapping of data values to color and opacity is defined by the SoDataRange and
 *   SoTransferFunction nodes. The application will normally use the DICOM Window Center and Window Width
 *   values to set the SoDataRange. For SoTransferFunction the application can use the predefined dark to light
 *   color map INTENSITY for displaying single images.
 *
 * - PALETTE COLOR : The reader converts the data values to RGBA values using the palette lookup tables in
 *   the DICOM file and alpha = 1 (opaque). SoDataRange and SoTransferFunction have no effect on an RGBA volume.
 *
 * - RGB : The reader converts the RGB values to RGBA values with alpha = 1 (opaque).
 *   SoDataRange and SoTransferFunction have no effect on an RGBA volume.
 *
 * Volume extent:
 *
 * The volume reader will compute the volume's width in X and height in Y from the volume dimensions
 * and the voxel size.  For example, if the volume dimension in X is 512 (voxels) and
 * the voxel size in X is 0.408 mm, then the volume width is (approximately) 209 mm.  The
 * reader will compute the volume's depth in Z from the min and max slice locations specified
 * in the file header(s).
 * Note that for a DICOM data set, the volume 'position' is considered to be the @I center@i
 * of the first voxel, however VolumeViz considers the volume extent (SoVolumeData::extent) to include
 * @I all@i of the first and last voxels.  Therefore the extent 'min' is alwaya the @I outside corner@i
 * of the first voxel.
 *
 * \par
 * @B NOTE@b: By default, the reader does @I not@i consider the DICOM volume origin (Image 
 * Position Patient attribute) when setting the volume extent. The volume extent set by the
 * reader is always centered at 0,0,0 in 3D world coordinates. There are two options to get the
 * volume correctly positioned in 3D space:
 * - Call the MedicalHelper method dicomAdjustVolume() after setting the files to be loaded. (Recommended)
 * - Set the preference variable VOLUMEVIZ_CENTERED_DICOM to false (default is true).
 *
 * Data Range:
 *
 * Use an SoDataRange node to specify the actual (or desired) range of data values to be mapped.
 * For example, a typical initial data range for DICOM data calibrated in Hounsfield units might be -1000 to 3000.
 *
 * \par
 * @B NOTE@b: By default, VolumeViz does @I not@i use the Window Center and Window Width values
 * (if any) in the DICOM image file. Instead it maps the entire range of the voxel's data type
 * (e.g. 0..65535 for unsigned short) into the colormap. There are two options to use the data
 * range specified in the image file:
 * - Call the MedicalHelper method dicomAdjustDataRange(). (Recommended)
 * - Query the Window Center and Window Width values using the MedicalHelper method getWindowCenterWidth().
 * - Query the Window Center (0028,1050) and Window Width (0028,1051) values using using an SoVRDicomData
 *   object (see getDicomData()) or an SiDicomDataSet object (see getDicomDataSet()).
 *
 * Coordinate Systems:
 *
 * In a medical imaging application there are at least three related coordinate systems.
 * - 3D world coordinates: @BR
 *   Also called XYZ coordinates.  This is the Cartesian coordinate system in which rendering objects 
 *   (volumes, geometry), cameras, lights, etc. are positioned.  In Open Inventor this is a right-handed
 *   coordinate system (same as OpenGL).  The default camera creates a view look down the Z axis, i.e.
 *   +Z is toward the user, +X is to the right and +Y is up.
 *
 * - Voxel coordinates: @BR
 *   Also called "IJK" coordinates.  Each voxel's position in the volume can be described by an IJK coordinate.
 *   In VolumeViz this is also a right-handed coordinate system, corresponding to the positive XYZ axes, i.e.
 *   in the default camera's view K is toward the user, I is to the right and J is up.
 *   You can convert between IJK and XYZ coordinates based on the volume dimensions and extent or use the
 *   conversion methods provided in the SoVolumeData class, e.g. voxelToXYZ().
 *
 * - Anatomical coordinates:
 *   The three standard viewing planes are Axial, Coronal and Sagittal.  The Axial plane divides the body into
 *   Superior (head) and Inferior (foot) sections.  The Coronal plane is a vertical plane dividing the body
 *   into Anterior (front) and Posterior (back) sections.  The Sagittal plane divides the body longitudinally
 *   into Right and Left sections.  The orientation of the patient in a medical data set is often specified
 *   using three letters to identify the primary axes, where each letter is one of the section names.  
 *   DICOM volumes typically have LPS (Left, Posterior, Superior) orientation.  In this case VolumeViz
 *   considers that the direction:
 *     - Toward the left side of the body (Left) is the I / +X / Sagittal axis,
 *     - Toward the back of the body (Posterior) is the J / +Y / Coronal axis, and
 *     - Toward the head (Superior) is the K / +Z / Axial axis.
 *
 *  In other words, the default camera view is looking at the top of the head with the body "face down".
 *
 * \par
 * @B NOTE@b: By default Open Inventor does not display DICOM images with the usual orientation.
 *  A convenient way to get the standard DICOM view orientations is to use the MedicalHelper method
 *  orientView(). This method is useful for orienting the camera to view along one of the standard
 *  medical axes (Axial, Coronal or Sagittal).
 *
 *  When viewing a slice, also remember that the volume is always
 *  a 3D object to Open Inventor and you must also set the 'axis' field on the SoOrthoSlice node.
 *  See the dicomImageViewer example in the examples/source/Medical/Input directory.
 *
 * DICOM Attributes
 *
 * All DICOM attributes, including hierarchical data like Structured Reports, can be traversed and
 * queried using SiDicomDataSet (see getDicomDataSet()).  See the DicomSimpleViewer example in the
 * examples/source/VolumeViz/examples directory.
 *
 * @SEE_ALSO
 *    SoVRDicomData, SiDicomDataSet
 */

 class VOLUMEVIZ_API SoVRDicomFileReader : public SoVolumeReader  
 {

   SO_FIELDCONTAINER_HEADER(SoVRDicomFileReader);
   SO_PIMPL_BASE_PUBLIC_HEADER(SoVRDicomFileReader);

 public:
  /** Constructor. */
  SoVRDicomFileReader ();

 /**
  * Specify the name of a DICOM file or a file containing a list of DICOM files to load.
  *
  * Using a list file is the same as loading a stack of images using the SoVRRasterStackReader
  * except the list file should not contain any header. Each line in the list file may be a
  * full path containing directory names or a simple file name. Simple file names are assumed
  * to be in the same directory as the list file. The application can also specify a list of
  * file paths using the setFilenameList() method. This is useful, for example, if the
  * application user is allowed to select a list of files in a file selection dialog.
  * All the specified files should be part of the same volume data set.
  *
  * Returns 0 if successful.
  */
  virtual int setFilename( const SbString& filename );

  /**
  * Specify a pattern of DICOM files to load.
  * @param pattern e.g. "data_directory/image_file*.dcm"
  * @return 0 if successful.
  *
  * @M_SINCE 9.8
  */
  int setFilenamePattern( const SbString& pattern );

  /**
  * Specify a list of DICOM files to load.
  * All the specified files should be part of the same volume data set.
  * Returns 0 if successful.
  *
  * @B Note:@b SbStringList is a list of pointers, not a list of objects, so
  * the destructor for this class does @I not@i free the memory associated with
  * the SbString objects.  The application is responsible for deleting each object
  * in the list.
  *
  * @M_SINCE 9.0
  */
  int setFilenameList(const SbStringList& filenameList);

  /**
   * Specify a list of DICOM files to load, using a sprintf-like format.
   * This will iterate from @B startIndex @b to @B endIndex @b(included) with increment of indexStep.
   * For each step, this will add a file of name f, where f is @B format @b sprintf-ed with the current iteration index.
   * All the specified files should be part of the same volume data set.
   * @param format The format of the file name, which must be an absolute file path. @BR
   * e.g.  "c:/testData/Brain.%04d.dcm", where "%04d" will be replaced by the current index,
   * producing "c:/testData/Brain.0000.dcm" and so on.
   * @param startIndex The index of the first file to add
   * @param endIndex The index of the last file to add
   * @param indexStep the stride to use (e.g. 2 will skip one out of two files, by generating indices 0, 2, 4, ...)
   * @return true if successful.
   *
   * @M_SINCE 9.4.1
   */
  SbBool setFilenameList(const SbString& format, const int startIndex = 0, const int endIndex = INT_MAX, const int indexStep = 1);

  /**
   * Specify a DICOM file and load all image files in the same directory that are
   * part of the same "series" based on the series UID (0x0020,0x000E).
   *
   * @param base The path to a DICOM image file.
   * @return true if successful.
   */
  bool setFilenameListFromSeries( const SbString& base );

  /**
   * Load the DICOM data into contiguous memory if possible.
   * Once the data is loaded, no additional file I/O is necessary and
   * data access calls to the reader, e.g. getSubSlice(), will be fast.
   *
   * \if_cpp
   * Example:
   * \code
   * SoVRDicomFileReader* volReader = new SoVRDicomFileReader();
   * volReader->setFilename( VOLUME_FILENAME );
   * volReader->loadInMemory();
   *
   * SoVolumeData* volData = new SoVolumeData();
   * volData->setReader( *volReader );
   * MedicalHelper::dicomAdjustVolume( volData );
   * \endcode
   * \endif
   * 
   * Notes:
   * - This is a blocking call (does not return until all data is loaded).
   * - It is not currently possible to directly access the data in memory.
   * - It is not currently possible to monitor the progress of the data loading.
   * - It is not currently possible to test if the data loading succeeded.
   * - Do NOT call this method before setting the data source (filename or directory).
   * - Do NOT call this method with an invalid data filename or directory.
   *
   * @M_SINCE 10.3
   */
  void loadInMemory();

  /**
   * Specify a directory containing DICOM files to load.
   * Each file in this folder that has a supported extension (currently
   * ".dc3", ".dic", ".dcm", or ".dicom") will be loaded.
   * If the DICOM files have an unsupported extension, use the #setFilenameList function instead.
   *
   * All the DICOM files in the specified directory should be part of the same
   * volume data set.  If not, the results are undefined.
   * @param folder The folder from which to load files.
   * @return true if successful.
   *
   * @M_SINCE 9.4.1
   */
  SbBool setDirectory(const SbString& folder);

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar (SbBox3f &size, SoDataSet::DataType &type, SbVec3i32 &dim) override;

  /** 
   * Requests that the input be converted (if necessary) to the specified data type.
   * This allows, for instance, conversion of float data to unsigned byte.
   * By default no conversion is done. If @I doChange@i is FALSE no type conversion is done.
   * Always returns TRUE.
   */
  virtual SbBool setOutputDataType( SbBool doChange, SoDataSet::DataType type );


  /**
   * @copydoc SoVolumeReader::getSubSlice(const SbBox2i32&,int,void*)
   */
  virtual void getSubSlice (const SbBox2i32 &subSlice, int sliceNumber, void *data);

  /** 
   * This method is optional. It returns the number of significant bits of the volume data.
   *
   * If it is not implemented, the default return value is 0, meaning the number of bits
   * depends on the data type. See the last parameter of SoVolumeData::setVolumeData().
   * This method is called immediately after #getDataChar().
   */
  int getNumSignificantBits();

  /**
   * Returns the reader type.
   */
  virtual SoVolumeReader::ReaderType getReaderType ()
  {return DICOM;}

  /**
   * Returns an instance of SoVRDicomData for the first image of this volume.
   *
   * Note that the first image of the volume does not necessarily correspond to first file
   * of file list. Images are automatically sorted using the Image Postion Patient attribute.
   *
   * This object can be queried for DICOM specific info, using
   * for example the SoVRDicomData::getDicomInfo() method.
   *
   * To conveniently access the DICOM attributes, see getDicomDataSet() method.
   */
  const SoVRDicomData& getDicomData() const;

  /**
   * Returns a DICOM data object for the specified slice containing all the DICOM attributes.
   * 
   * @param  sliceIdx The index of the slice to extract the dataset from.
   * @return          A pointer to a new dataset object, that must be deleted
   *                  by the application. Returns NULL if an error occured.
   */
  const SiDicomDataSet* getDicomDataSet( const size_t sliceIdx ) const;

  /**
   * Loads the Dicom data of the given slice.
   *
   * To conveniently access the DICOM attributes, see #getDicomDataSet method.
   */
  void loadSliceDicomData( int slice, SoVRDicomData& dicomData ) const;

  /** SoVRDicomFileReader is thread safe */
  virtual SbBool isThreadSafe() const;

  /**
   * Close all file handles currently opened by the reader.
   *
   * @see restoreAllHandles
   */
  virtual void closeAllHandles();

  /**
   * Restore all file handles closed by closeAllHandles method.
   *
   * @see closeAllHandles
   */
  virtual void restoreAllHandles();

 protected:

  /** Destructor. */
  virtual ~SoVRDicomFileReader ();

private:
  /**
   * s_extensions contains the list of extensions the reader will attempt to read.
   * Each of these extensions is used in a call to SoVolumeReader::registerVolumeReaderExtension().
   */
  static std::vector<SbString> s_extensions;
};

#ifdef _WIN32
#pragma warning( pop )
#endif

#endif /* _SO_VR_DICOM_FILE_READER_ */


