/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : G. SEGUIN (Oct 2001)
**=======================================================================*/
#ifndef  _SO_VR_GENERIC_FILE_READER_
#define  _SO_VR_GENERIC_FILE_READER_

#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/nodes/SoVolumeData.h>


/**
 * @VREXT Generic file reader.
 * 
 * @ingroup VolumeVizReaders
 * 
 * @DESCRIPTION
 * 
 * Reader for volume data files containing only binary data values or
 * else a fixed size header followed by binary data values.
 *
 * Many volume data formats that are not directly supported by VolumeViz
 * can be loaded using this reader.
 *
 * The application must know:
 * - The size of the header, if any, in bytes,
 * - The dimensions of the volume, and
 * - The data type (char, unsigned int, ...)
 * 
 * Steps to follow:
 * - Create an instance of SoVRGenericFileReader.
 *    - Call setFilename() to specify the filename.
 *    - Call setDataChar() to specify the dimensions, data type and header size.
 * - Create an instance of SoVolumeData.
 *    - Call setReader() with the instance of SoVRGenericFileReader.
 */
class VOLUMEVIZ_API SoVRGenericFileReader : public SoVolumeReader  
{
  SO_FIELDCONTAINER_HEADER(SoVRGenericFileReader);

 public:
  /** Constructor. */
  SoVRGenericFileReader(); 

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type,  SbVec3i32 &dim ) override
  {
    size = m_size;
    type = m_type;
    dim = m_dim;
    return RD_NO_ERROR;
  };

  /**
   * @copydoc SoVolumeReader::getSubSlice(const SbBox2i32&,int,void*)
   */
  virtual void getSubSlice( const SbBox2i32& subSlice, int sliceNumber, void * data );

  /**
   * Since the format is raw data, it is necessary to specify 
   * the dimension and data size of the volume. 
   * You can also specify a header size to add support for your own file format.
   */
  void setDataChar( const SbBox3f &size, const SoDataSet::DataType &type, const SbVec3i32 &dim, int headerSize = 0) {
    m_size = size;
    m_type = type;
    m_dim = dim;
    m_bytesPerVoxel = SoVolumeData::dataSize( type );
    m_headerSize = headerSize;
  }

  /**
   * Returns the reader type.
   */
  virtual SoVolumeReader::ReaderType getReaderType ()
  {return GENERIC;};

#if SoDEPRECATED_BEGIN(9500)

  /**
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SoDEPRECATED_METHOD(9500, "Use setDataChar( const SbBox3f &size, const SoDataSet::DataType &type, const SbVec3i32 &dim, int headerSize ) instead.")
  void setDataChar( SbBox3f &size, SoDataSet::DataType &type, SbVec3s &smalldim, int headerSize = 0) {
    SbVec3i32 dim(smalldim[0],smalldim[1],smalldim[2]);
    setDataChar( size, type, dim, headerSize );
  }

#endif /** @DEPRECATED_END */

 protected:
  SbBox3f   m_size;
  SbVec3i32 m_dim;
  SoDataSet::DataType m_type;

  int m_headerSize;
};

#endif // _SO_VR_GENERIC_FILE_READER_


