/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. Ognier (Jul 2005)
**=======================================================================*/
#ifndef  _SO_VR_RASTER_STACK_READER_
#define  _SO_VR_RASTER_STACK_READER_

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <LDM/readers/SoVolumeReader.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/image/SoRasterImageFile.h>
#include <Inventor/image/SoRasterReaderSet.h>
#include <Inventor/image/SoRasterImageRW.h>
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/string>
#include <VolumeViz/nodes/SoVolumeData.h>

/**
 *
 * @VREXT Raster stack reader.
 *
 * @ingroup VolumeVizReaders
 *
 * @DESCRIPTION
 *
 *  Reader for a volume defined by a "stack" of images.
 *
 *  A raster stack is a list of raster images (that is, containing pixels, not vector graphics)
 *  of the same size so as to create a voxel volume.
 *  This reader uses a descriptive file (.lst) including a header and a list of all the raster image filenames.
 *  The file is structured as follows:
 * \verbatim
    Parameters {
      Raw 0
      Dims 2048 2048 100
      Size 10.000000 10.000000 10.000000 1250.000000 1250.000000 500.000000
      Channel 2
    }

    C:/tmp/tiff/img00000.tiff
    C:/tmp/tiff/img00001.tiff
    C:/tmp/tiff/img00002.tiff
    C:/tmp/tiff/img00003.tiff
    C:/tmp/tiff/img00004.tiff
    ....
   \endverbatim
 *  OR
 * \verbatim
    Parameters {
      Raw 1
      Endianess 1
      HeaderLength 256
      PrimType 0
      Binary 0
      Dims 1024 1024 300
      Size 50.000000 50.000000 50.000000 1050.000000 1050.000000 250.000000
      Channel 1
    }

    img00000.raw
    img00001.raw
    img00002.raw
    img00003.raw
    img00004.raw
    ....
   \endverbatim
 *  @B Header Description@b
 *
 *   - Raw: 0 (formatted data file), 1 (raw data file).@BR
 *   - Endianess: 0 (little endian), 1 (big endian) (for raw data only).@BR
 *   - HeaderLength: length of header in characters/bytes (for raw data only).@BR
 *   - PrimType: See SoDataSet::DataType (for raw data only).@BR
 *   - Binary: 0 (ASCII), 1 (binary) (for raw data only).@BR
 *   - Dims: height, width, depth in pixels of the stack.@BR
 *   - Size: Xmin Ymin Zmin Xmax Ymax Zmax, bounding box of the data set.@BR
 *   - Channel: See SoVRRasterStackReader::ChannelSelection.@BR
 *
 * Images in the list can be of different formats (uses all files formats supported by Open Inventor).
 * VolumeViz can load image data in most common image formats including BMP, DDS, GIF, JPEG, JPEG2000,
 * PNG and TIFF.
 *
 * If images are different sizes, RD_UNSUPPORTED_DATA_TYPE_ERROR is returned by #getDataChar.
 *
 * The file path can be relative or absolute.
 *
 * It is mandatory to have a line break after each
 * property and/or filename otherwise VolumeViz could be unable to read the file properly.
 *
 *
 */
class VOLUMEVIZ_API SoVRRasterStackReader : public SoVolumeReader  
 {
  SO_FIELDCONTAINER_HEADER(SoVRRasterStackReader);

 public:

   /** Channel selection */
   enum ChannelSelection {
    /**
    *  Maximum of all channels.
    */
    MAX_CHANNEL = 0,   // Maximum of all channels
    /**
    *  Luminance.
    */
    LUMINANCE = 1,     // Luminance
    /**
    *  Red channel.
    */
    RED_CHANNEL = 2,   // Red channel
    /**
    *  Green channel.
    */
    GREEN_CHANNEL = 3, // Green channel
    /**
    *  Blue channel.
    */
    BLUE_CHANNEL = 4,  // Blue channel
    /**
    *  Alpha channel.
    */
    ALPHA_CHANNEL = 5, // Alpha channel
    /**
    *  All channels combined.
    */
    ALL_CHANNELS = 11  // All channels combined
   };

  /** Constructor. */
  SoVRRasterStackReader();

  /**
  * Specifies the path of the descriptive (.lst) file. Returns 0 for success. Any other return
  * value indicates failure.
  */
  virtual int setFilename(const SbString& filename);

  /**
  * Specify a list of files to load.
  * All the specified files should be part of the same volume data set.
  * @return true if successful.
  *
  * @M_SINCE 9.7
  */
  SbBool setFilenameList(const SbStringList& filenameList);

  /**
   * Specify a directory containing image files to load (jpg, png, tif,...).
   * Each image file in this folder with the exact same dimension and 
   * the exact same number of components will be loaded.
   *
   * Notes:
   * - The directory should contain only valid image files.
   * - The file format is automatically detected.
   * - If the images are RGB or RGBA, the volume is loaded as RGBA data.
   *   (It is not possible to load a single component using this method.)
   *
   * @param directory The directory from which to load files.
   * @return true if successful.
   *
   * @M_SINCE 9.5.0
   */
  SbBool setDirectory(const SbString& directory);

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar(SbBox3f& size, SoDataSet::DataType& type, SbVec3i32& dim) override;

  /**
   * @copydoc SoVolumeReader::getSubSlice(const SbBox2i32&,int,void*)
   */
  virtual void getSubSlice(const SbBox2i32& subSlice, int sliceNumber, void* data);

  /**
   * Defines the bounding box size in physical units of the data set.
   */
  void setSize(SbBox3f& size) {m_size = size;};

  /**
   * Selects channels to be used for data rendering.
   */
  void setChannelSelection(ChannelSelection cs) {m_channelSelection = cs;};

  /**
   * Returns the reader type.
   */
  virtual SoVolumeReader::ReaderType getReaderType ()  {return RASTERSTACK;};

  /**
   * Returns TRUE if the data set contains RGBA color values.
   */
  virtual SbBool isRGBA() const;

  /**
   * Return true as this reader is thread safe.
   */
  virtual SbBool isThreadSafe() const;

protected:

  int extractFileList(const SbString& headFile);
  SbBool extractParameters(const char* lineStr);

  SoDataSet::DataType findOutDestType(const SbRasterImage::Components nc, const unsigned int numComponent);

  void RGBA2HLS(SbVec3f& hls, float* rgba);

  float   convertFloat(float   value);
  short   convertShort(short   value);
  int32_t convertInt32(int32_t value);
  unsigned short convertUShort(unsigned short value);
  uint32_t convertUInt32(uint32_t value);
  SbBool needEndianConversion();

  int getChannelId(const char* channel);

  template <typename T>
  void getSubSliceInternal(const SbBox2i32 &subSlice, int sliceNumber, void* data);

  template <typename TSRC, typename T>
  void convertSubSliceInternal(TSRC* srcData, const SbVec2i32 srcSize, const unsigned int srcNumComponent, std::vector<T>& dstVector);

  /** Function to read the specified region of the image
   * Reads, decodes, and fills the @I rasterImage@i parameter.
   * If @I infoOnly@i is TRUE, then the buffer will not be read.
   * The parameter @I rasterImage@i will be set with raster size and raster number of components, the buffer will be NULL.
   *
   * @return success
   */
  SbBool readImage(int numImage, SbRasterImage& rasterImage, const SbBox2i32& region, SbBool infoOnly);

  template <typename T>
  void getSingleChannelData(std::vector<T> srcVector, SbVec2i32& size2s,
    unsigned int xMin, unsigned int xMax,
    unsigned int yMin, unsigned int yMax,
    void* data);

  template <typename T>
  T endianConverter(const T value);

  SbString  m_listDirectory;
  SbBox3f   m_size;
  SbVec3i32 m_dim;
  SbBool    m_isRawData;
  unsigned int m_nc;
  SoDataSet::DataType m_srcType;
  SoDataSet::DataType m_destType;
  ChannelSelection m_channelSelection;
  std::vector< SbString > m_fileList;
  unsigned int m_rawHeaderSize;
  SbBool m_rawIsBigEndian;
  SbBool m_rawIsBinary;

private:

  bool m_isInitialized;

  /*
   * Internal function to read a region of a raw file into a SbRasterImage
   * @return success
   */
  template <typename T>
  SbBool readRawBuffer(FILE* fp, SbRasterImage& rasterImage, const SbBox2i32& region);
};

#ifdef _WIN32
#pragma warning( pop )
#endif

#endif /* _SO_VR_RASTER_STACK_READER_ */

