/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#pragma once

#include <Inventor/devices/SoBufferObject.h>
#include <VolumeViz/readers/dicom/SoDicomTag.h>

class SiDicomElement;
class SiDicomValue;

/**
 * @VREXT DICOM data.
 *
 * @ingroup VolumeVizReaders
 *
 * @DESCRIPTION
 *  Access data in a DICOM data object.
 *
 *  DICOM is a widely used file format for storing medical image data (CT, MRI, etc),
 *  defined by the National Electrical Manufacturers Association (NEMA)
 *  (http://dicom.nema.org).
 *
 *  This class, along with SoDicomTag, SiDicomElement, SiDicomValue and SiDicomSequence,
 *  allows an application to iterate over or randomly access all the attributes in a
 *  DICOM data object. This includes attributes that are not used by VolumeViz for rendering.
 *
 *  This class cannot directly open a DICOM file.  An SiDicomDataSet object can be
 *  requested from an SoVRDicomFileReader object or an SoVRDicomData object using the
 *  getDicomDataSet() method.
 *
 * @SEE_ALSO
 *   SoVRDicomFileReader,
 *   SoVRDicomData,
 *   SoDicomTag,
 *   SiDicomElement,
 *   SiDicomValue,
 *   SiDicomSequence
 */
class VOLUMEVIZ_API SiDicomDataSet
{
public:
  /**
   * Destructor
   */
  virtual ~SiDicomDataSet() {}

  /**
   * Create a clone of the current data set.
   *
   * @return a newly created DICOM data set, that must be deleted by the
   *         caller.
   */
  virtual SiDicomDataSet* clone() const = 0;

  /**
   * Get the number of elements in the data set.
   */
  virtual size_t getLength() const = 0;

  /**
   * Get an element of this data set.
   *
   * @param  tag the tag of the element to find.
   * @return     a pointer to the found element or NULL if it cannot be found.
   */
  virtual const SiDicomElement* getElement( const SoDicomTag& tag ) const = 0;

  /**
   * Get the n-th element of this data set.
   *
   * @param  idx the index of the element in the data set.
   * @return     a pointer to the n-th element or NULL if idx is greater
   *             than the data set length.
   */
  virtual const SiDicomElement* getElement( const size_t idx ) const = 0;

  /**
   * Shortcut to get an element's value fast.
   *
   * @param tag The tag of the element to find.
   *
   * @return The element's value, or NULL if the element cannot be found.
   */
  virtual const SiDicomValue* getElementValue( const SoDicomTag& tag ) const = 0;

  /**
   * Shortcut to extract an element's buffer value.
   *
   * @param tag    The tag of the element to find.
   * @param buffer The buffer to fill with element's value.
   *
   * @return true if the extraction was successful, or false otherwise.
   */
  virtual bool getElementBufferValue( const SoDicomTag& tag, SoBufferObject* buffer ) const = 0;

  /**
   * Shortcut to extract a STRING element's value.
   *
   * If the stored value if not of type STRING, this method will return a string
   * representation of the value. currently works for INT, UINT, and DOUBLE.
   *
   * @param tag          The tag of the element to find.
   * @param idx          The index of the scalar value to extract.
   * @param defaultValue The value returned if the extraction fails.
   *
   * @return The string stored at the given index in the element with the given
   *         tag, or the default value if the extraction fails.
   */
  virtual SbString getElementStringValue( const SoDicomTag& tag, const size_t idx, const SbString& defaultValue ) const = 0;

  /**
   * Shortcut to extract a INT element's value.
   *
   * If the value is of a numeric type different from INT, it will be casted to
   * a 64 bits integer. That casting may cause a precision loss.
   *
   * @param tag          The tag of the element to find.
   * @param idx          The index of the scalar value to extract.
   * @param defaultValue The value returned if the extraction fails.
   *
   * @return The 64 bits integer stored at the given index in the element with
   *         the given tag, or the default value if the extraction fails.
   */
  virtual int64_t getElementIntValue( const SoDicomTag& tag, const size_t idx, const int64_t defaultValue ) const = 0;

  /**
   * Shortcut to extract a UINT element's value.
   *
   * If the value is of a numeric type different from UINT, it will be casted to
   * an unsigned 64 bits integer. That casting may cause a precision loss.
   *
   * @param tag          The tag of the element to find.
   * @param idx          The index of the scalar value to extract.
   * @param defaultValue The value returned if the extraction fails.
   *
   * @return The unsigned 64 bits integer stored at the given index in the
   *         element with the given tag, or the default value if the
   *         extraction fails.
   */
  virtual uint64_t getElementUIntValue( const SoDicomTag& tag, const size_t idx, const uint64_t defaultValue ) const = 0;

  /**
   * Shortcut to extract a DOUBLE element's value.
   *
   * If the value is of a numeric type different from DOUBLE, it will be casted
   * to a double. That casting may cause a precision loss.
   *
   * @param tag          The tag of the element to find.
   * @param idx          The index of the scalar value to extract.
   * @param defaultValue The value returned if the extraction fails.
   *
   * @return The double stored at the given index in the element with the given
   *         tag, or the default value if the extraction fails.
   */
  virtual double getElementDoubleValue( const SoDicomTag& tag, const size_t idx, const double defaultValue ) const = 0;
};
