/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#pragma once

#include <Inventor/SbString.h>
#include <Inventor/devices/SoBufferObject.h>

/**
 * @VREXT DICOM data.
 *
 * @ingroup VolumeVizReaders
 *
 * @DESCRIPTION
 * Access value of a DICOM element.
 *
 * @SEE_ALSO
 *   SoVRDicomFileReader,
 *   SoVRDicomData,
 *   SoDicomTag,
 *   SiDicomDataSet,
 *   SiDicomElement,
 *   SiDicomSequence
 */
class VOLUMEVIZ_API SiDicomValue
{
public:
  enum ValueType
  {
    BUFFER,
    STRING,
    INT,
    UINT,
    DOUBLE,
  };

  /**
   * Destructor
   */
  virtual ~SiDicomValue() {}

  /**
   * Create a clone of the value.
   */
  virtual SiDicomValue* clone() const = 0;

  /**
   * Get the type of the value.
   */
  virtual ValueType getValueType() const = 0;

  /**
   * Get the number of scalar values stored.
   */
  virtual size_t getLength() const = 0;

  /**
   * Return a printable version of the stored value.
   */
  virtual SbString asString() const = 0;

  /**
   * Copies the internal buffer to the given one.
   *
   * @param buffer the destination buffer.
   */
  virtual void getBuffer( SoBufferObject* buffer ) const = 0;

  /**
   * Get the internally stored string.
   *
   * If the value's type is not STRING, this method will return an empty string.
   *
   * @param  idx the index of the scalar value to get.
   */
  virtual SbString getString( const size_t idx = 0 ) const = 0;

  /**
   * Get an internally stored signed integer.
   *
   * If the value's type is not INT, this method will return 0.
   *
   * @param  idx the index of the scalar value to get.
   */
  virtual int64_t getInt( const size_t idx = 0 ) const = 0;

  /**
   * Get an internally stored unsigned integer.
   *
   * If the value's type is not UINT, this method will return 0.
   *
   * @param  idx the index of the scalar value to get.
   */
  virtual uint64_t getUInt( const size_t idx = 0 ) const = 0;

  /**
   * Get an internally stored double precision floating point number.
   *
   * If the value's type is not DOUBLE, this method will return 0.0.
   *
   * @param  idx the index of the scalar value to get.
   */
  virtual double getDouble( const size_t idx = 0 ) const = 0;
};
