#pragma once

#include <Inventor/SbString.h>

// The DICOM dictionary is initialized with the init method and is freed
// with the finish method.Without this initialization, most of this class
// behavior is undefined.

/**
 * @VREXT DICOM data.
 *
 * @ingroup VolumeVizReaders
 *
 * @DESCRIPTION
 * Holds data of a DICOM tag (its ID) and associates them with a DICOM
 * dictionary in order to get information from the specification such
 * as the name and Visual Representation (VR).
 *
 * @SEE_ALSO
 *   SoVRDicomFileReader,
 *   SoVRDicomData,
 *   SiDicomDataSet,
 *   SiDicomElement,
 *   SiDicomValue,
 *   SiDicomSequence
 */
class VOLUMEVIZ_API SoDicomTag
{
public:
  /**
   * VRs (Visual Representations) defined in the DICOM standard.
   *
   * @see ftp://dicom.nema.org/medical/DICOM/2013/output/chtml/part05/sect_6.2.html
   */
  enum VR
  {
    AE, ///< Application Entity
    AS, ///< Age String (Duration)
    AT, ///< Attribute Tag
    CS, ///< Code String
    DA, ///< Date String
    DS, ///< Decimal String
    DT, ///< Date Time String
    FL, ///< Floating point single precision
    FD, ///< Floating point double precision
    IS, ///< Integer String
    LO, ///< Long String (vector of 32bits signed integers)
    LT, ///< Long Text
    OB, ///< Other Byte String (byte vector)
    OD, ///< Other Double String (vector of double)
    OF, ///< Other Float String (vector of float)
    OL, ///< Other Long String (vector of int)
    OV, ///< Other 64 bit very long
    OW, ///< Other Word String (16 bits integers vector)
    PN, ///< Person Name
    SH, ///< Short String
    SL, ///< Signed 32 bits integer
    SQ, ///< Sequence
    SS, ///< Signed 16 bits integer
    ST, ///< Short Text
    SV, ///< Signed 64 bit very long
    TM, ///< Time String
    UC, ///< Unlimited Characters
    UI, ///< Unique Identifier
    UL, ///< Unsigned 32 bits integer
    UN, ///< Unknown
    UR, ///< Universal Resource Identifier
    US, ///< Unsigned 16 bits integer
    UT, ///< Unlimited Text
    UV, ///< Unsigned 64 bit very long
  };

  /**
   * Initialize the DICOM dictionnary.
   */
  static void initClass();

  /**
   * Free the space allocated to store the DICOM dictionnary.
   */
  static void exitClass();

  SoDicomTag() : m_id(0) {}

  /**
   * Create a DICOM tag from its 32 bit id.
   */
  SoDicomTag( unsigned int id );

  /**
   * Create a DICOM from its group and element values.
   */
  SoDicomTag( unsigned short group, unsigned short element );

  /**
   * Create a DICOM tag from its name in the dictionnary.
   *
   * If the name is not in the dictionnary, return a tag with an id of zero.
   * isKeyword could be an enum
   */
  SoDicomTag( const SbString& name, bool isKeyword = false );

  /**
   * Get the tag's id ( group and element combined ).
   */
  unsigned int getId() const;

  /**
   * Get the tag's group.
   */
  unsigned short getGroup() const;

  /**
   * Get the tag's element.
   */
  unsigned short getElement() const;

  /**
   * Returns the tag's name in the dictionary or "Unknown" if not found.
   */
  SbString getName() const;

  /**
   * Returns the tag's keyword in the dictionary or "Unknown" if not found.
   */
  SbString getKeyword() const;

  /**
   * Get the VR (Visual Representation) associated with this tag in
   * the specification, returning "UN" if the tag is not found.
   */
  VR getVR() const;

private:
  static void push( unsigned int id, const SbString& name, const SbString& keyword, VR vr );
  static void initSpec();

  unsigned int m_id;

  class DicomSpecManager;
  static DicomSpecManager* s_manager;
};
