#ifndef OIV_SHADER_VARIABLES_H
#define OIV_SHADER_VARIABLES_H

/*****************************************************************************/
/** IN **/

/**
 * Get the interpolated color for this fragment
 * (as defined by OivSetFrontColor and OivSetBackColor in previous stages).
 * Available in:
 *   Fragment Shader
 * The equivalent legacy value is gl_Color
 */
vec4 OivFragmentColor();

/**
 * Get the interpolated texture coordinates for this fragment.
 * Available in:
 *   Fragment Shader
 * The equivalent legacy value is gl_TexCoord[unit]
 */
vec4 OivFragmentTexCoord( in int unit );

/**
 * Get the fog frag coord of the input vertex or fragment.
 * Available in:
 *   Vertex Shader
 *   Fragment Shader
 * The equivalent legacy values are:
 *   Vertex Shader: gl_FogCoord
 *   Fragment Shader: gl_FogFragCoord
 */
float OivFogFragCoord();

/**
 * Get the front color of the input vertex or fragment at index.
 * Available in:
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy value is gl_in[index].gl_FrontColor
 */
vec4 OivFrontColor( in int index );

/**
 * Get the back color of the input vertex or fragment at index.
 * Available in:
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy value is gl_in[index].gl_BackColor
 */
vec4 OivBackColor( in int index );

/**
 * Get the texture coordinates of the input vertex at index.
 * Available in:
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy value is gl_in[index].gl_TexCoord[unit]
 */
vec4 OivTexCoord( in int index, in int unit );

/**
 * Get the fog frag coord of the input vertex at index.
 * Available in:
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy value is gl_in[index].gl_FogFragCoord
 */
float OivFogFragCoord( in int index );

/*****************************************************************************/
/** OUT **/

/**
 * Some GPU require vertex data to be initialized even if you don't use them.
 * If you encounter message error such as:
 * "The fragment shader uses varying _I;OIV_VERTEX_DATA;xxx, but previous shader does not write to it.",
 * you should call this method at the beginning of your vertex shader main.
 */
void OivInitVertexData();

/**
 * Set the front color of the output vertex for the next shader stage.
 * Available in:
 *   Vertex Shader
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy calls are:
 *   Vertex/Tessellation Evaluation/Geometry Shaders: gl_FrontColor = color;
 *   Tessellation Control Shader: gl_out[gl_InvocationID].gl_FrontColor = color;
 */
void OivSetFrontColor( in vec4 color );

/**
 * Set the back color of the output vertex for the next shader stage.
 * Available in:
 *   Vertex Shader
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy calls are:
 *   Vertex/Tessellation Evaluation/Geometry Shaders: gl_BackColor = color;
 *   Tessellation Control Shader: gl_out[gl_InvocationID].gl_BackColor = color;
 */
void OivSetBackColor( in vec4 color );

/**
 * Set the texture coordinates of the output vertex for the next shader stage.
 * Available in:
 *   Vertex Shader
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy calls are:
 *   Vertex/Tessellation Evaluation/Geometry Shaders: gl_TexCoord[unit] = coord;
 *   Tessellation Control Shader: gl_out[gl_InvocationID].gl_TexCoord[unit] = coord;
 */
void OivSetTexCoord( in int unit, in vec4 coord );

/**
 * Set the fog frag coord of the output vertex for the next shader stage.
 * Available in:
 *   Vertex Shader
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy calls are:
 *   Vertex/Tessellation Evaluation/Geometry Shaders: gl_FogFragCoord = coord;
 *   Tessellation Control Shader: gl_out[gl_InvocationID].gl_FogFragCoord = coord;
 */
void OivSetFogFragCoord( in float coord );

/**
 * Set the output value of the current fragment in the buffer specified by bufferIndex.
 * Available in:
 *   Fragment Shader
 * The equivalent legacy call is:
 *   gl_FragData[bufferIndex] = outputValue;
 */
void OivFragmentOutput( in int bufferIndex, in vec4 outputValue );

/**
 * Set the output value of the current fragment.
 * This is a convenient function that calls OivFragmentOutput(0, outputValue)
 * Available in:
 *   Fragment Shader
 * The equivalent legacy call is:
 *   gl_FragColor = outputValue;
 */
void OivFragmentOutput( in vec4 outputValue );

/*****************************************************************************/
/** PASSTHROUGH HELPERS **/

/**
 * Copy variables from the input vertex at inIndex to the output vertex.
 * Available in:
 *   Tessellation Control Shader
 *   Tessellation Evaluation Shader
 *   Geometry Shader
 * The equivalent legacy calls are:
 *   Tessellation Evaluation/Geometry Shaders:
 *     gl_FrontColor          = gl_in[inIndex].gl_FrontColor;
 *     gl_BackColor           = gl_in[inIndex].gl_BackColor;
 *     gl_TexCoord            = gl_in[inIndex].gl_TexCoord;
 *     gl_FogFragCoord        = gl_in[inIndex].gl_FogFragCoord;
 *   Tessellation Control Shaders:
 *     gl_out[gl_InvocationID].gl_FrontColor          = gl_in[inIndex].gl_FrontColor;
 *     gl_out[gl_InvocationID].gl_BackColor           = gl_in[inIndex].gl_BackColor;
 *     gl_out[gl_InvocationID].gl_TexCoord            = gl_in[inIndex].gl_TexCoord;
 *     gl_out[gl_InvocationID].gl_FogFragCoord        = gl_in[inIndex].gl_FogFragCoord;
 */
void OivSetupVertex( in int inIndex );

/**
 * Copy variables from the input vertex to the output vertex.
 * Available in:
 *   Vertex Shader
 * The equivalent legacy calls are:
 *   gl_FrontColor   = gl_Color;
 *   gl_BackColor    = gl_Color;
 *   gl_TexCoord[0]  = gl_MultiTexCoord0;
 *   gl_TexCoord[1]  = gl_MultiTexCoord1;
 *   gl_TexCoord[2]  = gl_MultiTexCoord2;
 *   gl_TexCoord[3]  = gl_MultiTexCoord3;
 *   gl_FogFragCoord = gl_FogCoord;
 */
void OivSetupVertex();

#endif /* OIV_SHADER_VARIABLES_H */
