/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#include <Inventor/ViewerComponents/MFC/RenderAreaInteractive.h>
#include <Inventor/ViewerComponents/nodes/SceneInteractor.h>

#include <Inventor/ViewerComponents/MFC/WinTimer.h>

#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoStereoCamera.h>
#include <Inventor/ViewerComponents/SoRenderAreaCore.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoSceneManager.h>

BEGIN_MESSAGE_MAP( RenderAreaInteractive, CWnd )
  ON_WM_LBUTTONDOWN()
  ON_WM_MBUTTONDOWN()
  ON_WM_RBUTTONDOWN()
  ON_WM_LBUTTONUP()
  ON_WM_MBUTTONUP()
  ON_WM_RBUTTONUP()
  ON_WM_LBUTTONDBLCLK()
  ON_WM_MBUTTONDBLCLK()
  ON_WM_RBUTTONDBLCLK()
  ON_WM_MOUSEMOVE()
  ON_WM_MOUSEWHEEL()
  ON_WM_PAINT()
  ON_WM_ERASEBKGND( )
  ON_WM_SIZE()
  ON_WM_MOUSELEAVE()
  ON_WM_VSCROLL()
  ON_WM_SETCURSOR()
END_MESSAGE_MAP()

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive()
  : RenderArea()
  , m_mouseTracking(false)
  , m_countDownButtons( 0 )
{
  init(true);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive( bool buildRootSceneGraph )
  : RenderArea()
  , m_mouseTracking(false)
  , m_countDownButtons( 0 )
{
  init(buildRootSceneGraph);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::~RenderAreaInteractive()
{
}

void
RenderAreaInteractive::startRender( SiRenderArea::RenderEventArg& )
{
  this->paintGL();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::init( bool buildRootSceneGraph )
{
  m_clippingMode = AUTO;
  m_interactiveMode = SoInteractiveComplexity::AUTO;
  m_isAutoInteractive = true;

  if ( buildRootSceneGraph )
  {
    m_rootSceneGraph = new SceneInteractor();
    buildSceneGraph();
  }

  SoDB::setSystemTimer( new WinTimer );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::buildSceneGraph()
{
  m_appSceneGraph = new SoGroup;
  m_rootSceneGraph->addChild(m_appSceneGraph);
  RenderArea::setSceneGraph(m_rootSceneGraph);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::initializeGL( HDC deviceContext, bool enableStereo )
{
  RenderArea::initializeGL( deviceContext, enableStereo );
  m_renderAreaCore->onStartRender().add( *this, &RenderAreaInteractive::startRender );
  m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
  m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
}


//------------------------------------------------------------------------------
void
RenderAreaInteractive::setSceneGraph(SoNode *sceneGraph)
{
  m_appSceneGraph->removeAllChildren();
  m_appSceneGraph->addChild(sceneGraph);
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getRootSceneGraph()
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getSceneInteractor() const
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
SoRenderAreaCore::RenderStatus
RenderAreaInteractive::render()
{
  if ( m_clippingMode == AUTO )
  {
    // Update camera clipping planes before each rendering.
    m_rootSceneGraph->adjustClippingPlanes( m_renderAreaCore->getSceneManager()->getViewportRegion() );
  }

  // render scene
  return RenderArea::render();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
{
  if (m_clippingMode != mode)
  {
    m_clippingMode = mode;
    OnPaint();
  }
}

//------------------------------------------------------------------------------
RenderAreaInteractive::ClippingPlanesAdjustMode
RenderAreaInteractive::getClippingPlanesAdjustMode()
{
  return m_clippingMode;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::viewAll(const SbViewportRegion &viewport)
{
  m_rootSceneGraph->viewAll(viewport);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::viewAxis( const SbVec3f& direction, const SbVec3f& up )
{
  m_rootSceneGraph->viewAxis( direction, up );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::activateStereo( bool activated )
{
  if ( activated && !isStereoSupported() )
  {
    MessageBox( _T( "Current camera is not a SoStereoCamera!" ), _T( "Warning" ), MB_ICONWARNING | MB_OK );
    return;
  }

  RenderArea::activateStereo( activated );
}

//------------------------------------------------------------------------------
bool
RenderAreaInteractive::isStereoSupported()
{
  // check camera
  SoCamera* camera = m_rootSceneGraph->getCamera();
  return (dynamic_cast<SoStereoCamera*>(camera) != NULL);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setStereoCameraOffset( float offset )
{
  SoCamera* camera = m_rootSceneGraph->getCamera();
  SoStereoCamera* stereoCamera = dynamic_cast<SoStereoCamera*>( camera );

  if ( stereoCamera == NULL )
    MessageBox( _T( "Current camera is not a SoStereoCamera!" ), _T( "Warning" ), MB_ICONWARNING | MB_OK );
  else
    stereoCamera->offset = offset;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setStereoCameraBalance( float balance )
{
  SoCamera* camera = m_rootSceneGraph->getCamera();
  SoStereoCamera* stereoCamera = dynamic_cast<SoStereoCamera*>( camera );

  if ( stereoCamera == NULL )
    MessageBox( _T( "Current camera is not a SoStereoCamera!" ), _T( "Warning" ), MB_ICONWARNING | MB_OK );
  else
    stereoCamera->balance = balance;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setInteractiveMode( SoInteractiveComplexity::InteractiveMode mode )
{
  m_interactiveMode = mode;
  m_isAutoInteractive = ( mode == SoInteractiveComplexity::AUTO );

  if ( m_renderAreaCore.ptr() != NULL )
  {
    m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
    m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
  }
}

//------------------------------------------------------------------------------
SoInteractiveComplexity::InteractiveMode
RenderAreaInteractive::getInteractiveMode() const
{
  return m_interactiveMode;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvent( const SoEvent *event )
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvent( event );

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvents( const std::vector<const SoEvent*>& eventList )
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvents( eventList );

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnLButtonDown(UINT flags, CPoint loc)
{
  mousePressEvent( loc, SoMouseButtonEvent::BUTTON1 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMButtonDown(UINT flags, CPoint loc)
{
  mousePressEvent( loc, SoMouseButtonEvent::BUTTON2 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnRButtonDown(UINT flags, CPoint loc)
{
  mousePressEvent( loc, SoMouseButtonEvent::BUTTON3 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnLButtonUp(UINT flags, CPoint loc)
{
  mouseReleaseEvent( loc, SoMouseButtonEvent::BUTTON1 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMButtonUp(UINT flags, CPoint loc)
{
  mouseReleaseEvent( loc, SoMouseButtonEvent::BUTTON2 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnRButtonUp(UINT flags, CPoint loc)
{
  mouseReleaseEvent( loc, SoMouseButtonEvent::BUTTON3 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnLButtonDblClk(UINT flags, CPoint loc)
{
  mouseDoubleClickEvent( loc, SoMouseButtonEvent::BUTTON1 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMButtonDblClk(UINT flags, CPoint loc)
{
  mouseDoubleClickEvent( loc, SoMouseButtonEvent::BUTTON2 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnRButtonDblClk(UINT flags, CPoint loc)
{
  mouseDoubleClickEvent( loc, SoMouseButtonEvent::BUTTON3 );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMouseMove( UINT flags, CPoint loc )
{
  if (!m_mouseTracking)
  {
    TRACKMOUSEEVENT tme;
    tme.cbSize = sizeof(TRACKMOUSEEVENT);
    tme.dwFlags = TME_LEAVE;
    tme.hwndTrack = this->m_hWnd;
    ::_TrackMouseEvent(&tme);

    SoEvent* event = MFCEventToSoEvent::getMouseEnterEvent(this);
    processEvent( event );

    m_mouseTracking = true;
  }

  SoEvent* event = MFCEventToSoEvent::getMouseMoveEvent( loc, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
BOOL
RenderAreaInteractive::OnMouseWheel(UINT flags, short zDelta, CPoint pt)
{
  SoEvent* event = MFCEventToSoEvent::getMouseWheelEvent( zDelta, this );
  return processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMouseHover(UINT nFlags, CPoint point)
{
  SoEvent* event = MFCEventToSoEvent::getMouseEnterEvent(this);
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::OnMouseLeave()
{
  m_mouseTracking = false;

  SoEvent* event = MFCEventToSoEvent::getMouseLeaveEvent(this);
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::keyPressEvent( WPARAM param )
{
  SoEvent* event = MFCEventToSoEvent::getKeyPressEvent( param, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::keyReleaseEvent( WPARAM param )
{
  SoEvent* event = MFCEventToSoEvent::getKeyReleaseEvent( param, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mousePressEvent( CPoint loc, SoMouseButtonEvent::Button button )
{
  setCaptureOnButtonDown();
  SoEvent* event = MFCEventToSoEvent::getMousePressEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mouseReleaseEvent( CPoint loc, SoMouseButtonEvent::Button button )
{
  releaseCaptureOnButtonUp();
  SoEvent* event = MFCEventToSoEvent::getMouseReleaseEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mouseDoubleClickEvent( CPoint loc, SoMouseButtonEvent::Button button )
{
  SoEvent* event = MFCEventToSoEvent::getMouseDoubleClickEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
BOOL
RenderAreaInteractive::PreTranslateMessage( MSG* pMsg )
{
  if ( pMsg->message == WM_KEYDOWN || pMsg->message == WM_SYSKEYDOWN )
  {
    keyPressEvent( pMsg->wParam );
  }
  else if ( pMsg->message == WM_KEYUP || pMsg->message == WM_SYSKEYUP )
  {
    keyReleaseEvent( pMsg->wParam );
  }
  return CWnd::PreTranslateMessage( pMsg );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setCaptureOnButtonDown()
{
  m_countDownButtons++;
  SetCapture();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::releaseCaptureOnButtonUp()
{
  m_countDownButtons--;
  if ( m_countDownButtons == 0 )
  {
    ReleaseCapture();
  }
}
