#include <Inventor/ViewerComponents/Qt/QML/RenderArea.h>

#include <Inventor/devices/SoGLContext.h>

#include <QOpenGLFramebufferObject>
#include <QQuickWindow>
#include <QScreen>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
#include <QQuickOpenGLUtils>
#endif

namespace openinventor {
namespace inventor {
namespace viewercomponents {
namespace qt {
namespace qml {

/**
 * Class in used to implement the rendering logic of RenderArea, which inherits from QQuickFramebufferObject.
 */
class QmlRenderer : public QQuickFramebufferObject::Renderer
{
public:
  QmlRenderer( RenderArea* renderArea )
    : m_window( nullptr )
    , m_region( SbVec2i32( 512, 512 ) )
    , m_renderArea( renderArea )
  {
  }

protected:
  /**
   * Calls when the rendeAreaCore should be rendered.
   */
  void
  render()
  {
    m_renderArea->render();
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
    QQuickOpenGLUtils::resetOpenGLState();
#else
    m_window->resetOpenGLState();
#endif
  }

  /*
   * Calls when QML need a new framebufferObject, for example when the render size changes.
   * This function creates a new SoRenderAreaCore.
   */
  QOpenGLFramebufferObject*
  createFramebufferObject( const QSize& size )
  {
    // only re-create the RenderAreaCore when necessary
    if ( !m_renderArea->m_renderAreaCore.ptr() || m_renderArea->m_renderAreaCore.ptr()->getGLContext() != SoGLContext::getCurrent( true ) )
    {
      m_renderArea->createRenderAreaCore( m_region );
    }
    QOpenGLFramebufferObject* localCreateFramebufferObject = QQuickFramebufferObject::Renderer::createFramebufferObject( size );
    QOpenGLFramebufferObjectFormat format = localCreateFramebufferObject->format();
    format.setAttachment( QOpenGLFramebufferObject::CombinedDepthStencil );
    // let Inventor manage the samples
    format.setSamples( 1 );
    format.setInternalTextureFormat( GL_RGBA8 );
    return new QOpenGLFramebufferObject( size, format );
  }

  /*
   * This function is called as a result of RenderArea::update() to update the renderer
   * with changes occurred in the RenderArea, for example when the scene graph changes.
   */
  void
  synchronize( QQuickFramebufferObject* item )
  {
    m_window = item->window();
    // Apply Device pixel ratio to width and height return by Qt
    const float ratio = m_window->effectiveDevicePixelRatio();
    m_region = SbVec2i32( int( item->width() * ratio ), int( item->height() * ratio ) );

    RenderArea* renderArea = dynamic_cast<RenderArea*>( item );
    if ( renderArea && renderArea->m_renderAreaCore.ptr() )
    {
      m_renderArea->m_renderAreaCore.ptr()->setSize( m_region );
    }
  }

private:
  QQuickWindow* m_window;
  SbVec2i32 m_region = SbVec2i32( 512, 512 );
  RenderArea* m_renderArea;
};

} // namespace qml
} // namespace qt
} // namespace viewercomponents
} // namespace inventor
} // namespace openinventor

using namespace openinventor::inventor::viewercomponents::qt::qml;

// Render area component
RenderArea::RenderArea()
  : m_renderAreaCore( nullptr )
  , m_sceneGraph( nullptr )
  , m_clearPolicy( COLORBUFFER_AND_DEPTHBUFFER )
  , m_color( 0, 0, 0, 0 )
  , m_depth( 1.0f )
  , m_size( 0, 0 )
  , m_stereo( false )
  , m_glRenderAction( nullptr )
  , m_transparencyType( SoGLRenderAction::TransparencyType::NO_SORT )
  , m_antialiasingMode( SoSceneManager::NO_ANTIALIASING )
  , m_antialiasingQuality( 0 )
  , m_stillAntialiasingDelay( 0 )
  , m_stillAntialiasingQuality( 0 )
{
  setMirrorVertically( true );
  connect( this, &QQuickItem::windowChanged, this, &RenderArea::onWindowChanged );
}

//------------------------------------------------------------------------------
RenderArea::~RenderArea()
{
  if ( m_sceneGraph != nullptr )
    m_sceneGraph->unref();
}

//------------------------------------------------------------------------------
void
RenderArea::setSceneGraph( SoNode* sceneGraph )
{
  if ( sceneGraph == m_sceneGraph )
    return;

  // unref previous scene graph
  if ( m_sceneGraph != nullptr )
    m_sceneGraph->unref();

  m_sceneGraph = sceneGraph;
  // keep an additional ref to the scene graph, because the scene renderer
  // may be created later.
  if ( m_sceneGraph != nullptr )
    m_sceneGraph->ref();

  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setSceneGraph( m_sceneGraph );

  update();
}

//------------------------------------------------------------------------------
SoNode*
RenderArea::getSceneGraph() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getSceneGraph();
  else
    return m_sceneGraph;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearPolicy( SiRenderArea::ClearPolicy policy )
{
  m_clearPolicy = policy;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setClearPolicy( m_clearPolicy );
}

//------------------------------------------------------------------------------
SiRenderArea::ClearPolicy
RenderArea::getClearPolicy() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getClearPolicy();
  else
    return m_clearPolicy;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearColor( const SbColorRGBA& color )
{
  m_color = color;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setClearColor( m_color );
}

//------------------------------------------------------------------------------
SbColorRGBA
RenderArea::getClearColor() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getClearColor();
  else
    return m_color;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearDepth( float depth )
{
  m_depth = depth;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setClearDepth( m_depth );
}

//------------------------------------------------------------------------------
float
RenderArea::getClearDepth() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getClearDepth();
  else
    return m_depth;
}

//------------------------------------------------------------------------------
void
RenderArea::setSize( const SbVec2i32& size )
{
  m_size = size;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setSize( m_size );
}

//------------------------------------------------------------------------------
SbVec2i32
RenderArea::getSize() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getSize();
  else
    return m_size;
}

//------------------------------------------------------------------------------
void
RenderArea::setGLRenderAction( SoGLRenderAction* glAction )
{
  m_glRenderAction = glAction;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setGLRenderAction( glAction );
}

//------------------------------------------------------------------------------
SoGLRenderAction*
RenderArea::getGLRenderAction() const
{
  if ( m_renderAreaCore.ptr() != nullptr )
    return m_renderAreaCore->getGLRenderAction();
  else
    return m_glRenderAction;
}

//------------------------------------------------------------------------------
void
RenderArea::setTransparencyType( SoGLRenderAction::TransparencyType type )
{
  m_transparencyType = type;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setTransparencyType( type );
}

//------------------------------------------------------------------------------
SoGLRenderAction::TransparencyType
RenderArea::getTransparencyType() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_transparencyType;
  return m_renderAreaCore.ptr()->getTransparencyType();
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingMode( SoSceneManager::AntialiasingMode mode )
{
  m_antialiasingMode = mode;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingMode( mode );
}

//------------------------------------------------------------------------------
SoSceneManager::AntialiasingMode
RenderArea::getAntialiasingMode() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingMode;
  return m_renderAreaCore.ptr()->getAntialiasingMode();
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingQuality( float quality )
{
  m_antialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getAntialiasingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingQuality;
  return m_renderAreaCore.ptr()->getAntialiasingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingQuality( float quality )
{
  m_stillAntialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getStillSuperSamplingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingQuality;
  return m_renderAreaCore.ptr()->getStillSuperSamplingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingDelay( unsigned int delay )
{
  m_stillAntialiasingDelay = delay;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingDelay( delay );
}

//------------------------------------------------------------------------------
unsigned int
RenderArea::getStillSuperSamplingDelay() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingDelay;
  return m_renderAreaCore.ptr()->getStillSuperSamplingDelay();
}

//------------------------------------------------------------------------------
void
RenderArea::activateStereo( bool activated )
{
  if ( m_stereo == activated )
    return;

  if ( activated && !isRawStereoAvailable() )
    qWarning( "Stereo buffers are not enabled" );
}

//------------------------------------------------------------------------------
bool
RenderArea::isStereoActivated() const
{
  SoDebugError::postWarning( "RenderArea::isStereoActivated", "Stereo is not compatible with QML RenderArea." );
  return false;
}

//------------------------------------------------------------------------------
void
RenderArea::setStereoParameters( SoStereoParameters* /*parameters*/ )
{
  SoDebugError::postWarning( "RenderArea::setStereoParameters", "Stereo is not compatible with QML RenderArea." );
}

//------------------------------------------------------------------------------
SoStereoParameters*
RenderArea::getStereoParameters() const
{
  SoDebugError::postWarning( "RenderArea::getStereoParameters", "Stereo is not compatible with QML RenderArea." );
  return nullptr;
}

//------------------------------------------------------------------------------
bool
RenderArea::isRawStereoAvailable()
{
  return false;
}

//------------------------------------------------------------------------------
SoRenderAreaCore::RenderStatus
RenderArea::render()
{
  if ( !isVisible() )
    return SoRenderAreaCore::ABORTED;

  // render the scene graph.
  SoRenderAreaCore::RenderStatus renderStatus = m_renderAreaCore->render();

  // Check if the rendering is made well
  // If not : don't swap the buffer because we can have actifacts in the buffer
  if ( renderStatus == SoRenderAreaCore::ABORTED )
  {
    // Ask for a new render if the previous frame was aborted (for example, a frame
    // may have been aborted because some GUI events were waiting to be processed)
    m_renderAreaCore->getSceneManager()->scheduleRedraw();
  }

  return renderStatus;

}

//------------------------------------------------------------------------------
bool
RenderArea::event( QEvent* e )
{
  if ( e->type() == QEvent::UpdateRequest || e->type() == QEvent::Show )
  {
    render();
    return true;
  }
  return QQuickItem::event( e );
}

//------------------------------------------------------------------------------
SbEventHandler<SiRenderArea::RenderEventArg&>&
RenderArea::onStartRender()
{
  return m_renderAreaCore->onStartRender();
}

//------------------------------------------------------------------------------
QQuickFramebufferObject::Renderer*
RenderArea::createRenderer() const
{
  return new QmlRenderer( ( RenderArea* )this );
}

//------------------------------------------------------------------------------
void
RenderArea::createRenderAreaCore( SbVec2i32 region )
{
  m_renderAreaCore = new SoRenderAreaCore( SoGLContext::getCurrent( true ) );
  m_renderAreaCore->setSize( region );
  m_renderAreaCore->setClearDepth( m_depth );
  m_renderAreaCore->setClearColor( m_color );
  m_renderAreaCore->setSceneGraph( m_sceneGraph );
  if ( m_glRenderAction != nullptr )
    m_renderAreaCore->setGLRenderAction( m_glRenderAction );
  m_renderAreaCore->setTransparencyType( m_transparencyType );
  m_renderAreaCore->setAntialiasingMode( m_antialiasingMode );
  m_renderAreaCore->setAntialiasingQuality( m_antialiasingQuality );
  m_renderAreaCore->setStillSuperSamplingDelay( m_stillAntialiasingDelay );
  m_renderAreaCore->setStillSuperSamplingQuality( m_stillAntialiasingQuality );
}

void
RenderArea::onWindowChanged( QQuickWindow* window )
{
  if ( window )
  {
    // Connect screen changed slot to update context when the user change application screen
    connect( window, &QQuickWindow::screenChanged, this, &RenderArea::onScreenChanged );

    // Update defaultPixelPerInch
    const float ratio = window->effectiveDevicePixelRatio();
    SbViewportRegion::setDefaultPixelsPerInch( SbViewportRegion::s_historicalPixelPerInch * ratio );
  }
}

void
RenderArea::onScreenChanged()
{
  // Change the display of the context when the user change application screen
  if ( m_renderAreaCore.ptr() )
    m_renderAreaCore.ptr()->getGLContext()->forceDisplay( SbGlContextHelper::getCurrentDisplay() );
}
