#include <Inventor/ViewerComponents/Qt/RenderAreaInteractive.h>

#include <Inventor/ViewerComponents/nodes/SceneInteractor.h>

#include <Inventor/elements/SoInteractionElement.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoStereoCamera.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/ViewerComponents/Qt/QtTimer.h>

#include <QKeyEvent>
#include <QMouseEvent>
#include <QWheelEvent>
#include <QWidget>

#ifdef _WIN32
#pragma warning( push )
#pragma warning( disable:4996 )
#endif

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive( QWidget* parent )
  : RenderArea( parent )
{
  init(true);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive( QWidget* parent, bool buildRootSceneGraph )
  : RenderArea( parent )
{
  init(buildRootSceneGraph);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::~RenderAreaInteractive()
{
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::startRender( SiRenderArea::RenderEventArg& )
{
  requestUpdate();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::init( bool buildRootSceneGraph )
{
  m_clippingMode = AUTO;
  m_isDragging = false;
  m_interactiveMode = SoInteractiveComplexity::AUTO;
  m_isAutoInteractive = true;

  if ( buildRootSceneGraph )
  {
    m_rootSceneGraph = new SceneInteractor();
    buildSceneGraph();
  }

  m_containerWidget->setMouseTracking(true);
  m_containerWidget->setAttribute(Qt::WA_AcceptTouchEvents, true);
  SoDB::setSystemTimer(new QtTimer);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::buildSceneGraph()
{
  m_appSceneGraph = new SoGroup;
  m_rootSceneGraph->addChild(m_appSceneGraph);
  RenderArea::setSceneGraph(m_rootSceneGraph);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setSceneGraph( SoNode *sceneGraph )
{
  m_appSceneGraph->removeAllChildren();
  m_appSceneGraph->addChild(sceneGraph);
}

//------------------------------------------------------------------------------
SoRenderAreaCore::RenderStatus
RenderAreaInteractive::render()
{
  if ( m_clippingMode == AUTO && m_renderAreaCore.ptr() != NULL )
  {
    // Update camera clipping planes before each rendering.
    m_rootSceneGraph->adjustClippingPlanes(m_renderAreaCore->getSceneManager()->getViewportRegion());
  }

  // render scene
  return RenderArea::render();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
{
  if (m_clippingMode != mode)
  {
    m_clippingMode = mode;
    requestUpdate();
  }
}

//------------------------------------------------------------------------------
RenderAreaInteractive::ClippingPlanesAdjustMode
RenderAreaInteractive::getClippingPlanesAdjustMode()
{
  return m_clippingMode;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::initializeGL()
{
  RenderArea::initializeGL();
  m_renderAreaCore->onStartRender().add( *this, &RenderAreaInteractive::startRender );
 // check if there are incoming events when rendering, and if so abort the rendering
  m_renderAreaCore->getSceneManager()->setAbortRenderCallback( RenderAreaInteractive::s_abortRenderCallback, this );
  m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
  m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getRootSceneGraph() const
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getSceneInteractor() const
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::viewAll( const SbViewportRegion &viewport )
{
  m_rootSceneGraph->viewAll(viewport);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::viewAxis( const SbVec3f& direction, const SbVec3f& up )
{
  m_rootSceneGraph->viewAxis( direction, up );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::activateStereo( bool activated )
{
  if ( activated )
  {
    // check camera
    if ( !isStereoSupported() )
    {
      qWarning("Cannot activate stereo: current camera is not a SoStereoCamera!");
      return;
    }
  }
  RenderArea::activateStereo(activated);
}

//------------------------------------------------------------------------------
bool
RenderAreaInteractive::isStereoSupported() const
{
  // check camera
  SoCamera* camera = m_rootSceneGraph->getCamera();
  return (dynamic_cast<SoStereoCamera*>( camera ) != NULL);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setStereoCameraOffset( float offset )
{
  SoCamera* camera = m_rootSceneGraph->getCamera();
  SoStereoCamera* stereoCamera = NULL;

  if ( (stereoCamera = dynamic_cast<SoStereoCamera*>( camera )) == NULL )
    qWarning("Current camera is not a SoStereoCamera!");
  else
    stereoCamera->offset = offset;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setStereoCameraBalance( float balance )
{
  SoCamera* camera = m_rootSceneGraph->getCamera();
  SoStereoCamera* stereoCamera = NULL;

  if ( (stereoCamera = dynamic_cast<SoStereoCamera*>( camera )) == NULL )
    qWarning("Current camera is not a SoStereoCamera!");
  else
    stereoCamera->balance = balance;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setInteractiveMode(SoInteractiveComplexity::InteractiveMode mode)
{
  m_interactiveMode = mode;
  m_isAutoInteractive = ( mode == SoInteractiveComplexity::AUTO );

  if ( m_renderAreaCore.ptr() != NULL )
  {
    m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
    m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
  }
}

//------------------------------------------------------------------------------
SoInteractiveComplexity::InteractiveMode
RenderAreaInteractive::getInteractiveMode() const
{
  return m_interactiveMode;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvent(const SoEvent *event)
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvent( event );

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvents(const std::vector<const SoEvent*>& eventList)
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvents( eventList );

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mousePressEvent( QMouseEvent* qevent )
{
  m_isDragging = true;
  SoEvent* event = QtEventToSoEvent::getMousePressEvent( qevent, QPoint( qevent->x() * m_containerWidget->devicePixelRatio(), ( ( m_containerWidget->height() - 1 ) - qevent->y() ) * m_containerWidget->devicePixelRatio() ) );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::mouseReleaseEvent( QMouseEvent* qevent )
{
  m_isDragging = false;
  SoEvent* event = QtEventToSoEvent::getMouseReleaseEvent(qevent, QPoint( qevent->x() * m_containerWidget->devicePixelRatio(), ( ( m_containerWidget->height() - 1 ) - qevent->y() ) * m_containerWidget->devicePixelRatio() ) );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::mouseMoveEvent( QMouseEvent* qevent )
{
  SoEvent* event = QtEventToSoEvent::getMouseMoveEvent(qevent, QPoint( qevent->x() * m_containerWidget->devicePixelRatio(), ( ( m_containerWidget->height() - 1 ) - qevent->y() ) * m_containerWidget->devicePixelRatio() ) );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::wheelEvent( QWheelEvent* qevent )
{
  SoEvent* event = QtEventToSoEvent::getMouseWheelEvent( qevent );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::keyPressEvent( QKeyEvent* qevent )
{
  SoEvent* event = QtEventToSoEvent::getKeyPressEvent( qevent );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::keyReleaseEvent( QKeyEvent* qevent )
{
  SoEvent* event = QtEventToSoEvent::getKeyReleaseEvent( qevent );
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::enterEvent(QEvent * /*qevent*/)
{
  SoEvent* event = QtEventToSoEvent::getMouseEnterEvent();
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::leaveEvent(QEvent * /*qevent*/)
{
  SoEvent* event = QtEventToSoEvent::getMouseLeaveEvent();
  processEvent(event);
}

//------------------------------------------------------------------------------
void 
RenderAreaInteractive::mouseDoubleClickEvent(QMouseEvent * qevent)
{
  SoEvent* event = QtEventToSoEvent::getMouseDoubleClickEvent(qevent, QPoint( qevent->x() * m_containerWidget->devicePixelRatio(), ( ( m_containerWidget->height() - 1 ) - qevent->y() ) * m_containerWidget->devicePixelRatio() ) );
  processEvent(event);
}

//------------------------------------------------------------------------------
bool 
RenderAreaInteractive::event(QEvent * qevent)
{
  QTouchEvent *qtouchEvent = dynamic_cast<QTouchEvent*>(qevent);
  if (qtouchEvent != NULL) 
  {
    const std::vector<const SoEvent*>& soeventlist =
      m_eventbuilder.getTouchEvents( qtouchEvent, m_containerWidget->height() );
    processEvents(soeventlist);
    return true;
  } else
    return RenderArea::event(qevent);
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::s_abortRenderCallback( SoAction* action, void* data )
{
  RenderAreaInteractive* renderArea = static_cast<RenderAreaInteractive*>( data );

  SoState* state = action->getState();

  // check if we are rendering a frame in INTERATIVE mode. In that case, don't abort rendering,
  // else we might abort all frame and we won't have any rendering.
  // if frame is in STILL mode (non interacting), then we will check for event in event queue
  // to get back interactive as soon as possible.
  if ( !SoInteractionElement::isInteracting( state ) )
  {
#if defined (_WIN32)
    SoSceneManager* sceneMgr = action->getSceneManager();
    if ( sceneMgr != NULL && sceneMgr->isAutoInteractiveMode() && renderArea->m_isDragging )
    {
      if ( GetQueueStatus( QS_MOUSEMOVE | QS_MOUSEBUTTON | QS_KEY ) )
        return TRUE;
    }
    else
    {
      if ( GetQueueStatus( QS_MOUSEBUTTON | QS_KEY ) )
        return TRUE;
    }
#elif defined(__APPLE__)
    //TODO: NOT IMPLEMENTED
#else // UNIX
    //TODO: NOT IMPLEMENTED
#endif
  }

  return FALSE;
}

#ifdef _WIN32
#pragma warning( pop )
#endif
