
#include "PreferencesDialog.h"
#include <Inventor/ViewerComponents/Qt/Viewers/ViewerExaminer.h>


PreferencesDialog* PreferencesDialog::m_instance = NULL;

//------------------------------------------------------------------------------
PreferencesDialog::PreferencesDialog( ViewerExaminer* viewerExaminer )
  : QDialog(viewerExaminer)
  , m_viewerExaminer( viewerExaminer )
{

  // This is a convenient method from Qt to add the widgets to the interface.
  setupUi( this );

  // set up navigation mode buttons
  m_navigationButtonGroup->setId( m_orbitRadioButton, SceneExaminer::ORBIT );
  m_navigationButtonGroup->setId( m_planeRadioButton, SceneExaminer::PLANE );
  switch( m_viewerExaminer->getNavigationMode() )
  {
  case SceneExaminer::PLANE:
    m_planeRadioButton->setChecked( true );
    break;
  case SceneExaminer::ORBIT:
  default:
    m_orbitRadioButton->setChecked( true );
    break;
  }

  m_clippingComboBox->setCurrentIndex( m_viewerExaminer->getClippingPlanesAdjustMode() );

  // Balance values are -3.0 to 3.0, slider wants values -300 to 300
  float balanceDefaultValue = 1.f;
  m_balanceSlider->setValue( balanceDefaultValue * 100 );
  m_balanceTextLabel->setText( QString::number( balanceDefaultValue ) );

  // Offset values are 0.0 to 2.0, slider wants values 0 to 200
  float offsetDefaultValue = 0.7f;
  m_offsetSlider->setValue( offsetDefaultValue * 100 );
  m_offsetTextLabel->setText( QString::number( offsetDefaultValue ) );

  connect( m_navigationButtonGroup, SIGNAL( buttonToggled(int, bool) ), this, SLOT( navigationModeChanged(int, bool) ) );
  connect( m_clippingComboBox, SIGNAL( currentIndexChanged(int) ), this, SLOT( clippingModeChanged(int) ) );
  connect( m_rawStereoCheckBox, SIGNAL( toggled(bool) ), this, SLOT( activateRawStereo(bool) ) );
  connect( m_balanceSlider, SIGNAL( valueChanged(int) ), this, SLOT( balanceValueChanged(int) ) );
  connect( m_balanceCheckBox, SIGNAL( toggled(bool) ), this, SLOT( slot_check_balance(bool) ) );
  connect( m_offsetSlider, SIGNAL( valueChanged(int) ), this, SLOT( offsetValueChanged(int) ) );
  connect( m_offsetCheckBox, SIGNAL( toggled(bool) ), this, SLOT( slot_check_offset(bool) ) );
}


PreferencesDialog*
PreferencesDialog::getInstance( ViewerExaminer* viewerExaminer )
{
  if ( m_instance == NULL || viewerExaminer != m_instance->m_viewerExaminer )
    m_instance = new PreferencesDialog( viewerExaminer );

  return m_instance;
}

void
PreferencesDialog::navigationModeChanged( int mode, bool checked )
{
  if ( checked )
    m_viewerExaminer->setNavigationMode( SceneExaminer::NavigationMode( mode ));
}

void
PreferencesDialog::clippingModeChanged( int index )
{
  m_viewerExaminer->setClippingPlanesAdjustMode( (RenderAreaInteractive::ClippingPlanesAdjustMode) index);
}

void
PreferencesDialog::activateRawStereo( bool activated )
{
  m_viewerExaminer->activateStereo( activated );
}

void
PreferencesDialog::slot_check_balance( bool checked )
{
  m_balanceSlider->setTracking( checked );
}

void
PreferencesDialog::slot_check_offset( bool checked )
{
   m_offsetSlider->setTracking( checked );
}

void
PreferencesDialog::balanceValueChanged( int value )
{
  // val is between -300 and 300 (UI) and we want values in -3.0 and 3.0 (API)
  float balance = value / 100.f;
  m_viewerExaminer->setStereoCameraBalance( balance );

  char localString[256];
  sprintf( localString, "%.2f", balance );
  m_balanceTextLabel->setText( QString(localString) );
}

void
PreferencesDialog::offsetValueChanged( int value )
{
  // val is between 0 and 200 (UI) and we want values in 0 and 2.0. (API)
  float offset = value / 100.f;
  m_viewerExaminer->setStereoCameraOffset( offset );

  char localString[256];
  sprintf( localString, "%.2f", offset );
  m_offsetTextLabel->setText( QString(localString) );
}

void
PreferencesDialog::enableStereoPreferences( bool enabled )
{
  if ( !enabled && m_rawStereoCheckBox->isChecked() )
  {
    // deactivate raw stereo
    m_rawStereoCheckBox->setChecked( FALSE );
  }

  m_rawStereoCheckBox->setEnabled( enabled );
  m_balanceCheckBox->setEnabled( enabled );
  m_balanceSlider->setEnabled( enabled );
  m_balanceTextLabel->setEnabled( enabled );
  m_offsetCheckBox->setEnabled(enabled);
  m_offsetSlider->setEnabled( enabled );
  m_offsetTextLabel->setEnabled( enabled );
}
